open Ast
open Ast_types
open Eval
open Environment
open Environment_types

exception Parse_error of exn * int * int * string

let module_depth = ref 0

let tab_space = 2

let create_string () : string = String.make (!module_depth*tab_space) ' '

let parse (s : string) : Ast.toplevel =
  let lexbuf = Lexing.from_string s in
  try Parser.parse Lexer.token lexbuf
  with exn ->
    let curr = lexbuf.Lexing.lex_curr_p in
    let line = curr.Lexing.pos_lnum in
    let char = curr.Lexing.pos_cnum - curr.Lexing.pos_bol in
    let token = Lexing.lexeme lexbuf in
    raise (Parse_error (exn, line, char, token))

(* top-level bindings *)
let top = ref (Util.HashSet.make ())

(* read lines from the console, appending them to s,
 * until the user enters ;; at end of line *)
let read_console () : string =
  let rec read_lines (s : string) : string =
    let input = Util.trim (read_line ()) in
    if Util.ends_with ";;" input
    then s ^ (String.sub input 0 (String.length input - 2))
    else read_lines (s ^ input ^ "\n")
  in read_lines ""

let eval_expression (e : expr) ((o,oT) : env*env_types) : env*env_types =
  let (typeE, p) = 
    if !Inference_types.type_inference 
    then Inference_types.getType e oT
    else (TDummy, Environment_types.empty) in
  let (v, t) = Eval.eval (e, o) in
  if !Inference_types.type_inference
  then Printf.printf "- : %s = %s\n" 
    (Ast_types.to_string typeE) (Printing.to_string (v, t))
  else print_endline (Printing.to_string (v, t));
  (* garbage collect *)
	(Environment.gc !top t, oT)

let rec eval_module (verbose : bool) ((o,oT) : env*env_types) (d : declaration)
    : env*env_types =
  match d with
  | Module(id,dl) ->
		(* TODO: Do static typechecking *)
    Printf.printf "%smodule %s : sig\n" (create_string ()) id;
    module_depth := !module_depth + 1;
    let oT2 = Environment_types.push_module id (Environment_types.empty_module)
    oT in
		let env = Environment.push_module id (Environment.empty_module) o in
    let (o,oT) = List.fold_left (declare verbose) (env,oT2) dl in
      module_depth := !module_depth - 1;
      Printf.printf "%send\n" (create_string ());
    	(Environment.pop_module id o, Environment_types.pop_module id oT)
  | _ -> assert false
and declare (verbose : bool) ((o,oT) : env*env_types) (d : declaration) 
    : env*env_types =
  (* a special treatment for typedefs and module declarations *)
  match d with
    | Typedef (a,b) -> 
      Typedef.check_typedef (a,b);
      (o, Environment_types.append_typedef oT (a,b))
    | Module _ -> eval_module verbose (o,oT) d
    | _ -> (
  let (f, args, e) = match d with
    | Def (f,_,a,_,e) |
      Defrec (f,_,a,_,e) -> (f,a,e)
    | Defcorec (s, f,_,a,_,e) -> 
      (f, [ ], EFunCorec(s,f,a,e))
    | Typedef _ -> assert false
    | Module _ -> assert false
  in
  let e = match args with
    | [] -> e
    | _ -> EFun (args, e) in
  let o = match d with
    | Def _ -> o
    | Defrec _ | Defcorec _ ->
        Environment.bind f EDummy o 
    | Typedef _ -> assert false
		| Module _ -> assert false in
  let (typeE, p) =
    if !Inference_types.type_inference
    then Inference_types.getTypeDecl d oT (* TODO: fix inference types to
    use the static environment *)
    else (TNull, oT) in
	let (v, t) = eval (e, o) in (* TODO: does eval need everything? *)
  let Schema (_, typ) as gen = Equations_types.generalize typeE p in
  Util.HashSet.add !top f;
  let env = Environment.bind f v t in
  if verbose then
    if !Inference_types.type_inference
    then 
      if !module_depth = 0
      then Printf.printf "val %s : %s = %s\n" f 
        (Ast_types.to_string typ) (Printing.to_string_bv (v, env) [ f ])
      else Printf.printf "%sval %s : %s\n" (create_string ())  f 
        (Ast_types.to_string typ)
    else Printf.printf "%s = %s\n" f (Printing.to_string (v, env));
  (env, Environment_types.bind f gen p))

(* TODO: Finish *)
let eval_declarations (dl : declaration list) ((o,oT) : env*env_types)
    (verbose : bool) : env*env_types =
  let (o,oT) = List.fold_left (declare verbose) (o,oT) dl in
  (* garbage collect *)
	(Environment.gc !top o, oT)

let eval_directive (d : directive) ((o,oT) : env*env_types)
    : env*env_types =
  match d with
    | Quit -> raise Exit
    | Chdir s ->
       if not (Sys.file_exists s) then runtime "directory does not exist" else
       if not (Sys.is_directory s) then runtime "file is not a directory" else
       Sys.chdir s; (o,oT)
    | Pwd -> print_endline (Sys.getcwd ()); (o,oT)
    | Load s ->
       if not (Sys.file_exists s) then runtime "file does not exist" else
       if Sys.is_directory s then runtime "file is a directory" else
       (match parse (read_file s) with
        | Declarations d -> eval_declarations d (o,oT) true
        | Expression e -> eval_expression e (o,oT)
        | Directive d -> runtime "directives not allowed in file")
    | Ls ->
       let cwd = Sys.getcwd () in
       let files = Sys.readdir cwd in
       Array.iter print_endline files; (o,oT)
    | Env ->
      (* TODO: Verify it is correct *)
				if !Inference_types.type_inference then
          let f x (Schema (_, typ)) =
	    Printf.printf "%s : %s\n" x (Ast_types.to_string typ) in
          Environment_types.iter f oT
      	else ();
      	Printing.print_env o; (o,oT)
    | Scope ->
       Eval.static_scope := not !static_scope;
       Printf.printf "%s scope\n" 
	 (if !static_scope then "static" else "dynamic"); (o,oT)
    | Type_inf ->
       Inference_types.type_inference := not !Inference_types.type_inference;
       Printf.printf "type inference %s\n" 
	 (if !Inference_types.type_inference then "on" else "off"); (o,oT)

let handle = function
(*  | Unbound s -> Printf.printf "Unbound variable %s\n" s *) (*TODOjb*)
  | Parse_error (_,l,c,_) -> 
    Printf.printf "Parse error at line %d char %d\n" l c
  | Lexer.LexError s -> Printf.printf "Lex Error: %s\n" s
  | Ast.Runtime s -> Printf.printf "Runtime error: %s\n" s
  | Equations_types.Type_error s ->
    Printf.printf "Type error: %s\n" s
  | Util.Fatal s -> 
    Printf.printf "Fatal error: %s\nSorry :(\n" s
  | Natives.Failwith s ->
    Printf.printf "Failure (thrown by the user): %s\n" s
  | e -> raise e

(* read eval print loop *)
let rec repl ((o,oT) : env*env_types) : unit =
  try
    print_string "> ";
    let cmd = parse (read_console()) in
    let ((o,oT) : env*env_types) =
      match cmd with
      | Directive d -> eval_directive d (o,oT)
      | Declarations d -> eval_declarations d (o,oT) true
      | Expression e -> eval_expression e (o,oT) in
    repl (o,oT)
  with
    | Exit -> print_endline "bye"
    | e -> handle e; repl (o,oT)

let eval_file f ((o,oT) : env*env_types) : env*env_types =
  try
    eval_directive (Load f) (o,oT)
  with e -> handle e; (o,oT)

let run () =
  let natives =
    List.fold_left
      (fun env (id, impl, typ) -> Environment.bind id (ENative (id, impl, typ))
			env)
      (Environment.empty) Natives.natives in

  let nativesT = 
    List.fold_left
      (fun env (id, impl, typ) -> 
        let (t, _) = Inference_types.getType (ENative (id, impl, typ)) env in
        Environment_types.bind id (Equations_types.generalize t env) env)
      Environment_types.empty Natives.natives in
      
  List.iter (fun (x, _, _) -> Util.HashSet.add !top x) Natives.natives;

  (* type list defined as a normal sum type, by hand *)
  let typedefs = [ "list", ([ "'a" ],
			    [ "[]", None; (* replace with "[]" *)
			      "::", Some(TTuple [
				TVar "'a";
				TUser ("list", [TVar "'a"])])
			    ]) ] in

  let nativesT = Environment_types.set_typedefs nativesT typedefs in
  (* load pervasives *)
  let pervasives =
    match parse (read_file "etc/pervasives.ml") with
    | Declarations dl -> 
      eval_declarations dl (natives, nativesT) false
    | _ -> raise (Util.Fatal "corrupt pervasives.ml file") in
  
  (* Only prompts if there are no arguments *)
  let has_args = ref false in
  let envs = ref pervasives in
  Arg.parse [ ] 
    (fun f -> has_args := true; envs := eval_file f !envs) 
    "CoCaml help message";
  if !has_args then () else repl pervasives
