open Ast
open Parser
open Util
open Environment
open Equality

exception Unmatched_pattern
  
type equation = expr * expr
(* first element represents an id:
   - either ESymbol(id) in my version 
   - or something of type var in Dexter's version *)

(* global flag for static or dynamic scope *)
let static_scope = ref true

(* read from a file *)
let read_file (filename : string) : string =
  let rec get_contents contents file =
    let input_line_option file =
      try Some ((input_line file) ^ "\n") with End_of_file -> None in
    match (input_line_option file) with
	Some x -> get_contents (contents ^ x) file
      | None -> contents in
  let file = try open_in filename with Not_found -> runtime "File not found" in
  let result = get_contents "" file in close_in file; result

(* get a fresh variable not occurring in an expression *)

(* Substitute var y for free occurrences of var x in e *)
let subst (y : id) (x : id) (e : expr) : expr =
  let match_args = function (EVar id) -> id = x | _ -> false in
  let rec sub (e : expr) : expr =
    match e with
      | EVar id -> if id = x then EVar y else EVar id
      | (EInt _ | EFloat _ | EString _ | EBool _ | EDummy | ENative _ | EUnit) -> e
      | EFun (args, c) -> 
	if List.exists match_args args then e else EFun (args, sub c)
      | EFunType (args,_,e) -> sub (EFun(args,e))
      | EFunCorec (s, f, args, c) -> 
        if List.exists match_args args then e else EFunCorec (s, f, args, sub c)
      | ELetType (id, t1, args, t2,e1,e2) -> 
	let e1 = if (List.exists match_args args) then e1 else sub e1 in
	let e2 = if id = x then e2 else sub e2 in
	ELetType (id, t1, args, t2, e1, e2)
      | ELetrecType (id, t1, args, t2, e1, e2) ->
	let e1 = 
	  if (id = x || List.exists match_args args) then e1 else sub e1 in
	let e2 = if id = x then e2 else sub e2 in
	ELetrecType (id, t1, args, t2, e1, e2)
      | ELetcorecType (s, id, t1, args, t2, e1, e2) ->
	let e1 = 
	  if (id = x || List.exists match_args args) then e1 else sub e1 in
	let e2 = if id = x then e2 else sub e2 in
        ELetcorecType (s, id, t1, args, t2, e1, e2)
      | EFunction l -> EFunction(
	List.map 
	  (fun (pi, ei) -> 
	    if List.mem x (Ast.bound_vars pi) then (pi, ei)
	    else (pi, sub ei))
	  l)
      | EApp (e1, e2) -> EApp (sub e1, sub e2)
      | EIf (e1, e2, e3) -> EIf (sub e1, sub e2, sub e3)
      | EWhile (e1, e2) -> EWhile (sub e1, sub e2)
      | EFor (id, e1, e2, e3) ->
        EFor ((if id = x then y else id), sub e1, sub e2, sub e3)
      | EMatch(e1, l) ->
	EMatch(sub e1,
	      List.map 
		(fun (pi, ei) -> 
		  if List.mem x (Ast.bound_vars pi) then (pi, ei)
		  else (pi, sub ei))
		l)
      | EAssign (id, e2) -> EAssign ((if id = x then y else id), sub e2)
      | ESeq (e1, e2) -> ESeq (sub e1, sub e2) 
      | ETuple lst -> ETuple (List.map (fun e1 -> sub e1) lst)
      | EInj(s, None) -> e
      | EInj(s, Some(e1)) -> EInj(s, Some(sub e1))
		(* Comparison operators *)
      | EBinop (b, e1, e2) -> EBinop (b, sub e1, sub e2)
      | ENot e1 -> ENot(sub e1)
      | EUnknown i -> EUnknown i 
      | ESymbol i -> ESymbol i 
  in sub e

let rec matches (p:pattern) (e:expr) (o:env) : (id * expr) list option =
  (* tests if pattern p matches expression e *)
  let e0 = (match e with EVar x -> Environment.lookup x o | _ -> e) in
  (* for the case where the structure is hidden in the capsule *)
  match (p, e0) with
    | PVar x, e -> Some [ x, e ]
    | PInt i1, EInt i2 when i1 = i2 -> Some [ ]
    | PFloat f1, EFloat f2 when f1 = f2 -> Some [ ]
    | PString s1, EString s2 when s1 = s2 -> Some [ ]
    | PBool b1, EBool b2 when b1 = b2 -> Some [ ]
    | PUnit, EUnit -> Some [ ]

    | PInj(i1, None), EInj(i2, None) when i1 = i2 -> Some [ ]
    | PInj(i1, Some p1), EInj(i2, Some e2) when i1 = i2 -> matches p1 e2 o
    | PUnknown p1, EUnknown e2 -> matches p1 e2 o

    | PTuple pl, ETuple el -> (* because of typing, they are the same size *)
      List.fold_left2 
        (fun bo pi ei -> match bo with None -> None
          | Some(b) -> (match matches pi ei o with 
              None -> None
              | Some(bi) -> Some(b @ bi)))
        (Some [ ]) pl el
    | PUnderscore, _ -> Some [ ]
    | PInt _, _ | PFloat _, _ | PString _, _ | PBool _, _ | PUnit, _ 
    | PInj _, _ | PTuple _, _ | PUnknown _, _ -> None

let rec eval_patterns 
    (e:expr) (pl:(pattern * expr) list) (o : env) fcorec unks repl
    : state * ((expr * expr) list) * bool = 
  match pl with
    | [ ] -> runtime ("EMatch failure on " ^ (Printing.to_string (e,o)))
    | (pi, ei) :: tl -> (
      match matches pi e o with 
	  None -> eval_patterns e tl o fcorec unks true
	| Some b -> if b = [ ] then evalp (ei, o) fcorec unks true else
            let (vars, args) = List.split b in
            let e2 = List.fold_left
              (fun fu arg -> EApp(fu, arg)) 
              (EFun(List.map (fun x -> EVar x) vars, ei)) args
            in evalp (e2, o) fcorec unks true )

and evalp ((e, o) : state) 
    (fcorec : (id * ((state * expr) list) * 
		 ((state -> state) * ((unit * env) -> state))) option) 
    (* the second element is the set of seen elements: association with their 
       variable. Changed to an expr to accomodate Dexter's view of things.
       Those expressions will be EUnknown(sth) if using my view,
      or of time var if using Dexter's view *)
    (unks : (bool * (id, expr) Hashtbl.t) option)
    (repl:bool)
    (* unks an interpretation for the unknowns *)
    : state * ((expr * expr) list) * bool =
   (* the list l is the list of unknowns appearing inside 
      (except if already seen) *)
   (* boolean b indicates if it is completely evaluated (true) or not *)
   (* DO NOT suppress b: l could be empty with b false, see 10 lines below *)
   (* partial evaluation; fcorec is the name of the corecursive function
      id * expr list are the corecursive calls, meaning the result of 
      fcorec(expr) is represented by EUnknown(id) *)
  match fcorec, e with
    | Some(f, seen, (uunk, ufresh)), EApp(EVar id, e1) when f = id ->
      let (e2, o2), l, b = evalp (e1, o) None unks true in 
      (* None because nesting not allowed *)
      if b then 
        try let name = Equality.assq (e2, o2) seen  in
            (uunk (name, o2)), [ ], false
        with Not_found -> 
          let freshid, o3 = ufresh((), o2) in
	  (uunk (freshid, o3)), [ freshid, e2 ], false
      else assert false
    | Some(f, _, _), EVar id when f = id -> 
      runtime "A corecursive call has to be applied right away"
    (* TODO: what if f is rebound later in its own body? *)
    | _ -> match e with
	| EVar id -> let r = Environment.lookup id o in
                    (* to handle corecursive definitions *)
                    if r = EDummy || (not(repl))
                    then (EVar id, o), [ ], true
                    else (r, o), [ ], true
	| EInt i -> (e, o), [ ], true
	| EFloat f -> (e, o), [ ], true
	| EString s -> (e, o), [ ], true
	| EBool b -> (e, o), [ ], true
	| EFun (_, _) -> (e, o), [ ], true
                (* TODO: check that fcorec does not appear in body *)
	| EFunType (args, _, c) -> (EFun (args,c), o), [ ], true
        | EFunCorec _ -> (e, o), [ ], true
	| EFunction l -> (e, o), [ ], true
	| ELetType (id, _, args, _, e1,e2) ->
	  if args = [] 
	  then evalp (EApp (EFun ([EVar id], e2), e1), o) fcorec unks true
	  else 
	    evalp (EApp (EFun ([EVar id], e2), EFun (args, e1)), o) 
	      fcorec unks true
	| ELetrecType (id, tid, args, targs, e1, e2) ->
	  if args = [] then
	    (* Let id = <dummy> in (id := e1); e2 *)
	    evalp
              (ELetType (id, tid, [], [], EDummy, ESeq (EAssign (id, e1), e2)), o) 
	      fcorec unks true
	  else
	    (* Let id = <dummy> in (id := fun args -> e1); e2 *)
	    evalp (ELetType (id, tid, [], [], EDummy, 
			    ESeq (EAssign (id, EFun (args, e1)), e2)), o)
	      fcorec unks true
        | ELetcorecType (s, id, _, [arg], _, e1, e2) -> (* TODO *)
          evalp (EApp (EFun ([EVar id], e2), EFunCorec (s, id, [arg], e1)), o)
	    fcorec unks true
        | ELetcorecType _ -> 
	  runtime "Corecursive functions can only have one argument"
	| EApp (e1, e2) ->
	  let (d, p1), l1, b1 = evalp (e1, o) fcorec unks true in
	  let (u, p2), l2, b2 = evalp (e2, p1) fcorec unks true in
          let l = l1 @ l2 and b = b1 && b2 in
          let fcorec2 = if l = [ ] then fcorec else None in
          if not(b) then (EApp(d, u), p2), l, b else
	    (match d with
  	      | EFun (args, c) -> (match args with
  	          | EVar x :: t -> 
		    let res1, res2, res3 = if !static_scope
                    then let y = fresh () in
                         evalp (subst y x (if t = [] then c else EFun (t, c)),
                                Environment.bind y u p2) fcorec2 unks true
                    else (* dynamic scope *)
  	              evalp ((if t = [] then c else EFun (t, c)),
                             Environment.bind x u p2) fcorec unks true
				  in res1, l @ res2, res3
                  | EUnit :: t -> if u = EUnit 
                    then let res1, res2, res3 =
                           evalp ((if t = [] then c else EFun (t, c)), p2) 
			     fcorec2 unks true
                         in res1, l @ res2, res3
                    else runtime "argument mismatch"
                  | _ -> runtime "not a funtion")
              | EFunCorec (solver, fname, [EVar x], body) ->
                if not(fcorec = None) 
                then 
                  (match fcorec with 
                      None -> assert false 
                    | Some(f, _, _) -> if f = fname then
                        runtime "Nested corecursive functions not allowed"
                      else
                        eval_corec solver fname (EFun([EVar x], body)) 
                          (u, p2) unks true, [ ], true)
                else if !static_scope
                then 
                  eval_corec solver fname (EFun([EVar x], body)) 
		    (u, p2) unks true, [ ], true
                else runtime 
		  "Corecursive functions not implemented for dynamic scope"
              | EFunCorec _ -> 
		runtime "Corecursive function must have exactly one argument"
	      | EFunction l ->
		eval_patterns u l p2 fcorec unks true
              | ENative (_, native, _) -> (native u, p2), [ ], true
  	      | _ -> print_string(Printing.to_string (d,o));
		runtime "can only apply a function")
	| EIf (e, e1, e2) ->
	  let (d, t), l, b = evalp (e, o) fcorec unks true in
          let fcorec2 = if l = [ ] then fcorec else None in
          if (not b) then (EIf(d, e1, e2), t), l, b else
	    ( match d with
	      | EBool b0 -> let res1, res2, res3 =
                             evalp ((if b0 then e1 else e2), t) 
			       fcorec2 unks true
                           in res1, l @ res2, res3
	      | _ -> runtime "'if' requires a boolean test")
	| EWhile (e, e1) ->
	  let (d,t), l, b = evalp (e,o) fcorec unks true in
          let fcorec2 = if l = [ ] then fcorec else None in
          if not b then (EWhile(d, e1), t), l, b else
	    (match d with
	      | EBool b0 -> if b then
                  let res1, res2, res3 =
                    evalp (ESeq (e1, EWhile (e, e1)), t) fcorec2 unks true
                  in res1, l @ res2, res3
		else (EUnit, t), l, b
	      | _ -> runtime "'while' requires a boolean test")
	| EFor (x, e1, e2, e3) -> 
	  let (d,t), l1, b1 = evalp (e1,o) fcorec unks true
          and (f,u), l2, b2 = evalp (e2,o) fcorec unks true in
          let l = l1 @ l2 in
	  let b = b1 && b2 in
          let fcorec2 = if l = [ ] then fcorec else None in
          if not b then (EFor(x, d, f, e3), t), l, b else
	    (match (d,f) with
  	      | ( EInt n, EInt p) -> if n>p then (EUnit,o), l, true
		else let res1, res2, res3 = 
		       evalp (ESeq (e3, EFor (x,  (EBinop (BPlus, d, EInt 1)),f, e3)),
			      Environment.bind x d o) fcorec2 unks true in
		     res1, l @ res2, res3
  	      | _ -> runtime 
		" the second and third arguments of 'for' must evaluate to integers ")
	| EAssign (x, e) ->
          if not (Environment.bound x o) then raise (Unbound x) else
            let (v, t), l, b = evalp (e, o) fcorec unks true in
            if not b then (EAssign (x, v), t), l, b else
              (EUnit, Environment.bind x v t), l, b
	| ESeq (e1, e2) ->
	  let (t1, o1), l1, b1 = evalp (e1, o) fcorec unks repl in
	  let (t2, o2), l2, b2 = evalp (e2, o1)
	    (if l1 = [ ] then fcorec else None) unks b1
          in let b = b1 && b2
             in if b then (t2, o2), l1 @ l2, b
               else ( ESeq (t1, t2), o2 ), l1 @ l2, b
        | EMatch(e1, l) ->
          let (r1, o1), l1, b1 = evalp (e1, o) fcorec unks true in
          if b1 then eval_patterns r1 l o1 fcorec unks true
          else (EMatch(r1, l), o1), l1, b1

        | ETuple l -> (* neither fold_left nor fold_right does the job *) 
          let rec iterate oi = function
            | [ ] -> ([ ], oi), [ ], true
            | h :: t ->
	      let (ej, oj), lj, bj = evalp (h, oi) fcorec unks repl in 
              let (l2, o2), lk, bk = iterate oj t in
              (ej :: l2, o2), lj @ lk, bj && bk in
          let (l2, o2), lcorec, bf = iterate o l in
          (ETuple l2, o2), lcorec, bf
        | EInj(s, None) -> (e, o), [ ], true
        | EInj(s, Some(e1)) ->
	  let (e2, o2), l, b = evalp (e1, o) fcorec unks false (* repl *) in
          (EInj(s, Some(e2)), o2), l, b

	| EBinop((BAnd|BOr as binop), e1, e2) ->
	  (* specific case needed to handle laziness of && and || *)
	  (match evalp (e1, o) fcorec unks true with
	  | (EBool b, t), l1, b1 as d -> 
	    if ((not b) && binop = BAnd || b && binop = BOr) then d else
      	      (match evalp (e2, t) fcorec unks true with
      	      | (EBool b, o2), l2, b2 -> (EBool b, o2), l1 @ l2, b1 && b2
              | (r2, o2), l2, b2 when not(b2) ->
                (EBinop(binop, EBool b, r2), o2), l1 @ l2, b1 && b2
      	      | _ -> runtime "can only use && and || on booleans")
          | (r1, o1), l1, b1 when not(b1) -> (* giving up laziness *)
            let (r2, o2), l2, b2 = evalp(e2, o1) fcorec unks true in
            (EBinop(binop, r1, r2), o2), l1 @ l2, b1 && b2
          | _ -> runtime "can only use && and || on booleans")
	  
	| EBinop(binop, e1, e2) ->
	  let (p1,t), l1, b1 = evalp (e1,o) fcorec unks true in
	  let (p2,r), l2, b2 = evalp (e2,t) fcorec unks true in
          let b = b1 && b2 and l = l1 @ l2 in
	  let comp_op = function 
	    | BLe -> (<=) | BLt -> (<) | BGt -> (>) |BGe -> (>=)
	    | _ -> assert false in
	  ((if (not b) then EBinop(binop, p1, p2) else
	      match (binop, p1, p2) with
		BEq, _, _ -> EBool(Equality.equal (p1, r) (p2, r))
	      | BNeq, _, _ -> EBool(not (Equality.equal (p1, r) (p2, r)))
	      | (BLe|BLt|BGt|BGe), EInt i, EInt j -> EBool((comp_op binop) i j)
	      | (BLe|BLt|BGt|BGe), EBool b, EBool c -> EBool((comp_op binop) b c)
	      | (BLe|BLt|BGt|BGe), EFloat f, EFloat g -> EBool((comp_op binop) f g)
	      | (BLe|BLt|BGt|BGe), EString s, EString t -> EBool((comp_op binop) s t)
	      | (BLe|BLt|BGt|BGe), ESymbol s, ESymbol t -> EBool((comp_op binop) s t)
	      | (BLe|BLt|BGt|BGe), _, _ ->
		runtime "can only compare ints, bools, floats, strings and symbols"
	      | BPlus, EInt i, EInt j -> EInt (i+j)
	      | BMinus, EInt i, EInt j -> EInt (i-j)
	      | BMul, EInt i, EInt j -> EInt (i*j)
	      | BDiv, EInt i, EInt j -> EInt (i/j)
	      | BMod, EInt i, EInt j -> EInt (i mod j)
	      | BPlusF, EFloat i, EFloat j -> EFloat(i+.j)
	      | BMinusF, EFloat i, EFloat j -> EFloat(i-.j)
	      | BMulF, EFloat i, EFloat j -> EFloat(i*.j)
	      | BDivF, EFloat i, EFloat j -> EFloat(i/.j)
	      | (BPlus|BMinus|BMul|BDiv|BMod|BPlusF|BMinusF|BMulF|BDivF), _, _ -> 
		runtime "can only do arithmetic on ints or floats"
	      | BConcat, EString i, EString j -> EString (i ^ j)
	      | BConcat, _, _ -> runtime "can only do string operations on strings"
	      | (BAnd|BOr), _, _ -> assert false
	   ), r), l, b
	  
	| ENot e1 -> (match evalp (e1, o) fcorec unks true with
	    | (EBool b, t), l, b1 -> (EBool (not b), t), l, b1
            | (r, t), l, b1 when not(b1) -> (ENot r, t), l, b1
	    | _ -> runtime "can only use negation on booleans")

	| EUnit -> (e, o), [ ], true
	| ENative _ -> (e, o), [ ], true

		(* EDummy is used internally *)
	| EDummy -> (e, o), [ ], true
        | EUnknown (ESymbol i) -> (match unks with
	  | None -> (EUnknown (ESymbol i), o), [ ], true (* why not false? *)
          | Some(b, t) -> (* if b is true it's a contriterator solver *)
	    try let r = Hashtbl.find t i in
		if b then match r with
		  | ETuple([r1; EInj _]) ->
		    (ETuple([r1; EVar (i ^ "_2")]), o), [ ], true
		  (* TODO: only on lists so far; extend to the general case *)
		  | _ -> (r, o), [ ], true
		else (r, o), [ ], true
            with Not_found -> (EUnknown (ESymbol i), o), [ ], true (* why not false? *))
	| ESymbol i -> (e, o), [ ], true (* appears on left side of equations *)
	| EUnknown _ -> assert false 

and eval_corec (solver:expr) (fname:id) (func:expr) 
    (arg, env:state) unks repl : state =
  let name, env, eqs = equations (arg, env) fname func solver in
  (* Printing equations for debugging purposes *)
  (*print_string
    ("Equations: find " ^ (Printing.to_string (name, env)) ^ " such that\n" ^
	(String.concat "\n"
	   (List.map (fun (i, rhs) -> (Printing.to_string (i, env)) ^ " = " ^ 
	     (Printing.to_string (rhs, Environment.empty))) eqs)));
  print_newline ();*)
  let id_of_expr = function ESymbol id -> id | _ -> assert false in
  let ids_of_eqs = function eqs ->
    List.map (fun (r, l) -> (id_of_expr r, l)) eqs in
  match solver with
    | EVar("constructor") -> 
      Constructor.solve (id_of_expr name) env (ids_of_eqs eqs)
   (* passing eval through is awkward *)
    | EApp(EVar "iterator", bot) ->
      let (bot1, env1), _, _ = evalp (bot, env) None unks repl in
      Iterator.solve 
	(fun x unks1 -> evalp x None (Some(false, unks1)) true) 
	bot1 (id_of_expr name) env1 (ids_of_eqs eqs)
    | EApp(EVar "constriterator", bot) ->
      let (bot1, env1), _, _ = evalp (bot, env) None unks repl in
      Constriterator.solve 
	(fun x unks1 -> evalp x None (Some(true, unks1)) true) 
	bot1 (id_of_expr name) env1 (ids_of_eqs eqs)
    | EVar("gaussian") -> Gaussian.solve (id_of_expr name) env (ids_of_eqs eqs)
    | EApp(EVar "appears", e) -> 
      let (e1, env1), _, _ = evalp (e, env) None unks repl in
      Appears.solve e1 name env1 eqs
      (* Solver defined by the user *)
    | _ -> (* Converting the equations to a CoCaml format *)
      let cocaml_eqs = List.fold_right (fun (v, rhs) acc ->
        EInj("::", Some(ETuple [ETuple [v; rhs]; acc]))
      ) eqs (EInj("[]", None)) in
      let usolver = match solver with
	| ETuple [_; _; s] -> s
	| _ -> solver in
      let (e1, env1), _, _ = 
	evalp (EApp(EApp(usolver, name), cocaml_eqs), env) None unks repl in
      (e1, env1)

and equations (arg, env: state) (fname : id) (func : expr) (solver : expr)
    (* solver arg only for Dexter's method *)
    : expr * env * (equation list) = 
  let (uunk, ufresh) = match solver with
    | ETuple [uunk; ufresh; _] -> 
      (fun (var, e) -> let (r, e1), _, _ = 
		    evalp (EApp(uunk, var), e) None None true in (r, e1)),
      (fun ((), e) -> let (r, e1), _, _ =
		   evalp (EApp(ufresh, EUnit), e) None None true in (r, e1))
    | _ -> (fun (x, e) -> EUnknown x, e), (fun ((), e) -> ESymbol(fresh()), e) in
  let rec equations_aux (name:expr) (arg, env: state) (eqs: equation list)
      (seen: (state * expr) list)
      (fname : id) (func : expr) unks
      : expr * env * (equation list) * ((state * expr) list) =
    let seen0 = ((arg, env), name) :: seen in
    let (eq_rhs, env0), argl, _ = 
      evalp (EApp(func, arg), env) 
	(Some (fname, seen0, (uunk, ufresh))) unks true in
    let eqs0 = (name, eq_rhs) :: eqs in
    let (envf, eqsf, seenf) = List.fold_left 
      (fun (envi, eqsi, seeni) (idi, argi) ->
	let (_, envj, eqsj, seenj) =
          equations_aux idi (argi, envi) eqsi seeni fname func unks
	in (envj, eqsj, seenj))
      (env0, eqs0, seen0) argl in
    (name, envf, eqsf, seenf) in
  let namef, envf, eqsf, _ =
    let freshid, env1 = ufresh ((), env) in
    let bool_unks = (match solver with 
	EApp(EVar "constriterator", _) -> true | _ -> false) in
    equations_aux (freshid) (arg, env1) [] [] fname func
      (Some(bool_unks, Hashtbl.create 1))
  in
  namef, envf, eqsf

and eval ((e, o) : state) : state =
  match evalp (e, o) None (Some(false, Hashtbl.create 1)) true with
    | s, [ ], true -> s
    | _ -> assert false
