open Ast
open Ast_types
open Map
open Util
open Str

exception Unbound of id

(* TODO: Change so that these types enforce what we want *)
type var = id (* These can be any variable string *)
type fv = id (* This should only be fresh variables. eg. strings starting w '%'
*)
type module_var = id (* These should be strings starting w a capital letter *)

(* EFunctional map with vars as keys *)
module EVarMap = Map.Make (struct
  type t = id
  let compare = String.compare
end)

module type FV = sig
  type data
  val free_vars : id Util.HashSet.t -> data -> unit
end
(*
module type ENVIRONMENT = functor (A : FV) ->
  sig
    type data = A.data
    type t
		type m
    val empty : t
    val bind : id -> data -> t -> t
    val bound : id -> t -> bool
		val module_bind : module_var -> m -> t -> t
		val module_bound : module_var -> t -> bool
		val empty_module : m
		val push_module : module_var -> m -> t -> t
		val pop_module : module_var -> t -> t
		val iter : (id -> data -> unit) -> t -> unit
    val map : (data -> data) -> t -> t
    val lookup : id -> t -> data
    val gc : id HashSet.t -> t -> t
  end
*)
module Dynamic_Environment = functor (A : FV) ->
	struct
		type data = A.data (* expr *)
    
    type module_elem = (fv EVarMap.t)*(fv EVarMap.t)
		type dyn_env = {
			moduleStack : module_elem list;
      expr_map : expr EVarMap.t;
      module_env : module_elem EVarMap.t
		}
		
    type t = dyn_env
		type m = module_elem
		
		(********************
		 * Helper EFunctions *
		*********************)
		
    let var_lookup (environment : t) (var_name : fv) : expr =
      EVarMap.find var_name environment.expr_map

		let module_lookup (environment : t) (m_v : fv) : module_elem =
			EVarMap.find m_v environment.module_env

		(* Searches through the environment, from top to bottom for the mapping of v
		to fv, and when found, returns the expr mapped to by fv *)
		let get_var (environment : t) v : expr option =
			let f (acc : 'a option) (elem : module_elem) =
				(match acc with
					| Some(x) -> acc
					| None ->
						try
							let fv = EVarMap.find v (fst elem) in
								Some(var_lookup environment fv)
						with Not_found -> None) in
				List.fold_left f None environment.moduleStack

		(* Searches through the environment, from top to bottom for the mapping of
		the module name to fv, and when found, returns the dynamic environment
		mapped to by fv *)
		let get_module (environment : t) (m_name : module_var) : m option =
			let f (acc : m option) (elem : m) =
				(match acc with
					| None ->
						(try
							let fv = EVarMap.find m_name (snd elem) in
								Some(module_lookup environment fv)
						with Not_found -> None)
					| Some(x) -> acc) in
				List.fold_left f None environment.moduleStack
			
		let update_var_in_environment (environment : t) (v : var) (fv_map : fv) : t =
      let m = List.hd environment.moduleStack in
        { moduleStack = ((EVarMap.add v fv_map (fst m)),(snd m))::
                         (List.tl environment.moduleStack);
         expr_map = environment.expr_map;
         module_env = environment.module_env }
	
		let update_module_in_environment (environment : t) (m_name : module_var)
			(fv_map : fv) : t =
      let m = List.hd environment.moduleStack in
        { moduleStack = ((fst m), (EVarMap.add m_name fv_map (snd m)))::
                        (List.tl environment.moduleStack);
          expr_map = environment.expr_map;
          module_env = environment.module_env }

		let update_expr_map (environment : t) (fv_name : fv) (e : expr) : t =
      { moduleStack = environment.moduleStack;
        expr_map = EVarMap.add fv_name e environment.expr_map;
        module_env = environment.module_env }

		let update_module_env_map (environment : t) (fv_name : fv) (module_map : m) : t =
      { moduleStack = environment.moduleStack;
        expr_map = environment.expr_map;
        module_env = EVarMap.add fv_name module_map environment.module_env }

		let get_vars (v : var) : (module_var list)*var =
			let lst = Str.split (Str.regexp "[.]") v in
				if List.length lst = 1 then ([],List.hd lst)
				else
					let rev_lst = List.rev lst in
						(List.rev(List.tl rev_lst), List.hd rev_lst)

		let lookup_in_module (environment : t) (modules : module_var list)
    (var_name : id) : expr =
			let fst_module = (match (get_module environment (List.hd modules)) with
				| None -> raise Not_found
				| Some(x) -> x) in
			let f (acc : m) (elem : module_var) : m =
				try
          EVarMap.find (EVarMap.find elem (snd acc)) environment.module_env
				with Not_found -> print_string "Module not found";
                          raise Not_found (*TODO: change error *) in
			let final_module = List.fold_left f fst_module (List.tl modules) in
				try
          EVarMap.find (EVarMap.find var_name (fst final_module))
            environment.expr_map
				with Not_found -> print_string "EVar in module not found";
                          raise Not_found (*TODO: change error *)

		let rhs_var_lookup (environment : t) v : expr option =
			try
				let (lst,var) = get_vars v in
					match lst with
            (* Look through entire stack for 'x' *)
						| [] ->	get_var environment v
            (* Only look through the modules for 'M1.x' *)
						| h::t -> Some(lookup_in_module environment lst var)
			with Not_found -> None

		let create_empty_module_elem () : m =
      let h1 : fv EVarMap.t = EVarMap.empty in
      let h2 : fv EVarMap.t = EVarMap.empty in
				(h1,h2)

		let create_new_dyn_env () : dyn_env =
			let h1 : expr EVarMap.t = EVarMap.empty in
			let h2 : module_elem EVarMap.t = EVarMap.empty in
        (* The toplevel is empty *)
				{ moduleStack = [(create_empty_module_elem ())];
				  expr_map = h1;
          module_env = h2 }

		let dynamic_var_bind v (e : expr) (environment : t) : t =
			let fresh_var = fresh () in
      let m = List.hd environment.moduleStack in
        { moduleStack = (EVarMap.add v fresh_var (fst m), (snd m))::
                        (List.tl environment.moduleStack);
          expr_map = EVarMap.add fresh_var e environment.expr_map;
          module_env = environment.module_env }

		let dynamic_module_bind (m : module_var) (elm : m) (environment : t) : t =
			let fresh_var = fresh () in
      let hd = List.hd environment.moduleStack in
        { moduleStack = ((fst hd), EVarMap.add m fresh_var (snd hd))::
                        (List.tl environment.moduleStack);
          expr_map = environment.expr_map;
          module_env = EVarMap.add fresh_var elm environment.module_env }

		let var_bound v (environment : t) : bool =
			match rhs_var_lookup environment v with
				| None -> false
				| Some(x) -> true

		let module_bound (v : module_var) (environment : t) : bool =
			match get_module environment v with
				| None -> false
				| Some(x) -> true

  	let empty = create_new_dyn_env ()
  	let bind = dynamic_var_bind
		let module_bind = dynamic_module_bind 
  	let bound = var_bound

		let empty_module : m = create_empty_module_elem ()

		let push_module (m : module_var) (elm : m) (environment : t) : t =
			let environment = dynamic_module_bind m elm environment in
				{ moduleStack = (elm::environment.moduleStack);
          expr_map = environment.expr_map;
				  module_env = environment.module_env }
				(* TODO : check this for efficnency *)

		let pop_module m (environment : t) : t =
      match environment.moduleStack with
        | h1::h2::t ->
          let fv = EVarMap.find m (snd h2) in
          let m_env = EVarMap.add fv h1 environment.module_env in
            { moduleStack = h2::t;
              expr_map = environment.expr_map;
              module_env = m_env }
        | _ -> print_string "Can't pop dynamic env module"; raise Not_found
				(* TODO : check this for efficnency *)

  	let iter (f : id -> expr -> unit) (env : t) : unit = EVarMap.iter f env.expr_map
		
    let map f env =
      { moduleStack = env.moduleStack;
        expr_map = EVarMap.map f env.expr_map;
        module_env = env.module_env }
  	
    let lookup var environment  =
			match rhs_var_lookup environment var with
				| None -> raise (Unbound var)
				| Some(x) -> x
      
    (* garbage collection *)
    (* starting at top-level, do bfs on the given environment *)
		(* TODO: Implement this *)
  	let gc (top : id HashSet.t) (o : t) : t = o
  end
  
module Environment = Dynamic_Environment (Ast)

module Static_Environment = functor (A : FV) ->
	struct
		type data = A.data (* schema *)
    
		type static_module_elem = {
			vars_map : data EVarMap.t;
			modules_map : static_module_elem EVarMap.t;
			typedefs_map : (id*typedef) list
		}

		type static_env = static_module_elem list
		
		type t = static_env
		type m = static_module_elem

		(********************
		 * Helper functions *
		 ********************)
		let get_var (environment : t) v : data option =
			let f (acc : data option) (elem : static_module_elem) =
				(match acc with
					| Some(x) -> acc
					| None ->
						try
							let s = EVarMap.find v elem.vars_map in
								Some(s)
						with Not_found -> None) in
					List.fold_left f None environment

    let get_module environment m : static_module_elem option =
			let f (acc : static_module_elem option) (elem : static_module_elem) =
				(match acc with
					| Some(x) -> acc
					| None ->
						try
							let s = EVarMap.find m elem.modules_map in
								Some(s)
						with Not_found -> None) in
				List.fold_left f None environment

    let get_vars v : (module_var list)*var =
      let lst = Str.split (Str.regexp "[.]") v in
        if List.length lst = 1 then ([], List.hd lst)
        else
          let rev_lst = List.rev lst in
            (List.rev(List.tl rev_lst), List.hd rev_lst)

    let lookup_in_module environment modules var : data =
      let fst_module = (match (get_module environment (List.hd modules)) with
        | None -> print_string "First module not found\n"; raise Not_found (* TODO: Change error *)
        | Some(x) -> x) in
      let f (acc : m) (elem : module_var) : m =
        try
          EVarMap.find elem acc.modules_map
        with Not_found -> print_string "Module not found in static search";
                          raise Not_found (* TODO: Change error *) in
      let final_module = List.fold_left f fst_module (List.tl modules) in
        try
          EVarMap.find var final_module.vars_map
        with Not_found -> print_string "EVar in module no found in static
        search"; raise Not_found (* TODO: Change error *)

    let rhs_var_lookup (environment : t) v : data option =
      try
        let (lst, var) = get_vars v in
          match lst with
            | [] -> get_var environment v
            | h::t -> Some(lookup_in_module environment lst var)
      with Not_found -> None

		let static_var_bind (v : id) (e : data) (environment : t) : t =
      let elm = List.hd environment in
      { vars_map = EVarMap.add v e elm.vars_map;
        modules_map = elm.modules_map;
        typedefs_map = elm.typedefs_map }::(List.tl environment)

		let static_module_bind (m : id) (elm : m) (environment : t) : t =
			let elm = List.hd environment in
      { vars_map = elm.vars_map;
        modules_map = EVarMap.add m elm elm.modules_map;
        typedefs_map = elm.typedefs_map }::(List.tl environment)
		
		let var_bound v (environment : t) : bool =
			match (rhs_var_lookup environment v) with
				| None -> false
				| Some(x) -> true

		let module_bound m (environment : t) : bool =
			match (get_module environment m) with
				| None -> false
				| Some(x) -> true
		
		let empty_module : m =
			let h1 : data EVarMap.t = EVarMap.empty in
			let h2 : m EVarMap.t = EVarMap.empty in
				{ vars_map = h1; modules_map = h2; typedefs_map = [] }

    let create_new_static_env () : static_env =
      [(empty_module)]

    (* Yea functional programming is so SEXY! :D *)
		let static_map (f : data -> data) (environment : t) : t =
      List.map (fun x ->
                  {vars_map = EVarMap.map f x.vars_map;
                   modules_map = x.modules_map;
                   typedefs_map = x.typedefs_map }) environment
		
		let bind  = static_var_bind
		let bound =  var_bound
		let module_bind = static_module_bind
		let module_bound = module_bound
		
		let push_module m_v elm (environment : t) : t =
			let environment = static_module_bind m_v elm environment in
			  elm::environment
		
		let empty = create_new_static_env ()
		
		let iter (f : id -> data -> unit) (environment : t) : unit =
      List.iter (fun x -> EVarMap.iter f x.vars_map) environment

		let lookup var environment =
			match rhs_var_lookup environment var with
				| None -> raise (Unbound var)
				| Some(x) ->  x

		let pop_module mv (environment : t) : t =
      match environment with
       | h1::h2::t -> ({ vars_map = h2.vars_map;
                         modules_map = EVarMap.add mv h1 h2.modules_map;
                         typedefs_map = h2.typedefs_map })::t
       | _ -> print_string "pop_module called incorrectly"; raise Not_found
		
		let map (f : data -> data) (environment : t) : t =
			static_map f environment

    (* garbage collection *)
    (* starting at top-level, do bfs on the given environment *)
		(* TODO: Implement this *)
  	let gc (top : id HashSet.t) (o : t) : t = o

    let get_all_typedefs (environment : t) : (id*typedef) list =
      let lst = List.fold_left (fun acc elm -> elm.typedefs_map::acc) []
      environment in
        List.flatten (List.rev lst)

    let get_module_typedefs (environment : t) modules : (id*typedef) list =
      let fst_module = (match (get_module environment (List.hd modules)) with
        | None -> print_string "First module not found for typedef\n"; raise
        Not_found
        |Some(x) -> x) in
      let f (acc : m) (elem : module_var) : m =
        try EVarMap.find elem acc.modules_map
        with Not_found -> print_string "Module not found in static td search";
                          raise Not_found in
      let final_module = List.fold_left f fst_module (List.tl modules) in
        final_module.typedefs_map

    let set_typedefs (environment : t) (typedefs_lst : (id*typedef) list) : t =
      let hd = List.hd environment in
      { vars_map = hd.vars_map;
        modules_map = hd.modules_map;
        typedefs_map = typedefs_lst }::(List.tl environment)

    let append_typedef (environment : t) (td : (id * typedef)) : t =
      let hd = List.hd environment in
      { vars_map = hd.vars_map;
        modules_map = hd.modules_map;
        typedefs_map = td::(hd.typedefs_map) }::(List.tl environment)

    let get_typedefs_list (environment : t) (t_name : id) : (id * (id * typedef)
    list) =
      let (lst, var) = get_vars t_name in
        match lst with
          | [] -> (t_name, get_all_typedefs environment)
          | h::t -> (var, get_module_typedefs environment lst)
  end

module Environment_types = Static_Environment (Ast_types)

type env = Environment.t
type env_types = Environment_types.t
type state = expr * env
