open Util

type id = string
type idtype = string
type unknown = string (* from Gensym *)

let fresh : unit -> id =
	let gs : Gensym.t = Gensym.make () in
		fun () -> Gensym.next gs

(* typ = type NoCaml *)
type typ =
  | TInt
  | TFloat
  | TString
  | TBool
  | TUnit
  | TVoid
  | TArrow of typ * typ
  | TVar of idtype
  | TTuple of typ list
  | TUser of string * typ list
  (* just the name of the type, followed by the list of argument types
     those argument types could be EVarType *)
  | TSymbol
  | TDummy
  (* Null means no type specified. It is used in the parser to handle the cases
     when the programmer doesn't declare all the types.
     EFor example, "fun x (y : int) -> x" creates the AST node
     EFunType ([TVar x; TVar y], [TNull; TInt], TVar x) *)
  | TNull

type pattern =
  | PUnderscore 
  | PVar of id
  | PInt of int
  | PFloat of float
  | PString of string
  | PBool of bool
  | PUnit
  | PTuple of pattern list
  | PInj of id * pattern option
  | PUnknown of pattern

type binop = 
  (* Comparison operators *)
  | BEq | BNeq | BLt | BLe | BGt | BGe
  (* Arithmetic operators on integers*)
  | BPlus | BMinus | BMul | BDiv | BMod
  (* Arithmetic operators on floats *)
  | BPlusF | BMinusF | BMulF | BDivF
  (* EString operators *)
  | BConcat
  (* EBoolean operators *)
  | BAnd | BOr

type expr =
  | EVar of id
  | EInt of int
  | EFloat of float
  | EString of string
  | EBool of bool
  | EFun of expr list * expr
  | EFunType of expr list * typ list * expr
  | EFunCorec of expr * id * expr list * expr
  | EFunction of (pattern * expr) list
  | ELetType of id * typ * expr list * typ list* expr * expr
  | ELetrecType of id * typ * expr list * typ list * expr * expr
  | ELetcorecType of expr * id * typ * expr list * typ list * expr * expr
  | EApp of expr * expr
  | EAssign of id * expr
  | ESeq of expr * expr
  | EIf of expr * expr * expr
  | EWhile of expr * expr
  | EFor of id * expr * expr * expr
  | EBinop of binop * expr * expr
  | ENot of expr
  (* EUnit *)
  | EUnit
  (* Tuples *)
  | ETuple of expr list
  (* Sum types *)
  | EInj of string * (expr option)
  (* Match *)
  | EMatch of expr * ((pattern * expr) list)
  (* Used internally *)
  | EDummy
  | EUnknown of expr (* in practice, expr is always a ESymbol *)
  | ESymbol of unknown (* for use in corecursive functions *)
  | ENative of id * (expr -> expr) * typ

type data = expr

type directive =
  | Load of string
  | Chdir of string
  | Pwd
  | Ls
  | Env
  | Scope
  | Type_inf
  | Quit

type typedef = (idtype list) * ((id * typ option) list)
(* type name, arguments as variables, list of contructors with arguments *)
(* for example 
   type ('a, 'b) tree = A of 'a | B of 'b 
   | T of tree * tree | E
   is represented as:
   ("tree", ["'a" ; "'b"], 
    ["A", Some (EVarType "'a"); "B", Some (EVarType "'b"); 
     "T", Some (ETupleType [UserType "tree"; UserType "tree"]);
     "E", None)
*)
   
type declaration =
  | Def of id * typ * expr list * typ list * expr
  | Defrec of id * typ * expr list * typ list * expr
  | Defcorec of expr * id * typ * expr list * typ list * expr
  | Typedef of id * typedef
  | Module of id * declaration list

type toplevel =
  | Directive of directive
  | Declarations of declaration list
  | Expression of expr

(* character encoding *)
let utf8 = ref false

let rec bound_vars (p:pattern) = match p with
  | PVar x -> [ x ]
  | PTuple l -> List.flatten(List.map bound_vars l)
  | PInj(_, None) -> [ ]
  | PInj(_, Some p) -> bound_vars p
  | PUnknown p -> bound_vars p
  | PInt _ | PFloat _ | PString _ | PBool _ | PUnit | PUnderscore -> [ ]

(* Collect all free variables in an expression *)
let free_vars (fv : id Util.HashSet.t) (e : expr) : unit =
  (* exclude bound vars in scope *)
  let free (bv : id list) (x : id) : unit =
    if not (List.mem x bv) then Util.HashSet.add fv x in
  let args s = List.filter ((!=) "()") (List.map (function (EVar x) -> x | _ -> "()") s) in
  let rec free_vars (bv : id list) (e : expr) : unit =
    match e with
    (* Binding operators *)
    | EFun (x, e) -> free_vars (args x @ bv) e
    | EFunType (x, _, e) -> free_vars bv (EFun (x, e))
    | EFunCorec (_, _, x, e) -> free_vars bv (EFun (x, e))
    | EFunction l ->
      List.iter (fun (pi, ei) -> free_vars ((bound_vars pi) @ bv) ei) l
    | ELetType (f, _, x, _, e1, e2) -> 
      free_vars (args x @ bv) e1; free_vars (f :: bv) e2
    | ELetrecType (f, _, x, _, e1, e2) -> 
      free_vars (f :: args x @ bv) e1; free_vars (f :: bv) e2
    | ELetcorecType(_, f, tf, x, tx, e1, e2) ->
        free_vars bv (ELetrecType(f, tf, x, tx, e1, e2))
    (* Non-binding operators *)
    | EVar x -> free bv x
    | EAssign (x, e) -> free bv x; free_vars bv e
    | EFor (x, e1, e2, e3) -> free bv x; List.iter (free_vars bv) [e1; e2; e3]
    | EMatch (e, l) -> 
      free_vars bv e; 
      List.iter (fun (pi, ei) -> free_vars ((bound_vars pi) @ bv) ei) l
    | (EInt _ | EFloat _ | EString _ | EBool _ | EDummy | ENative _ | EUnit) -> ()
    | EBinop(_, e1, e2) | EApp(e1, e2) | ESeq(e1, e2) | EWhile(e1, e2) ->
      free_vars bv e1; free_vars bv e2
    | EIf (e1, e2, e3) -> List.iter (free_vars bv) [e1; e2; e3]
    | ENot e1 -> free_vars bv e1 
    | ETuple l -> List.iter (free_vars bv) l 
    | EInj(_, None) -> ( )
    | EInj(_, Some e1) -> (free_vars bv e1)
    | EUnknown _ | ESymbol _ -> ( ) in
  free_vars [] e

let rec dump : expr -> string = function (* to debug. TODO: finish it *)
  | EVar x -> "EVar(" ^ x ^ ")"
  | EInt i -> "EInt(" ^ (string_of_int i) ^ ")"
  | EFloat f -> "EFloat(" ^ (string_of_float f) ^ ")"
  | EString s -> "EString " ^ s ^ ")"
  | EBool b -> "EBool(" ^ (string_of_bool b) ^ ")"
  | EFun(el, e) -> 
    "EFun([" ^ (String.concat "; " (List.map dump el)) ^ "], " ^ (dump e) ^ ")"
  | EFunType(el, _, e) ->
    "EFunType([" ^ (String.concat "; " (List.map dump el)) ^ "], ?," ^ 
      (dump e) ^ ")"
  | EFunCorec(s, i, el, e) -> 
    "EFunCorec(" ^ (dump s) ^ ", " ^ i ^ ", [" ^
      (String.concat "; " (List.map dump el)) ^ "], " ^ (dump e) ^ ")"
  | EFunction _ -> "EFunction(?)"
  | ELetType(i, _, el, _, e1, e2) ->
    "Let(" ^ i ^ ", ?, [" ^ (String.concat "; " (List.map dump el)) ^ "], ?, " ^
      (dump e1) ^ ", " ^ (dump e2) ^ ")"
  | ELetrecType(i, _, el, _, e1, e2) ->
    "ELetrecType(" ^ i ^ ", ?, [" ^ (String.concat "; " (List.map dump el)) ^ 
      "], ?, " ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  | ELetcorecType(s, i, _, el, _, e1, e2) ->
    "ELetcorecType(" ^ (dump s) ^ ", " ^ i ^ ", ?, [" ^ 
      (String.concat "; " (List.map dump el)) ^ 
      "], ?, " ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  | EApp(e1, e2) -> "EApp(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  | EAssign(i, e2) -> "EAssign(" ^ i ^ ", " ^ (dump e2) ^ ")"
  | ESeq(e1, e2) -> "ESeq(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  | EIf(e1, e2, e3) -> 
    "EIf(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ", " ^ (dump e2)^ ")"
  | EWhile(e1, e2) -> "EWhile(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  | EFor(i, e1, e2, e3) -> 
    "EFor(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ", " ^ (dump e3) ^ ")"
  (* Binary operators *)
  | EBinop(b, e1, e2) ->
    let dump_binop = function BEq -> "Eq" | BNeq -> "Neq"
      | BLt -> "Lt" | BLe -> "Le" | BGt -> "Gt" | BGe -> "Ge"
      | BPlus -> "Plus" | BMinus -> "Minus" | BMul -> "Mul" | BDiv -> "Div"
      | BMod -> "Mod" 
      | BPlusF -> "PlusF" | BMinusF -> "MinusF" | BMulF -> "MulF" | BDivF -> "DivF"
      | BConcat -> "Concat" | BAnd -> "And" | BOr -> "Or"
    in "EBinop(" ^ (dump_binop b) ^ ", " ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  | ENot(e1) -> "ENot(" ^ (dump e1) ^ ")"
  (* EUnit *)
  | EUnit -> "EUnit"
  (* ETuples *)
  | ETuple(el) -> 
    "ETuple [" ^ (String.concat "; " (List.map dump el)) ^ "]"
  (* Sum types *)
  | EInj(i, None) -> "EInj(" ^ i ^ ", None)"
  | EInj(i, Some(e)) -> "EInj(" ^ i ^ ", " ^ "Some(" ^ (dump e) ^ "))"
  (* EMatch *)
  | EMatch _ -> "EMatch(?)"
  (* Used internally *)
  | EDummy -> "EDummy"
  | EUnknown e1 -> "EUnknown(" ^ (dump e1) ^ ")" 
  | ESymbol id -> "ESymbol(" ^ id ^ ")"
  (* for use in corecursive functions *)
  | ENative(i, _, t) -> "ENative(" ^ i ^ ", " ^ "<fun>" ^ ")"

exception Runtime of string
(* for runtime errors other than unbound vars *)
let runtime (s : string) : 'a = raise (Runtime s)
