
open Ast

(* A type schema is made up of two elements:
   - a list of (universally quantified) type variables, and
   - a type
EFor example, fun x -> x has the schema "for all t1, t1 -> t1" represented 
by Schema ([t1], Arrow (t1, t1)) *)

type schema = Schema of id list * typ
type data = schema

(* Collect all free variables in a type expression *)
let free_vars (fv : id Util.HashSet.t) (Schema (bv, t) : schema) : unit =
  let rec free_vars (e : typ) : unit =
  match e with
    | (TInt | TFloat | TString | TBool | 
	TDummy | TUnit | TVoid | TNull | TSymbol) -> ()
    | TArrow (t1, t2) -> free_vars t1; free_vars t2
    | TVar x -> if not(List.mem x bv) then Util.HashSet.add fv x
    | TTuple l -> List.iter free_vars l
    | TUser(_, l) -> List.iter free_vars l in
  free_vars t

let to_string (t : typ) : string =
  (* convert to fresh variables *)
  let ls = Util.LexStream.make() in
  let fresh_vars = Hashtbl.create 11 in
  let rec to_string (t : typ) : string =
    match t with
      | TInt -> "int"
      | TFloat -> "float"
      | TString -> "string"
      | TBool -> "bool"
      | TUnit -> "unit"
      | TVoid -> "void"
      | TSymbol -> "symbol"
      | TArrow (TArrow _ as t1, t2) -> let s1 = to_string t1 in
				       let s2 = to_string t2 in
				       Printf.sprintf "(%s) -> %s" s1 s2
      | TArrow (t1, t2) -> let s1 = to_string t1 in
			   let s2 = to_string t2 in
			   Printf.sprintf "%s -> %s" s1 s2
      | TVar x ->
        (try Hashtbl.find fresh_vars x
         with Not_found -> let res = Util.LexStream.next ls in
			   Hashtbl.add fresh_vars x res; res)
      | TDummy -> "internal: dummy"
      | TNull -> "null" 
      | TTuple l -> 
	"(" ^ (String.concat " * " (List.map to_string l)) ^ ")"
      | TUser(s, [ ]) -> s
      | TUser(s, [ t1 ]) -> (to_string t1) ^ " " ^ s
      | TUser(s, l) -> 
	"(" ^ (String.concat " * " (List.map to_string l)) ^ ") " ^ s
     in
  to_string t

(* EFor debugging purposes *)
(* Prints an equation system *)
let print_aux l =
  let f (x, y) = Printf.printf "%s = %s\n" (to_string x) (to_string y) in
  print_endline "Equations start:";
  List.iter f l;
  print_endline "Equations end"
