%{
open Ast
%}

%token FUN FUNCTION LET REC COREC IN LOAD
%token EQ NEQ LT LE GT GE
%token PLUS MINUS MUL DIV CONCAT PLUSF MINUSF MULF DIVF MOD
%token COMMA
%token <int> INT
%token <float> FLOAT
%token <string> VAR
%token <string> CVAR
%token <string> STRING
%token IF THEN ELSE WHILE DO FOR TO DONE MATCH WITH
%token TRUE FALSE
%token SEQ IMP ASSG
%token CONS
%token LPAREN RPAREN LBRACKET RBRACKET
%token NOT AND OR
%token EOF
%token LOAD CHDIR PWD LS ENV SCOPE TYPE_INF QUIT
%token TYPEDEF OF BAR
%token <string> TYPE
%token TYPEDECL
%token MODULE SIG STRUCT END VALDEF
%token <string> TYPEVAR
%token UNDERSCORE
%token DOT

%nonassoc RPAREN RBRACKET /* lowest precedence - always reduce */
%right IMP
%left SEQ
%right ASSG
%left COMMA DOT
%nonassoc CVAR
%left OR
%left AND
%left EQ NEQ LT LE GT GE
%right CONS
%left PLUS MINUS PLUSF MINUSF
%left MUL DIV MOD MULF DIVF CONCAT
%nonassoc NOT
%nonassoc INT FLOAT VAR TRUE FALSE STRING TYPE UNDERSCORE
%nonassoc MODULEACESS
%left BAR
%nonassoc LPAREN LBRACKET FUN FUNCTION LET IN ELSE IF WHILE FOR TYPEDECL TYPEVAR MATCH WITH
/* highest precedence - always shift */

/* entry points */
%start parse
%type <Ast.toplevel> parse

%%

parse:
  | directive EOF { Directive $1 }
  | declaration_list EOF { Declarations $1 }
  | expr EOF { Expression $1 }
;

directive:
  | LOAD STRING  { Load $2 }
  | CHDIR STRING { Chdir $2 }
  | PWD { Pwd }
  | LS  { Ls }
  | ENV { Env }
  | SCOPE { Scope }
  | TYPE_INF { Type_inf }
  | QUIT  { Quit }
;  

sumtype:
  | CVAR OF parse_type { ($1, Some($3)) }
  | CVAR { ($1, None) }
;

sumtypes:
  | sumtype BAR sumtypes { $1 :: $3 }
  | sumtype { [$1] }
;

sumtypes_bar:
  | sumtypes { $1 }
  | BAR sumtypes { $2 }
;

declaration_list:
  | declaration declaration_list { $1 :: $2 }
  | declaration { [$1] }
;
/*
funprop:
  | { 0 }
  | REC { 1 }
  | COREC { 2 }
;
*/
declaration:
	| LET parse_var EQ expr %prec RPAREN
      { Def ($2, TNull, [], [], $4) }
	| LET parse_var TYPEDECL parse_type EQ expr %prec RPAREN
      { Def ($2, $4, [], [], $6) }
	| LET LPAREN parse_var TYPEDECL parse_type RPAREN EQ expr %prec RPAREN
      { Def ($3, $5, [], [], $8) }
	| LET parse_var typelist EQ expr %prec RPAREN
      { Def($2, TNull, fst (List.split $3), snd (List.split $3), $5) }
	| LET parse_var typelist TYPEDECL parse_type EQ expr %prec RPAREN
      { Def($2, $5, fst (List.split $3), snd (List.split $3), $7) }
	| LET REC parse_var EQ expr %prec RPAREN
      { Defrec ($3, TNull, [], [], $5) }
	| LET REC parse_var TYPEDECL parse_type EQ expr %prec RPAREN
      { Defrec ($3, $5, [], [], $7) }
	| LET REC LPAREN parse_var TYPEDECL parse_type RPAREN EQ expr %prec RPAREN
      { Defrec ($4, $6, [], [], $9) }
	| LET REC parse_var typelist EQ expr %prec RPAREN
      { Defrec($3, TNull, fst (List.split $4), snd (List.split $4), $6) }
	| LET REC parse_var typelist TYPEDECL parse_type EQ expr %prec RPAREN
      { Defrec($3, $6, fst (List.split $4), snd (List.split $4), $8) }
	| LET COREC LBRACKET expr RBRACKET parse_var EQ expr %prec RPAREN
      { Defcorec ($4, $6, TNull, [], [], $8) }
	| LET COREC LBRACKET expr RBRACKET parse_var TYPEDECL parse_type EQ expr %prec RPAREN
      { Defcorec ($4, $6, $8, [], [], $10) }
	| LET COREC LBRACKET expr RBRACKET LPAREN parse_var TYPEDECL parse_type RPAREN EQ expr %prec RPAREN
      { Defcorec ($4, $7, $9, [], [], $12) }
	| LET COREC LBRACKET expr RBRACKET parse_var typelist EQ expr %prec RPAREN
      { Defcorec($4, $6, TNull, fst (List.split $7), snd (List.split $7), $9) }
	| LET COREC LBRACKET expr RBRACKET parse_var typelist TYPEDECL parse_type EQ expr %prec RPAREN
      { Defcorec($4, $6, $9, fst (List.split $7), snd (List.split $7), $11) }
        | TYPEDEF parse_var EQ sumtypes_bar { Typedef($2, ([ ], $4)) }
        | TYPEDEF typevar parse_var EQ sumtypes_bar { Typedef($3, ($2, $5)) }
	| MODULE CVAR EQ STRUCT declaration_list END %prec RPAREN
			{ Module ($2, $5) }
;

typevar:
   | TYPEVAR { [ $1 ] }
   | LPAREN typevars RPAREN { $2 }
;

typevars: 
   | TYPEVAR { [ $1 ] }
   | TYPEVAR COMMA typevars { $1 :: $3 }
;

parse_var:
   | VAR { $1 }
   | LPAREN parse_var RPAREN { $2 }
;

typelist:   
   | LPAREN parse_var TYPEDECL parse_type RPAREN { [(EVar $2, $4)] }
   | LPAREN parse_var TYPEDECL parse_type RPAREN typelist { (EVar $2, $4) :: $6}
   | parse_var { [(EVar $1, TNull)] } 
   | parse_var typelist { (EVar $1, TNull) :: $2 }
   | LPAREN LPAREN RPAREN TYPEDECL parse_type RPAREN { [(EUnit, $5)] }
   | LPAREN LPAREN RPAREN TYPEDECL parse_type RPAREN typelist { (EUnit, $5) :: $7 }
   | LPAREN RPAREN { [(EUnit, TNull)] }
   | LPAREN RPAREN typelist { (EUnit, TNull) :: $3 }
;

product_type :
  | product_type MUL parse_type { $3 :: $1 } /* 1 shift/reduce conflict */
  | parse_type MUL parse_type { [ $3 ; $1 ] }
;

parse_types :
  | parse_type COMMA parse_types { $1 :: $3 }
  | parse_type COMMA parse_type { [ $1 ; $3 ] }
;

parse_type : 
	| TYPE { match $1 with
	    | "int" -> TInt
	    | "string" -> TString
	    | "bool" -> TBool
	    | "unit" -> TUnit
            | "float" -> TFloat
	    | "void" -> TVoid
	    | _ -> raise Parse_error }
        | VAR { TUser($1, [ ]) }
	| parse_type VAR { TUser($2, [ $1 ]) }
	| LPAREN parse_types RPAREN VAR { TUser($4, $2) }
	| TYPEVAR { TVar $1 }
	| TYPEVAR VAR { TUser($2, [ TVar $1 ]) }
	| parse_type IMP parse_type { TArrow ($1, $3) }
	| LPAREN parse_type RPAREN { $2 }
        | product_type { TTuple (List.rev $1) }
;

arg:
  | VAR { EVar $1 }
  | INT { EInt $1 }
  | FLOAT { EFloat $1 }
  | STRING { EString $1 }
  | TRUE { EBool true }
  | FALSE { EBool false }
  | LPAREN RPAREN { EUnit }
  | LPAREN expr RPAREN { $2 }
  | LBRACKET RBRACKET { EInj( "[]", None ) }
  | LBRACKET expr RBRACKET {
    (* solving the ambiguity on the semicolumn, which can be in a list
       or a sequence of instructions *)
    let rec convert tail = function
    (* ; is left associative whereas :: is right associative:
       we need to reverse everything *)
      | ESeq(hd, tl) ->
	convert (EInj("::", Some(ETuple [ tl; tail ]))) hd
      | e -> EInj("::", Some (ETuple [e; tail])) 
    in let res = convert (EInj("[]", None)) $2 in 
       res
  }
  | CVAR { EInj($1, None) }
;

tuple:
  | tuple COMMA expr { $3 :: $1 } /* 1 shift/reduce conflict */
  | expr COMMA expr { [ $3 ; $1 ] }
;

pattern_list:
  | pattern { PInj("::", Some(PTuple [$1; PInj("[]", None)])) }
  | pattern SEQ pattern_list { PInj("::", Some(PTuple [$1; $3])) }
;

pattern_tuple:
  | pattern_tuple COMMA pattern { $3 :: $1 } /* 1 shift/reduce conflict */
  | pattern COMMA pattern { [ $3; $1 ] }
;

pattern:
  | LPAREN pattern RPAREN { $2 }
  | VAR { PVar $1 }
  | INT { PInt $1 }
  | FLOAT { PFloat $1 }
  | STRING { PString $1 }
  | TRUE { PBool true }
  | FALSE { PBool false }
  | LPAREN RPAREN { PUnit }
  | pattern CONS pattern { PInj("::", Some (PTuple [$1; $3])) }
  | LBRACKET RBRACKET { PInj("[]", None) }
  | LBRACKET pattern_list RBRACKET { $2 }
  | pattern_tuple { PTuple (List.rev $1) }
  | CVAR { PInj($1, None) }
  | CVAR pattern { PInj($1, Some($2)) }
  | UNDERSCORE { PUnderscore }
;

pattern_expr:
  | pattern IMP expr { ($1, $3) }
;

patterns_expr: /* suppressing one or the other rule suppresses 1 shift/reduce conflict */
  | pattern_expr %prec RPAREN { [ $1 ] }
  | pattern_expr BAR patterns_expr { $1 :: $3 }
;

patterns:
  | patterns_expr { $1 }
  | BAR patterns_expr { $2 }
;

expr:
  | FUN typelist IMP expr %prec RPAREN { EFunType ( fst ( List.split $2), snd (List.split $2), $4) }
  | declaration IN expr %prec RPAREN {
    match $1 with
      | Def (x, typeX, a, typeA, e) -> ELetType (x, typeX, a, typeA, e, $3)
      | Defrec (x, typeX, a, typeA, e) -> ELetrecType (x, typeX, a, typeA, e, $3)
      | Defcorec (s, x, typeX, a, typeA, e) -> ELetcorecType(s, x, typeX, a, typeA, e, $3)
      | Typedef _ -> raise Parse_error
			| Module _ -> print_string "Modules not allowed to be declared in LET ... IN"; raise Parse_error
    }
  | FUNCTION patterns { EFunction $2 }
  | VAR ASSG expr { EAssign ($1, $3) }
  | expr SEQ expr { ESeq ($1, $3) }
  | IF expr THEN expr ELSE expr %prec RPAREN { EIf ($2, $4, $6) }
  | WHILE expr DO expr DONE { EWhile ($2, $4) }
  | FOR VAR EQ expr TO expr DO expr DONE { EFor ($2, $4, $6, $8) }
  /* can't factorize with a binop because of priorities */
  | expr EQ expr { EBinop (BEq, $1, $3) }
  | expr NEQ expr { EBinop (BNeq, $1, $3) }
  | expr LT expr { EBinop (BLt, $1, $3) }
  | expr LE expr { EBinop (BLe, $1, $3) }
  | expr GT expr { EBinop (BGt, $1, $3) }
  | expr GE expr { EBinop (BGe, $1, $3) }
  | expr PLUS expr { EBinop (BPlus, $1, $3) }
  | expr MINUS expr { EBinop (BMinus, $1, $3) }
  | expr MUL expr { EBinop (BMul, $1, $3) }
  | expr DIV expr { EBinop (BDiv, $1, $3) }
  | expr MOD expr { EBinop (BMod, $1, $3) }
  | expr PLUSF expr { EBinop (BPlusF, $1, $3) }
  | expr MINUSF expr { EBinop (BMinusF, $1, $3) }
  | expr MULF expr { EBinop (BMulF, $1, $3) }
  | expr DIVF expr { EBinop (BDivF, $1, $3) }
  | expr CONCAT expr { EBinop (BConcat, $1, $3) }
  | expr AND expr { EBinop (BAnd, $1, $3) }
  | expr OR expr { EBinop (BOr, $1, $3) }
  | MINUS expr { EBinop (BMinus, EInt 0, $2) }
  | MINUSF expr { EBinop (BMinusF, EFloat 0., $2) }
  | NOT expr { ENot ($2) }
  | expr CONS expr { EInj("::", Some (ETuple [$1; $3])) }
  | expr arg { EApp ($1, $2) }
  | arg { $1 }
  | tuple { ETuple(List.rev $1) }
  | CVAR arg { EInj($1, Some($2)) }
  | MATCH expr WITH patterns { EMatch($2, $4) }
;

%%
