{
open Parser
exception LexError of string

let keywords =
    [
     "fun", FUN;
     "function", FUNCTION;
     "let", LET;
     "rec", REC;
     "corec", COREC;
     "in", IN;
     "if", IF;
     "then", THEN;
     "else", ELSE;
     "true", TRUE;
     "false", FALSE;
     "while", WHILE;
     "do", DO;
     "for", FOR;
     "to", TO;
     "done", DONE;
     "not", NOT;
     "type", TYPEDEF;
     "of", OF;
     "match", MATCH;
     "with", WITH;
     "mod", MOD;
		 "module", MODULE; (* Keywords added for modules*)
		 "sig", SIG;
		 "struct", STRUCT;
		 "end", END;
		 "val", VALDEF;
     "int", TYPE "int" ;
     "string", TYPE "string";
     "bool", TYPE "bool" ;
     "unit", TYPE "unit" ;
     "void", TYPE "void" ;
     "float", TYPE "float";
    ]


let keyword_tbl = Hashtbl.create 16
let _ = List.iter (Util.uncurry (Hashtbl.add keyword_tbl)) keywords 
let newline lexbuf =
    let curr_p = lexbuf.Lexing.lex_curr_p in
    lexbuf.Lexing.lex_curr_p <-
        { curr_p with
          Lexing.pos_lnum = curr_p.Lexing.pos_lnum + 1;
          Lexing.pos_bol = curr_p.Lexing.pos_cnum }
}

let id_elem = ['a'-'z'] ['_' '\'' 'A'-'Z' 'a'-'z' '0'-'9']*
let cid_elem = ['A'-'Z'] ['_' '\'' 'A'-'Z' 'a'-'z' '0'-'9']*

let cid = cid_elem(['.']cid_elem)*
let id = (cid['.'])*id_elem

let nl = ['\n' '\r']
let ws = ['\n' '\t' '\r' ' ']
let int = ['0'-'9']+
let float = ['0'-'9']+['.']['0'-'9']*
let directive = '#' id


rule token = parse
  | '\n'   { newline lexbuf; token lexbuf }
  | ws     { token lexbuf }
  | id as id {
		if String.contains id '.' then
			let loc = String.rindex id '.' in
			let id_elm = String.sub id (loc+1) ((String.length id)-loc-1) in
				try Hashtbl.find keyword_tbl
					(String.concat "." [(String.sub id 0 loc);(String.lowercase id_elm)])
				with Not_found -> VAR id
		else
			try Hashtbl.find keyword_tbl id with Not_found -> VAR id }
  | cid as cid { CVAR cid }
	(*
	| module_id as module_id { MODULEVAR module_id }
	| module_access as module_access { MODULEACCESS module_access }
	*)
  | int as int { INT (int_of_string int) }
  | float as float { FLOAT (float_of_string float) }
  | "\""   { STRING (String.concat "" (string lexbuf)) }
  | "(*"   { comment lexbuf }
  | ';'    { SEQ }
  | "->"   { IMP }
  | '='    { EQ }
  | "=="   { EQ }
  | "!="   { NEQ }
  | "<>"   { NEQ }
  | '<'    { LT }
  | "<="   { LE }
  | '>'    { GT }
  | ">="   { GE }
  | "+."   { PLUSF }
  | "-."   { MINUSF }
  | "*."   { MULF }
  | "/."   { DIVF }
  | '+'    { PLUS }
  | '-'    { MINUS }
  | '*'    { MUL }
  | '/'    { DIV }
  | ":="   { ASSG }
  | ':'    { TYPEDECL } 
  | "^"    { CONCAT }
  | '('    { LPAREN }
  | ')'    { RPAREN }
  | '['    { LBRACKET }
  | ']'    { RBRACKET }
  | "::"   { CONS }
  | "&&"   { AND }
  | "||"   { OR }
  | ","    { COMMA }
  | '\''   { typevar lexbuf }
  | "|"    { BAR }
  | "_"    { UNDERSCORE }
	| "."		 { DOT }
  (* top-level directives *)
  | "#load"{ LOAD }
  | "#cd"  { CHDIR }
  | "#ls"  { LS }
  | "#env" { ENV }
  | "#pwd" { PWD }
  | "#scope" { SCOPE }
  | "#inf" {TYPE_INF}
  | "#quit"{ QUIT }
  | eof    { EOF }
  | _      { raise (LexError "Unknown symbol") }

and string = parse
  | "\\\\" { "\\" :: string lexbuf }
  | "\\\"" { "\"" :: string lexbuf }
  | "\\n"  { "\n" :: string lexbuf }
  | "\\t"  { "\t" :: string lexbuf }
  | "\""   { [] }
  | eof    { raise (LexError "Unexpected end of input stream") }
  | _ as c { String.make 1 c :: string lexbuf }

and typevar = parse
  | id as id { TYPEVAR ("'" ^ id) }
  | _        { raise (LexError "Improper type variable") }

and comment = parse
  | "*)"   { token lexbuf }
  | eof    { raise (LexError "Unexpected end of input stream") }
  | _      { comment lexbuf }
