open Ast
open Ast_types
open Eval
open Environment
open Environment_types
open Modular_Environment

(*type env = Environment.t
type env_types = Environment_types.t
type typedefs = typedef list

type env_stack = env*env_types*typedefs list
*)
exception Parse_error of exn * int * int * string
                              
let parse (s : string) : Ast.toplevel =
  let lexbuf = Lexing.from_string s in
  try Parser.parse Lexer.token lexbuf
  with exn ->
    let curr = lexbuf.Lexing.lex_curr_p in
    let line = curr.Lexing.pos_lnum in
    let char = curr.Lexing.pos_cnum - curr.Lexing.pos_bol in
    let token = Lexing.lexeme lexbuf in
    raise (Parse_error (exn, line, char, token))

(* top-level bindings *)
let top = ref (Util.HashSet.make ())

(* read lines from the console, appending them to s,
 * until the user enters ;; at end of line *)
let read_console () : string =
  let rec read_lines (s : string) : string =
    let input = Util.trim (read_line ()) in
    if Util.ends_with ";;" input
    then s ^ (String.sub input 0 (String.length input - 2))
    else read_lines (s ^ input ^ "\n")
  in read_lines ""


let eval_expression (e : expr) (environment : env_stack) 
                    : env_stack =
	let ((o,oT,td),tl) =
		match environment with
			| [] -> failwith "No environment found"
			| h::t -> (h,t) in
  let (typeE, p, td2) = 
    if !Inference_types.type_inference 
    then Inference_types.getType e (oT, td)
    else (DummyType, Environment_types.empty, [ ]) in
  let (v, o) = Eval.eval (e, o) in
  if !Inference_types.type_inference
  then Printf.printf "- : %s = %s\n" 
    (Ast_types.to_string typeE) (Printing.to_string (v, o))
  else print_endline (Printing.to_string (v, o));
  (* garbage collect *)
  (* TODO: Verify *)
  Modular_Environment.garbage_collect top ((o,oT,td)::tl)
  
let rec eval_module (verbose : bool) (environment : env_stack) (d : declaration)
    : env_stack =
  match d with
  | Module(id,dl) ->
    let new_environment = Modular_Environment.create_new_environment () in
    let environment = List.fold_left (declare verbose)
      (new_environment::environment) dl in
    Modular_Environment.insert_environment id (List.hd environment);
    environment
  | _ -> assert false

and declare (verbose : bool) (environment : env_stack) (d : declaration) 
    : env_stack =
  let ((o,oT,td),tl) =
    match environment with
    | [] -> failwith "No environment found"
    | h :: t -> (h,t) in
  (* a special treatment for typedefs *)
  match d with
    | Typedef t -> 
      Typedef.check_typedef t; 
      (o, oT, t :: (List.remove_assoc (fst t) td))::tl
    | Module _ -> eval_module verbose environment d
    | _ -> (
      let (f, args, e) = match d with
	| DefType (f,_,a,_,e) |
	    DefrecType (f,_,a,_,e) -> (f,a,e)
	| DefcorecType (s, f,_,a,_,e) -> 
	  (f, [ ], FunCorec(s,f,a,e))
	| Typedef _ -> assert false
	| Module _ -> assert false
      in
      let e = match args with
	| [] -> e
	| _ -> Fun (args, e) in
      let o = match d with
	| DefType _ -> o
	| DefrecType _ | DefcorecType _ ->
          Environment.bind f Dummy o 
	| Typedef _ -> assert false
	| Module _ -> assert false in
      let (typeE, p, td2) =
	if !Inference_types.type_inference
	then Inference_types.getTypeDecl d (oT, td)
	else (Null, oT, td) in
      let (v, t) = eval (e, o) in
      let Schema (_, typ) as gen = Equations_types.generalize typeE p in
      Util.HashSet.add !top f;
      let env = Environment.bind f v t in
      if verbose then
	if !Inference_types.type_inference
	then Printf.printf "val %s : %s = %s\n" f 
	  (Ast_types.to_string typ) (Printing.to_string_bv (v, env) [ f ])
	else Printf.printf "%s = %s\n" f (Printing.to_string (v, env));
      (env, Environment_types.bind f gen p, td2)::tl)

let eval_declarations (dl : declaration list) (environment : env_stack)
    (verbose : bool) : env_stack =
  let environment = List.fold_left (declare verbose) environment dl in
  (* garbage collect *)
  (* TODO: Fix for modular environment *)
  Modular_Environment.garbage_collect top environment
    
let eval_directive (d : directive) (environment : env_stack)
    : env_stack =
	(*let (o,oT,td) =
		match environment with
			| [] -> raise "No environment found"
			| h::t -> h in*)
  match d with
    | Quit -> raise Exit
    | Chdir s ->
       if not (Sys.file_exists s) then runtime "directory does not exist" else
       if not (Sys.is_directory s) then runtime "file is not a directory" else
       Sys.chdir s; environment
    | Pwd -> print_endline (Sys.getcwd ()); environment
    | Load s ->
       if not (Sys.file_exists s) then runtime "file does not exist" else
       if Sys.is_directory s then runtime "file is a directory" else
       (match parse (read_file s) with
        | Declarations d -> eval_declarations d environment true
        | Expression e -> eval_expression e environment
        | Directive d -> runtime "directives not allowed in file")
    | Ls ->
       let cwd = Sys.getcwd () in
       let files = Sys.readdir cwd in
       Array.iter print_endline files; environment
    | Env ->
      (* TODO: Verify it is correct *)
      let f (o,oT,td) = (
	if !Inference_types.type_inference then
          let f x (Schema (_, typ)) =
	    Printf.printf "%s : %s\n" x (Ast_types.to_string typ) in
          Environment_types.iter f oT
      	else ();
      	Environment.iter 
	  (fun x v -> Printf.printf "%s = %s\n" x 
            (Printing.to_string (v, Environment.empty))) o) in
      List.iter f environment; environment
    | Scope ->
       Eval.static_scope := not !static_scope;
       Printf.printf "%s scope\n" 
	 (if !static_scope then "static" else "dynamic"); environment
    | Type_inf ->
       Inference_types.type_inference := not !Inference_types.type_inference;
       Printf.printf "type inference %s\n" 
	 (if !Inference_types.type_inference then "on" else "off"); environment

let handle = function
  | Unbound s -> Printf.printf "Unbound variable %s\n" s
  | Parse_error (_,l,c,_) -> 
    Printf.printf "Parse error at line %d char %d\n" l c
  | Lexer.LexError s -> Printf.printf "Lex Error: %s\n" s
  | Ast.Runtime s -> Printf.printf "Runtime error: %s\n" s
  | Equations_types.Type_error s ->
    Printf.printf "Type error: %s\n" s
  | Util.Fatal s -> 
    Printf.printf "Fatal error: %s\nSorry :(\n" s
  | Natives.Failwith s ->
    Printf.printf "Failure (thrown by the user): %s\n" s
  | e -> raise e

(* read eval print loop *)
let rec repl (environment : env_stack) : unit =
  try
    print_string "> ";
    let cmd = parse (read_console()) in
    let (environment : env_stack) =
      match cmd with
      | Directive d -> eval_directive d environment
      | Declarations d -> eval_declarations d environment true
      | Expression e -> eval_expression e environment in
    repl environment
  with
    | Exit -> print_endline "bye"
    | e -> handle e; repl environment

let eval_file f (environment : env_stack)
    : env_stack =
  try
    eval_directive (Load f) environment
  with e -> handle e; environment

let run () =
  let natives =
    List.fold_left
      (fun env (id, impl, typ) -> 
	Environment.bind id (Native (id, impl, typ)) env)
      Environment.empty Natives.natives in

  let nativesT = 
    List.fold_left
      (fun env (id, impl, typ) -> 
        let (t, _, _) = 
	  Inference_types.getType (Native (id, impl, typ)) (env, [ ]) in
        Environment_types.bind id (Equations_types.generalize t env) env)
      Environment_types.empty Natives.natives in
      
  List.iter (fun (x, _, _) -> Util.HashSet.add !top x) Natives.natives;

  (* type list defined as a normal sum type, by hand *)
  let typedefs = [ "list", ([ "'a" ],
			    [ "[]", None; (* replace with "[]" *)
			      "::", Some(TupleType [
				VarType "'a";
				UserType ("list", [VarType "'a"])])
			    ]) ] in

  (* load pervasives *)
  let pervasives =
    match parse (read_file "etc/pervasives.txt") with
    | Declarations dl -> 
      eval_declarations dl [(natives, nativesT, typedefs)] false
    | _ -> raise (Util.Fatal "corrupt pervasives.txt file") in
  
  (* Only prompts if there are no arguments *)
  let has_args = ref false in
  let envs = ref pervasives in
  Arg.parse [ ] 
    (fun f -> has_args := true; envs := eval_file f !envs) 
    "CoCaml help message";
  if !has_args then () else repl pervasives
    
