open Ast
open Environment

let precedence = function
	| Dummy -> failwith "unexpected dummy"
	| Symbol _ -> 9
        | Unknown _ -> 9
	| Native _ -> 9
	| Var _ -> 9
	| Int _ -> 9
	| Float _ -> 9
	| Strg _ -> 9
	| Bool _ -> 9
	| Unit -> 9
	| Not _ -> 8
	| App _ -> 7
        | Inj _ -> 7
	| Eq _ -> 5
	| Neq _ -> 5
	| Lt _ -> 5
	| Le _ -> 5
	| Gt _ -> 5
	| Ge _ -> 5
	| Plus _ -> 4
	| Minus _ -> 4
	| PlusF _ -> 4
	| MinusF _ -> 4
	| Concat _ -> 4
	| Or _ -> 4
	| Mul _ -> 3
	| Div _ -> 3
        | Mod _ -> 3
	| MulF _ -> 3
	| DivF _ -> 3
	| And _ -> 3
        | Tuple _ -> 3 (* should be 2.5 *)
	| Assg _ -> 2
	| Seq _ -> 1
        | Match _ -> 0
	| If _ -> 0
	| While _ -> 0
	| For _ -> 0
	| Fun _ -> 0
	| FunType _ -> 0
        | FunCorec _ -> 0
	| Function _ -> 0
	| LetType _ -> 0
	| LetrecType _ -> 0
        | LetcorecType _ -> 0
	

type assoc = LEFT | RIGHT | PREFIX | SUFFIX | ASSOC | NONASSOC

let vars2ids l = 
  List.map (function Var x -> x 
  | _ -> Ast.runtime "arguments have to be variables") l

let assoc = function
  | App _ -> LEFT
  | Assg _ -> RIGHT
  | Seq _ -> ASSOC
  | If _ -> PREFIX
  | Fun _ -> PREFIX
  | LetType _ | LetrecType _ | LetcorecType _ -> PREFIX
  | Not _ -> PREFIX
  | _ -> NONASSOC

let rec pattern_to_string (p : pattern) : (string * id list) = 
  match p with
  (* second argument is the list of variables bound *)
  | PVar id -> id, [ id ]
  | PInt i -> string_of_int i, [ ]
  | PFloat f -> string_of_float f, [ ]
  | PStrg s -> s, [ ]
  | PBool b -> string_of_bool b, [ ]
  | PUnit -> "()", [ ]
  | PTuple(l) -> 
    let s, l = List.split (List.map pattern_to_string l) in
    Printf.sprintf "(%s)" (String.concat "," s),
    List.flatten l
  (* Special treatment for lists *)
  | PInj("[]", None) -> "[]", [ ]
  | PInj("::", Some(PTuple [p1; p2])) ->
    let s1, l1 = pattern_to_string p1 
    and s2, l2 = pattern_to_string p2 in
    Printf.sprintf "%s :: %s" s1 s2, l1 @ l2  
  (* Other sum types *)
  | PInj(i, None) -> i,  [ ]
  | PInj(i, Some p) -> let s, l = pattern_to_string p in
		       Printf.sprintf "%s %s" i s, l
  | PUnknown p -> let s, l = (pattern_to_string p) in
		  Printf.sprintf "Unknown %s" s, l
  | PUnderscore -> "_", [ ]

let to_string_aux2 (e, env : state) (d : int option) (bv : id list)
    (is_value:bool) : string * int option =
  let rec to_string_aux (e : expr) d bv is_value : string * int option =
    (* we only go to depth d to print, to avoid looping forever *)
    (* bv represents the variables that are bounded and should 
       not be looked up in the environment *)
    (* parenthesize based on associativity and precedence *)
    (* is_value is only used to know if we should print a list 
       as a :: b :: ...  (inside a function, the list might not be a value)
       or rather as [a;b ... ] *)
    let protect_left e e1 d bv is_value: string * int option =
      let (p, p1) = (precedence e, precedence e1) in
      let s, d1 = to_string_aux e1 d bv is_value in
      (if (p1 < p && assoc e1 <> SUFFIX) || 
	  (p = p1 && (assoc e1 = RIGHT || assoc e1 = NONASSOC)) 
       then Printf.sprintf "(%s)" s else s), d1 in
    let protect_right e e1 d bv is_value: string * int option =
      let (p, p1) = (precedence e, precedence e1) in
      let s, d1 = to_string_aux e1 d bv is_value in
      (if (p1 < p && assoc e1 <> PREFIX) ||
	  (p = p1 && (assoc e1 = LEFT || assoc e1 = NONASSOC))
       then Printf.sprintf "(%s)" s else s), d1 in
    let binop_print op e e1 e2 d =
      let s1, d1 = protect_left e e1 d bv is_value in
      let s2, d2 = protect_right e e2 d1 bv is_value in
      Printf.sprintf "%s %s %s" s1 op s2, d2 in
    let arglist x = String.concat " " (vars2ids x) in
    if d = Some(0) then " ... ", d else match e with
    | Var x ->
      if List.mem x bv || d = None then x, d else 
        if Environment.bound x env 
        then (to_string_aux (Environment.lookup x env)
                (match d with None -> assert false 
		| Some(d0) -> Some(d0-1))
		bv is_value)
        else x, d
    | Symbol s -> "Symb" ^ s, d
    | Unknown e1 -> 
      let s1, d1 = protect_right e e1 None bv is_value in "Unknown" ^ s1, d1
    | Int x -> string_of_int x, d
    | Float x -> string_of_float x, d
    | Strg x -> "\"" ^ String.escaped x ^ "\"", d
    | Bool b -> string_of_bool b, d
    | Fun (x, e1) -> 
      let s, d1 = protect_left e e1 None ((vars2ids x) @ bv) false in
      Printf.sprintf "fun %s -> %s" (arglist x) s, d1
    | FunType(x, _, e1) -> 
      to_string_aux (Fun(x,e1)) None ((vars2ids x) @ bv) is_value
    | FunCorec (s, f, x, e1) ->
      let s1, _ = to_string_aux s d bv false in
      let s2, d2 = protect_left e e1 None (f ::((vars2ids x) @ bv)) false in
      Printf.sprintf "fun %s corec[%s] %s -> %s" f s1 (arglist x) s2, d2
    | Function l -> 
      let sl, dl = List.fold_right
	(fun (pi, ei) (li, di) ->
	  let si1, l = pattern_to_string pi in
	  let si2, di2 = protect_right e ei di (l @ bv) false in
	  (si1 ^ " -> " ^ si2) :: li, di2) l ([ ], d) in
      Printf.sprintf "function %s" (String.concat " | " sl), dl
    | If (e1, e2, e3) ->
      let s1, d1 = protect_left e e1 d bv false in
      let s2, d2 = protect_left e e2 d1 bv false in
      let s3, d3 = protect_left e e3 d2 bv false in
      Printf.sprintf "if %s then %s else %s" s1 s2 s3, d3
    | While (e1, e2) ->
      let s1, d1 = protect_left e e1 d bv false in
      let s2, d2 = protect_left e e2 d1 bv false in
      Printf.sprintf "while %s do %s done" s1 s2, d2
    | For (x, e1, e2, e3) ->
      let s1, d1 = protect_left e e1 d bv false in
      let s2, d2 = protect_left e e2 d1 (x :: bv) false in
      let s3, d3 = protect_left e e3 d2 (x :: bv) false in
      Printf.sprintf "for %s = %s to %s do %s done" x s1 s2 s3, d3
    | LetType (x, _, [], _, e1, e2) ->
      let s1, d1 = protect_left e e1 d bv false in
      let s2, d2 = protect_left e e2 d1 (x :: bv) false in
        Printf.sprintf "let %s = %s in %s" x s1 s2, d2
    | LetType (x, _, y, _, e1, e2) ->
      let s1, d1 = protect_left e e1 d bv false in
      let s2, d2 = protect_left e e2 d1 (x :: ((vars2ids y) @ bv)) false in
      Printf.sprintf "let %s %s = %s in %s" x (arglist y) s1 s2, d2
    | LetrecType (x, _, [], _, e1, e2) ->
      let s1, d1 = protect_left e e1 d bv false in
      let s2, d2 = protect_left e e2 d1 (x :: bv) false in
      Printf.sprintf "let rec %s = %s in %s" x s1 s2, d2
    | LetrecType (x, _, y, _, e1, e2) ->
      let s1, d1 = protect_left e e1 d bv false in
      let s2, d2 = protect_left e e2 d1 (x :: ((vars2ids y) @ bv)) false in
      Printf.sprintf "let rec %s %s = %s in %s" x (arglist y) s1 s2, d2
    | LetcorecType (s, x, _, [], _, e1, e2) ->
      let s1, d1 = to_string_aux s d bv false in
      let s2, d2 = protect_left e e1 d1 bv false in
      let s3, d3 = protect_left e e2 d2 (x :: bv) false in
      Printf.sprintf "let corec[%s] %s = %s in %s" s1 x s2 s3, d3
    | LetcorecType (s, x, _, y, _, e1, e2) ->
      let s1, d1 = to_string_aux s d bv false in
      let s2, d2 = protect_left e e1 d1 bv false in
      let s3, d3 = protect_left e e2 d2 (x :: ((vars2ids y) @ bv)) false in
      Printf.sprintf "let corec[%s] %s %s = %s in %s" s1 x (arglist y) s2 s3, d3
    | App (e1, e2) ->
      let s1, d1 = protect_left e e1 d bv false in
      let s2, d2 = protect_right e e2 d1 bv false in
      Printf.sprintf "%s %s" s1 s2, d2
    | Assg (x, e1) -> let s1, d1 = protect_right e e1 d bv false in
		      Printf.sprintf "%s := %s" x s1, d1
    | Seq (e1, e2) -> 
      let s1, d1 = protect_left e e1 d bv false in
      let s2, d2 = protect_left e e2 d1 bv false in
      Printf.sprintf "%s; %s" s1 s2, d2
    | Match (e1, l) ->
      let s1, d1 = protect_left e e1 d bv false in
      let sl, dl = List.fold_right
	(fun (pi, ei) (li, di) ->
	  let si1, l = pattern_to_string pi in
	  let si2, di2 = protect_right e ei di (l @ bv) false in
	  (si1 ^ " -> " ^ si2) :: li, di2) l ([ ], d1) in
      Printf.sprintf "match %s with %s" s1 (String.concat " | " sl), dl

    (* Comparison operators *)
    | Eq (e1, e2) -> binop_print "=" e e1 e2 d
    | Neq (e1, e2) -> binop_print "!=" e e1 e2 d
    | Le (e1, e2) -> binop_print "<=" e e1 e2 d
    | Lt (e1, e2) -> binop_print "<" e e1 e2 d
    | Gt (e1, e2) -> binop_print ">" e e1 e2 d
    | Ge (e1, e2) -> binop_print ">=" e e1 e2 d

    (* Arithmetic operators on integers*)
    | Plus (e1, e2) -> binop_print "+" e e1 e2 d
    | Minus (e1, e2) -> binop_print "-" e e1 e2 d 
    | Mul (e1, e2) -> binop_print "*" e e1 e2 d
    | Div (e1, e2) -> binop_print "/" e e1 e2 d
    | Mod (e1, e2) -> binop_print "mod" e e1 e2 d

    (* Arithmetic operators on floats *)
    | PlusF (e1, e2) -> binop_print "+." e e1 e2 d
    | MinusF (e1, e2) -> binop_print "-." e e1 e2 d
    | MulF (e1, e2) -> binop_print "*." e e1 e2 d
    | DivF (e1, e2) -> binop_print "/." e e1 e2 d

    (* String operators *)
    | Concat (e1, e2) -> binop_print "^" e e1 e2 d

    (* Boolean operators *)
    | Not e1 -> let s1, d1 = protect_left e e1 d bv false in
		Printf.sprintf "not %s" s1, d1
    | And (e1,e2) -> binop_print "&&" e e1 e2 d
    | Or (e1,e2) ->  binop_print "||" e e1 e2 d
    | Tuple l -> (* it is important to go from left to right *)
      let s3, d3 = List.fold_left 
	(fun (s1, d1) e1 -> let s2, d2 = protect_right e e1 d1 bv is_value in
			    s2 :: s1, d2) ([ ], d) l in
      String.concat ", " (List.rev s3), d3
	  
    (* Special treatment for lists *)
    | Inj("[]", None) -> "[]", d
    | Inj("::", _) ->
      let rec string_of_list di = function
	| Inj("[]", None) -> "", di
	| Inj("::", Some(Tuple [e1; e2])) -> 
	  let s1, d1 = protect_left e e1 di bv is_value in
	  let s2, d2 = string_of_list d1 e2 in
	  (if s2 = "" then (* s2 represented the empty list *)
	      if is_value then s1
	      else s1 ^ " :: " ^ "[]"
	   else
	      if is_value then s1 ^ "; " ^ s2
	      else s1 ^ " :: " ^ s2), d2
	| Var x -> (* not so nice, but avoids repeating code *)
	  let res, d1 = to_string_aux (Var x) di bv is_value in
	  (if res = x then x (* x wasn't resolved *)
	  (* TODO: distinguish if it was a value or not *)
	   else String.sub res 1 (String.length res-2)), d1
	| e -> to_string_aux e di bv is_value in
      let sf, df = string_of_list d e in
      (if is_value then Printf.sprintf "[%s]" sf
       else Printf.sprintf "%s" sf), df

    | Inj(s, None) -> s, d
    | Inj(s, Some(e2)) -> let s1, d1 = protect_right e e2 d bv is_value in
			  Printf.sprintf "%s %s" s s1, d1
    | Unit -> "()", d
    | Dummy -> "<dummy>", d
    | Native _ -> "<native>", d
in to_string_aux e d bv is_value

let to_string_bv s (bv : string list) = match (fst s) with
 | Fun _ | FunType _ | FunCorec _ -> fst(to_string_aux2 s None bv true)
  (* to avoid replacing themselves in recursive functions *)
 | _ -> fst(to_string_aux2 s (Some 20) [ ] true) 
(* TODO: was 20: combinatorial explosion with a binary tree. Fix it! *)
(* None means do not replace anything *)
let to_string s = to_string_bv s [ ]
