open Ast
open Parser
open Util
open Environment
open Equality

exception Unmatched_pattern
  
type equation = expr * expr
(* first element represents an id:
   - either Symbol(id) in my version 
   - or something of type var in Dexter's version *)

(* global flag for static or dynamic scope *)
let static_scope = ref true

(* global internal symbol generator *)
let fresh : unit -> id =
  let gs : Gensym.t = Gensym.make () in
  fun () -> Gensym.next gs

(* read from a file *)
let read_file (filename : string) : string =
  let rec get_contents contents file =
    let input_line_option file =
      try Some ((input_line file) ^ "\n") with End_of_file -> None in
    match (input_line_option file) with
	Some x -> get_contents (contents ^ x) file
      | None -> contents in
  let file = try open_in filename with Not_found -> runtime "File not found" in
  let result = get_contents "" file in close_in file; result

(* get a fresh variable not occurring in an expression *)

(* Substitute var y for free occurrences of var x in e *)
let subst (y : id) (x : id) (e : expr) : expr =
  let match_args = function (Var id) -> id = x | _ -> false in
  let rec sub (e : expr) : expr =
    match e with
      | Var id -> if id = x then Var y else Var id
      | (Int _ | Float _ | Strg _ | Bool _ | Dummy | Native _ | Unit) -> e
      | Fun (args, c) -> 
	if List.exists match_args args then e else Fun (args, sub c)
      | FunType (args,_,e) -> sub (Fun(args,e))
      | FunCorec (s, f, args, c) -> 
        if List.exists match_args args then e else FunCorec (s, f, args, sub c)
      | LetType (id, t1, args, t2,e1,e2) -> 
	let e1 = if (List.exists match_args args) then e1 else sub e1 in
	let e2 = if id = x then e2 else sub e2 in
	LetType (id, t1, args, t2, e1, e2)
      | LetrecType (id, t1, args, t2, e1, e2) ->
	let e1 = 
	  if (id = x || List.exists match_args args) then e1 else sub e1 in
	let e2 = if id = x then e2 else sub e2 in
	LetrecType (id, t1, args, t2, e1, e2)
      | LetcorecType (s, id, t1, args, t2, e1, e2) ->
	let e1 = 
	  if (id = x || List.exists match_args args) then e1 else sub e1 in
	let e2 = if id = x then e2 else sub e2 in
        LetcorecType (s, id, t1, args, t2, e1, e2)
      | Function l -> Function(
	List.map 
	  (fun (pi, ei) -> 
	    if List.mem x (Ast.bound_vars pi) then (pi, ei)
	    else (pi, sub ei))
	  l)
      | App (e1, e2) -> App (sub e1, sub e2)
      | If (e1, e2, e3) -> If (sub e1, sub e2, sub e3)
      | While (e1, e2) -> While (sub e1, sub e2)
      | For (id, e1, e2, e3) ->
        For ((if id = x then y else id), sub e1, sub e2, sub e3)
      | Match(e1, l) ->
	Match(sub e1,
	      List.map 
		(fun (pi, ei) -> 
		  if List.mem x (Ast.bound_vars pi) then (pi, ei)
		  else (pi, sub ei))
		l)
      | Assg (id, e2) -> Assg ((if id = x then y else id), sub e2)
      | Seq (e1, e2) -> Seq (sub e1, sub e2) 
      | Tuple lst -> Tuple (List.map (fun e1 -> sub e1) lst)
      | Inj(s, None) -> e
      | Inj(s, Some(e1)) -> Inj(s, Some(sub e1))
		(* Comparison operators *)
      | Eq (e1, e2) -> Eq (sub e1, sub e2)
      | Neq (e1, e2) -> Neq (sub e1, sub e2)
      | Lt (e1, e2) -> Lt (sub e1, sub e2)
      | Le (e1, e2) -> Le (sub e1, sub e2)
      | Gt (e1, e2) -> Gt (sub e1, sub e2)
      | Ge (e1, e2) -> Ge (sub e1, sub e2)
		(* Arithmetic operators on integers *)
      | Plus (e1, e2) -> Plus (sub e1, sub e2)
      | Minus (e1, e2) -> Minus (sub e1, sub e2)
      | Mul (e1, e2) -> Mul (sub e1, sub e2)
      | Div (e1, e2) -> Div (sub e1, sub e2)
      | Mod (e1, e2) -> Mod (sub e1, sub e2)
		(* Arithmetic operators on floats *)
      | PlusF (e1, e2) -> PlusF (sub e1, sub e2)
      | MinusF (e1, e2) -> MinusF (sub e1, sub e2)
      | MulF (e1, e2) -> MulF (sub e1, sub e2)
      | DivF (e1, e2) -> DivF (sub e1, sub e2)
		(* String operators *)
      | Concat (e1, e2) -> Concat (sub e1, sub e2)
		(* Boolean operators *)
      | Not e1 -> Not(sub e1)
      | And (e1,e2) -> And(sub e1, sub e2)
      | Or (e1,e2 )-> Or(sub e1, sub e2)
      | Unknown i -> Unknown i 
      | Symbol i -> Symbol i 
  in sub e

let rec matches (p:pattern) (e:expr) (o:env) : (id * expr) list option =
  (* tests if pattern p matches expression e *)
  let e0 = (match e with Var x -> Environment.lookup x o | _ -> e) in
  (* for the case where the structure is hidden in the capsule *)
  match (p, e0) with
    | PVar x, e -> Some [ x, e ]
    | PInt i1, Int i2 when i1 = i2 -> Some [ ]
    | PFloat f1, Float f2 when f1 = f2 -> Some [ ]
    | PStrg s1, Strg s2 when s1 = s2 -> Some [ ]
    | PBool b1, Bool b2 when b1 = b2 -> Some [ ]
    | PUnit, Unit -> Some [ ]

    | PInj(i1, None), Inj(i2, None) when i1 = i2 -> Some [ ]
    | PInj(i1, Some p1), Inj(i2, Some e2) when i1 = i2 -> matches p1 e2 o
    | PUnknown p1, Unknown e2 -> matches p1 e2 o

    | PTuple pl, Tuple el -> (* because of typing, they are the same size *)
      List.fold_left2 
        (fun bo pi ei -> match bo with None -> None
          | Some(b) -> (match matches pi ei o with 
              None -> None
              | Some(bi) -> Some(b @ bi)))
        (Some [ ]) pl el
    | PUnderscore, _ -> Some [ ]
    | PInt _, _ | PFloat _, _ | PStrg _, _ | PBool _, _ | PUnit, _ 
    | PInj _, _ | PTuple _, _ | PUnknown _, _ -> None

let rec eval_patterns 
    (e:expr) (pl:(pattern * expr) list) (o : env) fcorec unks repl
    : state * ((expr * expr) list) * bool = 
  match pl with
    | [ ] -> runtime ("Match failure on " ^ (Printing.to_string (e,o)))
    | (pi, ei) :: tl -> (
      match matches pi e o with 
	  None -> eval_patterns e tl o fcorec unks true
	| Some b -> if b = [ ] then evalp (ei, o) fcorec unks true else
            let (vars, args) = List.split b in
            let e2 = List.fold_left
              (fun fu arg -> App(fu, arg)) 
              (Fun(List.map (fun x -> Var x) vars, ei)) args
            in evalp (e2, o) fcorec unks true )

and evalp ((e, o) : state) 
    (fcorec : (id * ((state * expr) list) * 
		 ((state -> state) * ((unit * env) -> state))) option) 
    (* the second element is the set of seen elements: association with their 
       variable. Changed to an expr to accomodate Dexter's view of things.
       Those expressions will be Unknown(sth) if using my view,
      or of time var if using Dexter's view *)
    (unks : (id, expr) Hashtbl.t option)
    (repl:bool)
    (* unks an interpretation for the unknowns *)
    : state * ((expr * expr) list) * bool =
   (* the list l is the list of unknowns appearing inside 
      (except if already seen) *)
   (* boolean b indicates if it is completely evaluated (true) or not *)
   (* DO NOT suppress b: l could be empty with b false, see 10 lines below *)
   (* partial evaluation; fcorec is the name of the corecursive function
      id * expr list are the corecursive calls, meaning the result of 
      fcorec(expr) is represented by Unknown(id) *)
  match fcorec, e with
    | Some(f, seen, (uunk, ufresh)), App(Var id, e1) when f = id ->
      let (e2, o2), l, b = evalp (e1, o) None unks true in 
      (* None because nesting not allowed *)
      if b then 
        try let name = Equality.assq (e2, o2) seen  in
            (uunk (name, o2)), [ ], false
        with Not_found -> 
          let freshid, o3 = ufresh((), o2) in
	  (uunk (freshid, o3)), [ freshid, e2 ], false
      else assert false
    | Some(f, _, _), Var id when f = id -> 
      runtime "A corecursive call has to be applied right away"
    (* TODO: what if f is rebound later in its own body? *)
    | _ -> match e with
	| Var id -> let r = Environment.lookup id o in
                    (* to handle corecursive definitions *)
                    if r = Dummy || (not(repl))
                    then (Var id, o), [ ], true
                    else (r, o), [ ], true
	| Int i -> (e, o), [ ], true
	| Float f -> (e, o), [ ], true
	| Strg s -> (e, o), [ ], true
	| Bool b -> (e, o), [ ], true
	| Fun (_, _) -> (e, o), [ ], true
                (* TODO: check that fcorec does not appear in body *)
	| FunType (args, _, c) -> (Fun (args,c), o), [ ], true
        | FunCorec _ -> (e, o), [ ], true
	| Function l -> (e, o), [ ], true
	| LetType (id, _, args, _, e1,e2) ->
	  if args = [] 
	  then evalp (App (Fun ([Var id], e2), e1), o) fcorec unks true
	  else 
	    evalp (App (Fun ([Var id], e2), Fun (args, e1)), o) 
	      fcorec unks true
	| LetrecType (id, tid, args, targs, e1, e2) ->
	  if args = [] then
	    (* Let id = <dummy> in (id := e1); e2 *)
	    evalp
              (LetType (id, tid, [], [], Dummy, Seq (Assg (id, e1), e2)), o) 
	      fcorec unks true
	  else
	    (* Let id = <dummy> in (id := fun args -> e1); e2 *)
	    evalp (LetType (id, tid, [], [], Dummy, 
			    Seq (Assg (id, Fun (args, e1)), e2)), o)
	      fcorec unks true
        | LetcorecType (s, id, _, [arg], _, e1, e2) -> (* TODO *)
          evalp (App (Fun ([Var id], e2), FunCorec (s, id, [arg], e1)), o)
	    fcorec unks true
        | LetcorecType _ -> 
	  runtime "Corecursive functions can only have one argument"
	| App (e1, e2) ->
	  let (d, p1), l1, b1 = evalp (e1, o) fcorec unks true in
	  let (u, p2), l2, b2 = evalp (e2, p1) fcorec unks true in
          let l = l1 @ l2 and b = b1 && b2 in
          let fcorec2 = if l = [ ] then fcorec else None in
          if not(b) then (App(d, u), p2), l, b else
	    (match d with
  	      | Fun (args, c) -> (match args with
  	          | Var x :: t -> 
		    let res1, res2, res3 = if !static_scope
                    then let y = fresh () in
                         evalp (subst y x (if t = [] then c else Fun (t, c)),
                                Environment.bind y u p2) fcorec2 unks true
                    else (* dynamic scope *)
  	              evalp ((if t = [] then c else Fun (t, c)),
                             Environment.bind x u p2) fcorec unks true
				  in res1, l @ res2, res3
                  | Unit :: t -> if u = Unit 
                    then let res1, res2, res3 =
                           evalp ((if t = [] then c else Fun (t, c)), p2) 
			     fcorec2 unks true
                         in res1, l @ res2, res3
                    else runtime "argument mismatch"
                  | _ -> runtime "not a funtion")
              | FunCorec (solver, fname, [Var x], body) ->
                if not(fcorec = None) 
                then 
                  (match fcorec with 
                      None -> assert false 
                    | Some(f, _, _) -> if f = fname then
                        runtime "Nested corecursive functions not allowed"
                      else
                        eval_corec solver fname (Fun([Var x], body)) 
                          (u, p2) unks true, [ ], true)
                else if !static_scope
                then 
                  eval_corec solver fname (Fun([Var x], body)) 
		    (u, p2) unks true, [ ], true
                else runtime 
		  "Corecursive functions not implemented for dynamic scope"
              | FunCorec _ -> 
		runtime "Corecursive function must have exactly one argument"
	      | Function l ->
		eval_patterns u l p2 fcorec unks true
              | Native (_, native, _) -> (native u, p2), [ ], true
  	      | _ -> print_string(Printing.to_string (d,o)); runtime "can only apply a function")
	| If (e, e1, e2) ->
	  let (d, t), l, b = evalp (e, o) fcorec unks true in
          let fcorec2 = if l = [ ] then fcorec else None in
          if (not b) then (If(d, e1, e2), t), l, b else
	    ( match d with
	      | Bool b0 -> let res1, res2, res3 =
                             evalp ((if b0 then e1 else e2), t) 
			       fcorec2 unks true
                           in res1, l @ res2, res3
	      | _ -> runtime "'if' requires a boolean test")
	| While (e, e1) ->
	  let (d,t), l, b = evalp (e,o) fcorec unks true in
          let fcorec2 = if l = [ ] then fcorec else None in
          if not b then (While(d, e1), t), l, b else
	    (match d with
	      | Bool b0 -> if b then
                  let res1, res2, res3 =
                    evalp (Seq (e1, While (e, e1)), t) fcorec2 unks true
                  in res1, l @ res2, res3
		else (Unit, t), l, b
	      | _ -> runtime "'while' requires a boolean test")
	| For (x, e1, e2, e3) -> 
	  let (d,t), l1, b1 = evalp (e1,o) fcorec unks true
          and (f,u), l2, b2 = evalp (e2,o) fcorec unks true in
          let l = l1 @ l2 in
	  let b = b1 && b2 in
          let fcorec2 = if l = [ ] then fcorec else None in
          if not b then (For(x, d, f, e3), t), l, b else
	    (match (d,f) with
  	      | ( Int n, Int p) -> if n>p then (Unit,o), l, true
		else let res1, res2, res3 = 
		       evalp (Seq (e3, For (x,  (Plus (d, (Int 1))),f, e3)),
			      Environment.bind x d o) fcorec2 unks true in
		     res1, l @ res2, res3
  	      | _ -> runtime 
		" the second and third arguments of 'for' must evaluate to integers ")
	| Assg (x, e) ->
          if not (Environment.bound x o) then raise (Unbound x) else
            let (v, t), l, b = evalp (e, o) fcorec unks true in
            if not b then (Assg (x, v), t), l, b else
              (Unit, Environment.bind x v t), l, b
	| Seq (e1, e2) ->
	  let (t1, o1), l1, b1 = evalp (e1, o) fcorec unks true in
	  let (t2, o2), l2, b2 = evalp (e2, o1)
	    (if l1 = [ ] then fcorec else None) unks true
          in let b = b1 && b2
             in if b then (t2, o2), l1 @ l2, b
               else ( Seq (t1, t2), o2 ), l1 @ l2, b
        | Match(e1, l) ->
          let (r1, o1), l1, b1 = evalp (e1, o) fcorec unks true in
          if b1 then eval_patterns r1 l o1 fcorec unks true
          else (Match(r1, l), o1), l1, b1

        | Tuple l -> (* neither fold_left nor fold_right does the job *) 
          let rec iterate oi = function
            | [ ] -> ([ ], oi), [ ], true
            | h :: t ->
	      let (ej, oj), lj, bj = evalp (h, oi) fcorec unks repl in 
              let (l2, o2), lk, bk = iterate oj t in
              (ej :: l2, o2), lj @ lk, bj && bk in
          let (l2, o2), lcorec, bf = iterate o l in
          (Tuple l2, o2), lcorec, bf
        | Inj(s, None) -> (e, o), [ ], true
        | Inj(s, Some(e1)) ->
	  let (e2, o2), l, b = evalp (e1, o) fcorec unks repl in
          (Inj(s, Some(e2)), o2), l, b
                (* TODO: DO NOT replace under a constructor 
                   replace 'repl' by 'false' 
                   but breaks the separation solver*)
		(* Comparison operators *)
	| Eq (e1, e2) ->
	  let (p1,t), l1, b1 = evalp (e1,o) fcorec unks true in
	  let (p2,r), l2, b2 = evalp (e2,t) fcorec unks true in
          let b = b1 && b2 in
          let l = l1 @ l2 in if (not b) then 
              (Eq(p1, p2), r), l, b else (
                (Bool(Equality.equal (p1, r) (p2, r)), r), l, b)
	| Neq (e1, e2) ->
	  let (d,t), l, b = evalp (Eq (e1, e2), o) fcorec unks true in
	  (match d with
	    | Bool b -> (Bool (not b), t), l, b
            | Eq(r1, r2) -> (Neq(r1, r2), t), l, b
	    | _ ->  raise (Fatal "Eq did not return Bool"))
	| Lt (e1, e2) -> eval_comp (<) (<) (<) e1 e2 o 
	  (fun (x,y) -> Lt (x,y)) fcorec unks true
	| Le (e1, e2) -> eval_comp (<=) (<=) (<=) e1 e2 o 
	  (fun (x,y) -> Le (x,y)) fcorec unks true
	| Gt (e1, e2) -> eval_comp (>) (>) (>) e1 e2 o
	  (fun (x,y) -> Gt (x,y)) fcorec unks true
	| Ge (e1, e2) -> eval_comp (>=) (>=) (>=) e1 e2 o 
	  (fun (x,y) -> Ge (x,y)) fcorec unks true

	(* Arithmetic operators on integers *)
	| Plus (e1, e2) -> eval_arith (+) e1 e2 o 
	  (fun (x,y) -> Plus (x,y)) fcorec unks true
	| Minus (e1, e2) -> eval_arith (-) e1 e2 o
	  (fun (x,y) -> Minus (x,y)) fcorec unks true
	| Mul (e1, e2) -> eval_arith ( * ) e1 e2 o
	  (fun (x,y) -> Mul (x,y)) fcorec unks true
	| Div (e1, e2) -> eval_arith (/) e1 e2 o
	  (fun (x,y) -> Div (x,y)) fcorec unks true
	| Mod (e1, e2) -> eval_arith (mod) e1 e2 o
	  (fun (x,y) -> Mod (x,y)) fcorec unks true

	(* Arithmetic operators on floats *)
	| PlusF (e1, e2) -> eval_arithF (+.) e1 e2 o 
	  (fun (x,y) -> PlusF (x,y)) fcorec unks true
	| MinusF (e1, e2) -> eval_arithF (-.) e1 e2 o
	  (fun (x,y) -> MinusF (x,y)) fcorec unks true
	| MulF (e1, e2) -> eval_arithF ( *. ) e1 e2 o
	  (fun (x,y) -> MulF (x,y)) fcorec unks true
	| DivF (e1, e2) -> eval_arithF (/.) e1 e2 o 
	  (fun (x,y) -> DivF (x,y)) fcorec unks true

		(* String operators *)
	| Concat (e1, e2) -> eval_strg (^) e1 e2 o
	  (fun (x,y) -> Concat (x,y)) fcorec unks true

		(* Boolean operators *)
	| Not e1 -> (match evalp (e1, o) fcorec unks true with
	    | (Bool b, t), l, b1 -> (Bool (not b), t), l, b1
            | (r, t), l, b1 when not(b1) -> (Not r, t), l, b1
	    | _ -> runtime "can only use negation on booleans"
	)
	| And (e1, e2) -> (match evalp (e1, o) fcorec unks true with
	    | (Bool b, t), l1, b1 as d -> if b then
      		(match evalp (e2, t) fcorec unks true with
      		  | (Bool b, o2), l2, b2 -> (Bool b, o2), l1 @ l2, b1 && b2
                  | (r2, o2), l2, b2 when not(b2) ->
                    (And(Bool b, r2), o2), l1 @ l2, b1 && b2
      		  | _ -> runtime "can only use && on booleans 1")
              else d
            | (r1, o1), l1, b1 when not(b1) -> (* giving up laziness *)
              let (r2, o2), l2, b2 = evalp(e2, o1) fcorec unks true in
              (And(r1, r2), o2), l1 @ l2, b1 && b2
            | _ -> runtime "can only use && on booleans 2")
	| Or (e1, e2) -> (match evalp (e1, o) fcorec unks true with
	    | (Bool b, t), l1, b1 as d -> if b then d else
      		(match evalp (e2, t) fcorec unks true with
      		  | (Bool b, o), l2, b2 -> (Bool b, o), l1 @ l2, b1 && b2
                  | (r2, o2), l2, b2 when not(b2) ->
                    (Or(Bool b, r2), o2), l1 @ l2, b1 && b2
      		  | (r2, o2), _, _ -> 
                    print_string(Printing.to_string (r2, o2));
                    runtime "can only use || on booleans 1")
            | (r1, o1), l1, b1 when not(b1) -> (* giving up laziness *)
              let (r2, o2), l2, b2 = evalp(e2, o1) fcorec unks true in
              (Or(r1, r2), o2), l1 @ l2, b1 && b2
            | (r1, o1), _, _ -> 
              print_string(Printing.to_string (r1, o1));
              runtime "can only use || on booleans 2")

	| Unit -> (e, o), [ ], true
	| Native _ -> (e, o), [ ], true

		(* Dummy is used internally *)
	| Dummy -> (e, o), [ ], true
        | Unknown (Symbol i) -> (match unks with
	    | None -> (Unknown (Symbol i), o), [ ], true (* why not false? *)
            | Some t -> try (Hashtbl.find t i, o), [ ], true
              with Not_found -> (Unknown (Symbol i), o), [ ], true (* why not false? *))
	| Symbol i -> (e, o), [ ], true (* appears on left side of equations *)
	| Unknown _ -> assert false 

and eval_comp 
    (op_i:int->int->bool) (op_b:bool->bool->bool) (op_s:string->string->bool)
    (e1:expr) (e2:expr) (o:env)
    constr fcorec unks repl : state * ((expr * expr) list) * bool =
  let (p1,t), l1, b1 = evalp (e1,o) fcorec unks repl in
  let (p2,r), l2, b2 = evalp (e2,t) fcorec unks repl in
  let l = l1 @ l2 in
  let b = b1 && b2 in
  if not(b) then (constr(p1,p2), r), l, b else
    (match (p1,p2) with
      | (Int i, Int j) -> (Bool (op_i i j), r), l, b
      | (Bool b, Bool c) -> (Bool (op_b b c), r), l, b
      | (Strg s, Strg t) -> (Bool (op_s s t), r), l, b
      | (Symbol s, Symbol t) -> (Bool (op_s s t), r), l, b
	(* Implicitly imposes unknown = string *)
      | _ -> runtime "can only compare ints to ints and bools to bools")
      
and eval_arith 
    (op:int->int->int) (e1:expr) (e2:expr) (o:env) constr fcorec unks repl
    : state * ((expr * expr) list) * bool =
  let (p1,t), l1, b1 = evalp (e1,o) fcorec unks repl in
  let (p2,r), l2, b2 = evalp (e2,t) fcorec unks repl in
  let l = l1 @ l2 in
  let b = b1 && b2 in
  if not(b) then (constr(p1,p2), r), l, b else
    (match (p1,p2) with
      | (Int i, Int j) -> (Int (op i j), r), l, b
      | _ -> (print_string(Printing.to_string(p1,o));
	      print_string(Printing.to_string(p2,o));
	      runtime "can only do arithmetic on ints or floats"))

and eval_arithF 
    (op:float->float->float) (e1:expr) (e2:expr) (o:env) constr fcorec unks repl
    : state * ((expr * expr) list) * bool =
  let (p1,t), l1, b1 = evalp (e1,o) fcorec unks repl in
  let (p2,r), l2, b2 = evalp (e2,t) fcorec unks repl in
  let l = l1 @ l2 in
  let b = b1 && b2 in
  if not(b) then (constr(p1,p2), r), l, b else (
    match (p1,p2) with
      | (Float i, Float j) -> (Float (op i j), r), l, b
      | _ -> runtime "can only do arithmetic on ints or floats")

and eval_strg 
    (op:string->string->string) (e1:expr) (e2:expr) (o : env)
    constr fcorec unks repl
    : state * ((expr * expr) list) * bool =
  let (p1,t), l1, b1 = evalp (e1,o) fcorec unks repl in
  let (p2,r), l2, b2 = evalp (e2,t) fcorec unks repl in
  let l = l1 @ l2 in
  let b = b1 && b2 in
  if not(b) then (constr(p1,p2), r), l, b else
    (match (p1,p2) with
      | (Strg i, Strg j) -> (Strg (op i j), r), l, b
      | _ -> runtime "can only do string operations on strings")

and eval_corec (solver:expr) (fname:id) (func:expr) 
    (arg, env:state) unks repl : state =
  let name, env, eqs = equations (arg, env) fname func solver in
  (* Printing equations for debugging purposes *)
  (* print_string
    ("Equations: find " ^ (Printing.to_string (name, env)) ^ " such that\n" ^
	(String.concat "\n"
	   (List.map (fun (i, rhs) -> (Printing.to_string (i, env)) ^ " = " ^ 
	     (Printing.to_string (rhs, Environment.empty))) eqs)) ^
	"\n"); *)
  let id_of_expr = function Symbol id -> id | _ -> assert false in
  let ids_of_eqs = function eqs ->
    List.map (fun (r, l) -> (id_of_expr r, l)) eqs in
  match solver with
    | Var("constructor") -> 
      Constructor.solve (id_of_expr name) env (ids_of_eqs eqs)
   (* passing eval through is awkward *)
    | App(Var "iterator", bot) ->
      let (bot1, env1), _, _ = evalp (bot, env) None unks repl in
      Iterator.solve 
	(fun x unks1 -> evalp x None (Some(unks1)) true) 
	bot1 (id_of_expr name) env1 (ids_of_eqs eqs)
    | Var("gaussian") -> Gaussian.solve (id_of_expr name) env (ids_of_eqs eqs)
    | App(Var "appears", e) -> 
      let (e1, env1), _, _ = evalp (e, env) None unks repl in
      Appears.solve e1 name env1 eqs
    | Var("separate") -> Separate.solve (id_of_expr name) env (ids_of_eqs eqs)
      (* Solver defined by the user *)
    | _ -> (* Converting the equations to a CoCaml format *)
      let cocaml_eqs = List.fold_right (fun (v, rhs) acc ->
        Inj("::", Some(Tuple [Tuple [v; rhs]; acc]))
      ) eqs (Inj("[]", None)) in
      let usolver = match solver with
	| Tuple [_; _; s] -> s
	| _ -> solver in
      let (e1, env1), _, _ = 
	evalp (App(App(usolver, name), cocaml_eqs), env) None unks repl in
      (e1, env1)

and equations (arg, env: state) (fname : id) (func : expr) (solver : expr)
    (* solver arg only for Dexter's method *)
    : expr * env * (equation list) = 
  let (uunk, ufresh) = match solver with
    | Tuple [uunk; ufresh; _] -> 
      (fun (var, e) -> let (r, e1), _, _ = 
		    evalp (App(uunk, var), e) None None true in (r, e1)),
      (fun ((), e) -> let (r, e1), _, _ =
		   evalp (App(ufresh, Unit), e) None None true in (r, e1))
    | _ -> (fun (x, e) -> Unknown x, e), (fun ((), e) -> Symbol(fresh()), e) in
  let rec equations_aux (name:expr) (arg, env: state) (eqs: equation list)
      (seen: (state * expr) list)
      (fname : id) (func : expr) unks
      : expr * env * (equation list) * ((state * expr) list) =
    let seen0 = ((arg, env), name) :: seen in
    let (eq_rhs, env0), argl, _ = 
      evalp (App(func, arg), env) 
	(Some (fname, seen0, (uunk, ufresh))) unks true in
    let eqs0 = (name, eq_rhs) :: eqs in
    let (envf, eqsf, seenf) = List.fold_left 
      (fun (envi, eqsi, seeni) (idi, argi) ->
	let (_, envj, eqsj, seenj) =
          equations_aux idi (argi, envi) eqsi seeni fname func unks
	in (envj, eqsj, seenj))
      (env0, eqs0, seen0) argl in
    (name, envf, eqsf, seenf) in
  let namef, envf, eqsf, _ =
    let freshid, env1 = ufresh ((), env) in
    equations_aux (freshid) (arg, env1) [] [] fname func (Some(Hashtbl.create 1))
  in
  namef, envf, eqsf

and eval ((e, o) : state) : state =
  match evalp (e,o) None (Some(Hashtbl.create 1)) true with
    | s, [ ], true -> s
    | _ -> assert false
