open Ast
open Ast_types
open Map
open Util

exception Unbound of id

(* Functional map with vars as keys *)
module VarMap = Map.Make (struct
  type t = id
  let compare = String.compare
end)

module type FV = sig
  type data
  val free_vars : id Util.HashSet.t -> data -> unit
end

module type ENVIRONMENT = functor (A : FV) ->
  sig
    type data = A.data
    type t
    val empty : t
    val bind : id -> data -> t -> t
    val bound : id -> t -> bool
    val iter : (id -> data -> unit) -> t -> unit
    val fold : (id -> data -> 'b -> 'b) -> t -> 'b -> 'b
    val map : (data -> data) -> t -> t
    val remove : id -> t -> t
    val lookup : id -> t -> data
    val gc : id HashSet.t -> t -> t
  end

module Make_Environment : ENVIRONMENT = functor (A : FV) ->
  struct
    type data = A.data
  	type t = data VarMap.t
  	let empty = VarMap.empty
  	let bind = VarMap.add
  	let bound = VarMap.mem
  	let iter = VarMap.iter
    let fold = VarMap.fold

     let map = VarMap.map
  	let remove = VarMap.remove
  	let lookup x o = try VarMap.find x o 
	  with Not_found -> print_string "REMenv"; raise (Unbound x)
      
    (* garbage collection *)
    (* starting at top-level, do bfs on the given environment *)    
  	let gc (top : id HashSet.t) (o : t) : t =
      (* put all top-level items on the worklist *)
      let worklist = HashSet.make () in
      HashSet.add_all worklist top;
      (* accumulate accessible vars in accum *)
      let accessible = HashSet.make() in
      let next = HashSet.make() in
      while not (HashSet.empty worklist) do
        HashSet.add_all accessible worklist;
        (* get free vars of bindings in worklist *)
        HashSet.clear next;
        let add_to_next = A.free_vars next in
        HashSet.iter (fun x -> add_to_next (lookup x o)) worklist;
        (* filter out ones already seen *)
        HashSet.remove_all next accessible;
        HashSet.clear worklist;
        HashSet.add_all worklist next;
      done;
      (* accessible now contains all vars accessible from top *)
      HashSet.fold (fun x p -> bind x (lookup x o) p) accessible empty
  end
  
module Environment = Make_Environment (Ast)
module Environment_types = Make_Environment (Ast_types)

type env = Environment.t
type env_types = Environment_types.t
type typedefs = typedef list
type env_tuple = env*env_types*typedefs
type env_stack = (env*env_types*typedefs) list
type state = expr * env
type stack_state = expr * env_stack

module Modular_Environment = struct
  let env_dictionary : (id, env_tuple) Hashtbl.t = Hashtbl.create 11
  (* Find the parents of the name *)
  (*let rec find_parents (name : id) : env_tuple list = *)
  let create_new_environment () : env_tuple =
    let new_env = Environment.empty in
    let new_env_types = Environment_types.empty in
    let new_typedefs = [] in
    (new_env,new_env_types,new_typedefs)
  let insert_environment (name : id) (environment : env_tuple) : unit =
    Hashtbl.add env_dictionary name environment; ()
  let garbage_collect top (environment : env_stack) : env_stack =
    (* calling the gc on !top should only be done at the very bottom
       of the stack, not above; because !top contains exactly all the
       names bound at the toplevel, at the bottom level;
       above it should be called with something else, TBD.
       for now, suppressing the gc [Jean-Baptiste] *)
    environment
   (*let f acc (o,oT,td) = (Environment.gc !top o, oT,td)::acc in
   List.rev(List.fold_left f [] environment) *)
end
