open Ast
open Ast_types
open Map
open Util

exception Unbound of id

(* Functional map with vars as keys *)
module VarMap = Map.Make (struct
  type t = id
  let compare = String.compare
end)

module type FV = sig
  type data
  val free_vars : id Util.HashSet.t -> data -> unit
end

module type ENVIRONMENT = functor (A : FV) ->
  sig
    type data = A.data
  	type t
  	val empty : t
  	val bind : id -> data -> t -> t
  	val bound : id -> t -> bool
    val iter : (id -> data -> unit) -> t -> unit
    val fold : (id -> data -> 'b -> 'b) -> t -> 'b -> 'b
    val map : (data -> data) -> t -> t
    val remove : id -> t -> t
  	val lookup : id -> t -> data
  	val gc : id HashSet.t -> t -> t
  end

module Make_Environment : ENVIRONMENT = functor (A : FV) ->
  struct
    type data = A.data
  	type t = data VarMap.t
  	let empty = VarMap.empty
  	let bind = VarMap.add
  	let bound = VarMap.mem
  	let iter = VarMap.iter
    let fold = VarMap.fold

     let map = VarMap.map
  	let remove = VarMap.remove
  	let lookup x o = try VarMap.find x o with Not_found -> raise (Unbound x)
      
    (* garbage collection *)
    (* starting at top-level, do bfs on the given environment *)    
  	let gc (top : id HashSet.t) (o : t) : t =
      (* put all top-level items on the worklist *)
      let worklist = HashSet.make () in
      HashSet.add_all worklist top;
      (* accumulate accessible vars in accum *)
      let accessible = HashSet.make() in
      let next = HashSet.make() in
      while not (HashSet.empty worklist) do
        HashSet.add_all accessible worklist;
        (* get free vars of bindings in worklist *)
        HashSet.clear next;
        let add_to_next = A.free_vars next in
        HashSet.iter (fun x -> add_to_next (lookup x o)) worklist;
        (* filter out ones already seen *)
        HashSet.remove_all next accessible;
        HashSet.clear worklist;
        HashSet.add_all worklist next;
      done;
      (* accessible now contains all vars accessible from top *)
      HashSet.fold (fun x p -> bind x (lookup x o) p) accessible empty
  end
  
module Environment = Make_Environment (Ast)
module Environment_types = Make_Environment (Ast_types)

type env = Environment.t
type state = expr * env
