
type id = string
type idtype = string

(* typeNC = type NoCaml *)
type typeNC =
  | Integer
  | FloatType
  | String
  | Boolean
  | UnitType
  | Arrow of typeNC * typeNC
  | VarType of idtype
  | TupleType of typeNC list
  | UserType of string * typeNC list
    (* just the name of the type, followed by the list of argument types
       those argument types could be VarType *)
  | DummyType
(* Null means no type specified. It is used in the parser to handle the cases
   when the programmer doesn't declare all the types.
   For example, "fun x (y : int) -> x" creates the AST node
   FunType ([Var x; Var y], [Null; Integer], Var x) *)
  | Null

type pattern =
  | PVar of id
  | PInt of int
  | PFloat of float
  | PStrg of string
  | PBool of bool
  | PUnit
  | PTuple of pattern list
  | PInj of id * pattern option

type expr =
  | Var of id
  | Int of int
  | Float of float
  | Strg of string
  | Bool of bool
  | Fun of expr list * expr
  | FunType of expr list * typeNC list * expr
  | FunCorec of expr * id * expr list * expr
  | Let of id * expr list * expr * expr
  | LetType of id * typeNC * expr list * typeNC list* expr * expr
  | Letrec of id * expr list * expr * expr
  | LetrecType of id * typeNC * expr list * typeNC list * expr * expr
  | Letcorec of expr * id * expr list * expr * expr
  | LetcorecType of expr * id * typeNC * expr list * typeNC list * expr * expr
  | App of expr * expr
  | Assg of id * expr
  | Seq of expr * expr
  | If of expr * expr * expr
  | While of expr * expr
  | For of id * expr * expr * expr
  (* Comparison operators *)
  | Eq of expr * expr
  | Neq of expr * expr
  | Lt of expr * expr
  | Le of expr * expr
  | Gt of expr * expr
  | Ge of expr * expr
  (* Arithmetic operators on integers*)
  | Plus of expr * expr
  | Minus of expr * expr
  | Mul of expr * expr
  | Div of expr * expr
  | Mod of expr * expr
  (* Arithmetic operators on floats *)
  | PlusF of expr * expr
  | MinusF of expr * expr
  | MulF of expr * expr
  | DivF of expr * expr
  (* String operators *)
  | Concat of expr * expr
  (* Boolean operators *)
  | Not of expr
  | And of expr * expr
  | Or of expr * expr
  (* Unit *)
  | Unit
  (* Tuples *)
  | Tuple of expr list
  (* Sum types *)
  | Inj of string * (expr option)
  (* Match *)
  | Match of expr * ((pattern * expr) list)
  (* Used internally *)
  | Dummy
  | Unknown of id (* for use in corecursive functions *)
  | Native of id * (expr -> expr) * typeNC

type data = expr

type directive =
  | Load of string
  | Chdir of string
  | Pwd
  | Ls
  | Env
  | Scope
  | Type_inf
  | Quit

type typedef =  id * ((idtype list) * ((id * typeNC option) list))
 (* type name, arguments as variables, list of contructors with arguments *)
type declaration =
  | Def of id * expr list * expr
  | DefType of id * typeNC * expr list * typeNC list * expr
  | Defrec of id * expr list * expr
  | DefrecType of id * typeNC * expr list * typeNC list * expr
  | Defcorec of expr * id * expr list * expr
  | DefcorecType of expr * id * typeNC * expr list * typeNC list * expr
  | Typedef of typedef

type toplevel =
  | Directive of directive
  | Declarations of declaration list
  | Expression of expr

(* character encoding *)
let utf8 = ref false

let rec bound_vars (p:pattern) = match p with
  | PVar x -> [ x ]
  | PTuple l -> List.flatten(List.map bound_vars l)
  | PInj(_, None) -> [ ]
  | PInj(_, Some p) -> bound_vars p
  | PInt _ | PFloat _ | PStrg _ | PBool _ | PUnit -> [ ]

(* Collect all free variables in an expression *)
let free_vars (fv : id Util.HashSet.t) (e : expr) : unit =
  (* exclude bound vars in scope *)
  let free (bv : id list) (x : id) : unit =
    if not (List.mem x bv) then Util.HashSet.add fv x in
  let args s = List.filter ((!=) "()") (List.map (function (Var x) -> x | _ -> "()") s) in
  let rec free_vars (bv : id list) (e : expr) : unit =
    match e with
    (* Binding operators *)
    | Fun (x, e) -> free_vars (args x @ bv) e
    | FunType (x, _, e) -> free_vars bv (Fun (x, e))
    | FunCorec (_, _, x, e) -> free_vars bv (Fun (x, e))
    | Let (f, x, e1, e2) -> free_vars (args x @ bv) e1; free_vars (f :: bv) e2
    | LetType (f, _, x, _, e1, e2) -> free_vars bv (Let (f, x, e1, e2))
    | Letrec (f, x, e1, e2) -> free_vars (f :: args x @ bv) e1; free_vars (f :: bv) e2
    | LetrecType (f, _, x, _, e1, e2) -> free_vars bv (Letrec (f, x, e1, e2))
    | Letcorec (_, f, x, e1, e2) | LetcorecType(_, f, _, x, _, e1, e2) ->
        free_vars bv (Letrec(f, x, e1, e2))
    (* Non-binding operators *)
    | Var x -> free bv x
    | Assg (x, e) -> free bv x; free_vars bv e
    | For (x, e1, e2, e3) -> free bv x; List.iter (free_vars bv) [e1; e2; e3]
    | Match (e, l) -> free_vars bv e; 
                      List.iter (fun (pi, ei) -> free_vars ((bound_vars pi) @ bv) ei) l
    | (Int _ | Float _ | Strg _ | Bool _ | Dummy | Native _ | Unit) -> ()
    | (Eq (e1, e2) | Neq (e1, e2) | Lt (e1, e2) | Le (e1, e2) | Gt (e1, e2) | 
       Ge (e1, e2) | Plus (e1, e2) | Minus (e1, e2) | Mul (e1, e2) | Div (e1, e2) |
       Mod(e1, e2) |
       PlusF (e1, e2) | MinusF (e1, e2) | MulF (e1, e2) | DivF (e1, e2) |
       Concat (e1, e2) | App (e1, e2) | Seq (e1, e2) | While (e1, e2) |
       And (e1, e2) | Or (e1, e2)) -> free_vars bv e1; free_vars bv e2
    | If (e1, e2, e3) -> List.iter (free_vars bv) [e1; e2; e3]
    | Not e1 -> free_vars bv e1 
    | Tuple l -> List.iter (free_vars bv) l 
    | Inj(_, None) -> ( )
    | Inj(_, Some e1) -> (free_vars bv e1)
    | Unknown _ -> ( ) in
  free_vars [] e

let rec dump = function (* to debug. TODO: finish it *)
  | Var x -> "Var(" ^ x ^ ")"
  | Int i -> "Int(" ^ (string_of_int i) ^ ")"
  | Float f -> "Float(" ^ (string_of_float f) ^ ")"
  | Strg s -> "Strg(" ^ s ^ ")"
  | Bool b -> "Bool(" ^ (string_of_bool b) ^ ")"
(*  | Fun of expr list * expr
  | FunType of expr list * typeNC list * expr
  | FunCorec of expr * id * expr list * expr
  | Let of id * expr list * expr * expr
  | LetType of id * typeNC * expr list * typeNC list* expr * expr
  | Letrec of id * expr list * expr * expr
  | LetrecType of id * typeNC * expr list * typeNC list * expr * expr
  | Letcorec of expr * id * expr list * expr * expr
  | LetcorecType of expr * id * typeNC * expr list * typeNC list * expr * expr
*)
  | App(e1, e2) -> "App(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  | Assg(i, e2) -> "Assg(" ^ i ^ ", " ^ (dump e2) ^ ")"
  | Seq(e1, e2) -> "Seq(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  | If(e1, e2, e3) -> 
    "If(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ", " ^ (dump e2)^ ")"
  | While(e1, e2) -> "While(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  | For(i, e1, e2, e3) -> 
    "For(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ", " ^ (dump e3) ^ ")"
  (* Comparison operators *)
  | Eq(e1, e2) -> "Eq(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  | Neq(e1, e2) -> "Neq(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  | Lt(e1, e2) -> "Lt(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  | Le(e1, e2) -> "Le(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  | Gt(e1, e2) -> "Gt(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  | Ge(e1, e2) -> "Ge(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  (* Arithmetic operators on integers*)
  | Plus(e1, e2) -> "Plus(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  | Minus(e1, e2) -> "Minus(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  | Mul(e1, e2) -> "Mul(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  | Div(e1, e2) -> "Div(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  | Mod(e1, e2) -> "Mod(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  (* Arithmetic operators on floats *)
  | PlusF(e1, e2) -> "PlusF(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  | MinusF(e1, e2) -> "MinusF(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  | MulF(e1, e2) -> "MulF(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  | DivF(e1, e2) -> "DivF(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  (* String operators *)
  | Concat(e1, e2) -> "Concat(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  (* Boolean operators *)
  | Not(e1) -> "Not(" ^ (dump e1) ^ ")"
  | And(e1, e2) -> "And(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  | Or(e1, e2) -> "Or(" ^ (dump e1) ^ ", " ^ (dump e2) ^ ")"
  (* Unit *)
  | Unit -> "Unit"
  (* Tuples *)
  | Tuple(el) -> 
    "Tuple [" ^ (String.concat "; " (List.map dump el)) ^ "]"
  (* Sum types *)
  | Inj(i, None) -> "Inj(" ^ i ^ ", None)"
  | Inj(i, Some(e)) -> "Inj(" ^ i ^ ", " ^ "Some(" ^ (dump e) ^ "))"
  (* Match *)
(*  | Match of expr * ((pattern * expr) list) *)
  (* Used internally *)
  | Dummy -> "Dummy"
  | Unknown id -> "Unknown(" ^ id ^ ")" 
  (* for use in corecursive functions *)
  | Native(i, _, t) -> "Native(" ^ i ^ ", " ^ "<fun>" ^ ")"
  | _ -> failwith "TODO dump"

exception Runtime of string
(* for runtime errors other than unbound vars *)
let runtime (s : string) : 'a = raise (Runtime s)
