
exception Fatal of string

let uncurry f (a, b) = f a b

(*****************************************************
 * string operations
 *****************************************************)

let ends_with (s : string) (t : string) : bool =
  let n = String.length t - String.length s in
  n >= 0 && Str.string_match (Str.regexp_string s) t n
  
let trim (s : string) : string =
  let s = Str.split (Str.regexp "[ \t\n]+") s in
  String.concat " " s
  
(*****************************************************
 * list operations
 *****************************************************)

(* Returns the list of elements in l1 not in l2 *)
let rec list_difference l1 l2 =
  List.filter (fun x -> not (List.mem x l2)) l1

(* Removes duplicates from a list *)
let rec remove_duplicates l =
  List.fold_right (fun x t -> if List.mem x t then t else x :: t) l []

(*****************************************************
 * I/O
 *****************************************************)
let output_endline (out : out_channel) (s : string) : unit =
  output_string out s;
  output_char out '\n'
  
let copy_lines in_channel out_channel : unit =
  try
    while true do
      output_endline out_channel (input_line in_channel)
    done
  with End_of_file -> ()
  
(*****************************************************
 * HashSet
 *****************************************************)
module type HashSet = sig
  type 'a t
  val empty : 'a t -> bool
  val make : unit -> 'a t
  val clear : 'a t -> unit
  val mem : 'a t -> 'a -> bool
  val add : 'a t -> 'a -> unit
  val remove : 'a t -> 'a -> unit
	val iter : ('a -> unit) -> 'a t -> unit
  val fold : ('a -> 'c -> 'c) -> 'a t -> 'c -> 'c
  val add_all : 'a t -> 'a t -> unit
  val remove_all : 'a t -> 'a t -> unit
  val to_list : 'a t -> 'a list
end

module HashSet : HashSet = struct
  type 'a t = ('a, unit) Hashtbl.t
  let empty h = Hashtbl.length h = 0
  let make () = Hashtbl.create 11
  let clear = Hashtbl.clear
  let mem h x = Hashtbl.mem h x
  let add h x = if not (mem h x) then Hashtbl.add h x ()
  let remove h x =
    while mem h x do
      Hashtbl.remove h x
    done
	let iter f = Hashtbl.iter (fun x _ -> f x)
	let fold f = Hashtbl.fold (fun x _ -> f x)
	let add_all h = iter (fun x -> add h x) 
	let remove_all h = iter (fun x -> remove h x)
  let to_list h = fold (fun x l -> x :: l) h []
end

(*****************************************************
 * Gensym - generate fresh symbols
 *****************************************************)
module type Gensym = sig
  type t
  val make : unit -> t
  val reset : t -> unit
  val next : t -> string
end

module Gensym : Gensym = struct
  type t = int ref
        
  let make () = ref 0
  
  let reset h = h := 0

  let next h =
    let l = !h in
    incr h; "%" ^ string_of_int l
end

(*****************************************************
 * stream of strings in length-lexicographic order
 *****************************************************)
module LexStream : Gensym = struct
  type t = char list ref
  
  let next (c : char) : char =
    Char.chr (Char.code c + 1)
  
  let rec inc (s : char list) : char list =
    match s with
      | [] -> ['a']
      | x :: t ->
          if Char.compare x 'z' < 0 then (next x) :: t
          else 'a' :: inc t
        
  let make () = ref ['a']
  
  let reset h = h := ['a']

  let next h =
    let l = !h in
    h := inc l;
    "'" ^ (String.concat "" (List.map (String.make 1) (List.rev l)))
end
