
open Ast
open Parser
open Util
open Environment

exception Runtime of string

type env = Environment.t
type state = expr * env

(* global flag for static or dynamic scope *)
let static_scope = ref true

(* for runtime errors other than unbound vars *)
let runtime (s : string) : 'a = raise (Runtime s)

(* global internal symbol generator *)
let fresh : unit -> id =
  let gs : Gensym.t = Gensym.make () in
  fun () -> Gensym.next gs

(* read from a file *)
let read_file (filename : string) : string =
	let rec get_contents contents file =
		let input_line_option file =
			try Some ((input_line file) ^ "\n") with End_of_file -> None in
		match (input_line_option file) with
		  Some x -> get_contents (contents ^ x) file
		| None -> contents in
	let file = try open_in filename with Not_found -> runtime "File not found" in
	let result = get_contents "" file in close_in file; result

(* get a fresh variable not occurring in an expression *)

(* Substitute var y for free occurrences of var x in e *)
let subst (y : id) (x : id) (e : expr) : expr =
  let match_args = function (Var id) -> id = x | _ -> false in
  let rec sub (e : expr) : expr =
	match e with
		| Var id -> if id = x then Var y else Var id
		| (Int _ | Float _ | Strg _ | Bool _ | Dummy | Native _ | Unit) -> e
		| Fun (args, c) -> if List.exists match_args args then e else Fun (args, sub c)
		| FunType (args,_,e) -> sub (Fun(args,e))
		| Let (id, args, e1, e2) ->
			  let e1 = if (List.exists match_args args) then e1 else sub e1 in
			  let e2 = if id = x then e2 else sub e2 in
			  Let (id, args, e1, e2)
		| LetType (id, _, args,_,e1,e2) -> sub (Let (id, args, e1, e2))
		| Letrec (id, args, e1, e2) ->
			  let e1 = if (id = x || List.exists match_args args) then e1 else sub e1 in
			  let e2 = if id = x then e2 else sub e2 in
			  Letrec (id, args, e1, e2)
		| LetrecType (id, _, args,_,e1,e2) -> sub (Letrec (id, args, e1, e2))
		| App (e1, e2) -> App (sub e1, sub e2)
		| If (e1, e2, e3) -> If (sub e1, sub e2, sub e3)
		| While (e1, e2) -> While (sub e1, sub e2)
		| For (id, e1, e2, e3) ->
        For ((if id = x then y else id), sub e1, sub e2, sub e3)
		| Assg (id, e2) -> Assg ((if id = x then y else id), sub e2)
		| Seq (e1, e2) -> Seq (sub e1, sub e2) 
		(* Lists *)
		| ListMake e1 -> ListMake (sub e1)
		| Cons (e1, e2) -> Cons (sub e1, sub e2)
		| List lst -> List (List.map (fun e1 -> sub e1) lst)
		(* Comparison operators *)
		| Eq (e1, e2) -> Eq (sub e1, sub e2)
		| Neq (e1, e2) -> Neq (sub e1, sub e2)
		| Lt (e1, e2) -> Lt (sub e1, sub e2)
		| Le (e1, e2) -> Le (sub e1, sub e2)
		| Gt (e1, e2) -> Gt (sub e1, sub e2)
		| Ge (e1, e2) -> Ge (sub e1, sub e2)
		(* Arithmetic operators on integers *)
		| Plus (e1, e2) -> Plus (sub e1, sub e2)
		| Minus (e1, e2) -> Minus (sub e1, sub e2)
		| Mul (e1, e2) -> Mul (sub e1, sub e2)
		| Div (e1, e2) -> Div (sub e1, sub e2)
		(* Arithmetic operators on floats *)
		| PlusF (e1, e2) -> PlusF (sub e1, sub e2)
		| MinusF (e1, e2) -> MinusF (sub e1, sub e2)
		| MulF (e1, e2) -> MulF (sub e1, sub e2)
		| DivF (e1, e2) -> DivF (sub e1, sub e2)
		(* String operators *)
		| Concat (e1, e2) -> Concat (sub e1, sub e2)
    (* Boolean operators *)
		| Not e1 -> Not(sub e1)
		| And (e1,e2) -> And(sub e1, sub e2)
		| Or (e1,e2 )-> Or(sub e1, sub e2) in
  sub e
		
let rec eval ((e, o) : state) : state =
	match e with
		| Var id -> (Environment.lookup id o, o)
		| Int i -> (e, o)
		| Float f -> (e, o)
		| Strg s -> (e, o)
		| Bool b -> (e, o)
		| Fun (_, _) -> (e, o)
		| FunType (args, _, c) -> (Fun (args,c), o)
		| Let (id, args, e1, e2) ->
			if args = [] then eval (App (Fun ([Var id], e2), e1), o)
			else eval (App (Fun ([Var id], e2), Fun (args, e1)), o)
		| LetType (id, _, args, _, e1,e2) ->
		  eval (Let (id, args, e1, e2), o)
		| Letrec (id, args, e1, e2) ->
			if args = [] then
				(* Let id = <dummy> in (id := e1); e2 *)
				eval (Let (id, [], Dummy, Seq (Assg (id, e1), e2)), o)
			else
				(* Let id = <dummy> in (id := fun args -> e1); e2 *)
				eval (Let (id, [], Dummy, Seq (Assg (id, Fun (args, e1)), e2)), o)
		| LetrecType (id, _, args, _, e1,e2) ->
		  eval (Letrec (id, args, e1, e2), o)
		| App (e1, e2) ->
				let (d, p1) = eval (e1, o) in
			  let (u, p2) = eval (e2, p1) in
				(match d with
  				| Fun (args, c) ->
  					(match args with
  					  | Var x :: t ->
                if !static_scope then
                	let y = fresh () in
                  eval (subst y x (if t = [] then c else Fun (t, c)), Environment.bind y u p2)
                else (* dynamic scope *)
  			          eval ((if t = [] then c else Fun (t, c)), Environment.bind x u p2)
              | Unit :: t ->
                 if u = Unit then eval ((if t = [] then c else Fun (t, c)), p2)
                 else runtime "argument mismatch"
              | _ -> runtime "not a funtion")
          | Native (_, native, _) -> (native u, p2)
  				| _ -> runtime "can only apply a function")
		| If (e, e1, e2) ->
			  let (d, t) = eval (e, o) in
		  	(match d with
			  	| Bool b -> eval ((if b then e1 else e2), t)
			  	| _ -> runtime "'if' requires a boolean test")
		| While (e, e1) ->
			let (d,t) = eval (e,o) in
			(match d with
				| Bool b ->
					if b then eval (Seq (e1, While (e, e1)), t)
					else (Unit, t)
				| _ -> runtime "'while' requires a boolean test")
		| For (x, e1, e2, e3) -> 
		  let (d,t) = eval (e1,o) and (f,u) = eval (e2,o) in
			(match (d,f) with
  			| ( Int n, Int p) ->
          if n>p then (Unit,o)
          else eval (Seq (e3, For (x,  (Plus (d, (Int 1))),f, e3)), Environment.bind x d o)
  			| _ -> runtime " the second and third arguments of 'for' must evaluate to integers ") 	
		| Assg (x, e) ->
        if not (Environment.bound x o) then raise (Unbound x) else
        let (v, t) = eval (e, o) in
        (v, Environment.bind x v t)
		| Seq (e1, e2) ->
			  let (_, t) = eval (e1, o) in
			  eval (e2, t)
        
		(* Lists *)
		| ListMake e1 ->
			(match e1 with
				| Seq (e2, e3) ->
					let (d, p1) = eval (ListMake e2, o) in
					let (u, p2) = eval (e3, p1) in
					(match d with
						| List lst -> (List (lst @ [u]), p2)
						| _ -> runtime "improper list")
				| _ -> let (d,p) = eval (e1,o) in (List [d],p))
		| Cons (e1, e2) ->
			let (d, p1) = eval (e1, o) in
			let (u, p2) = eval (e2, p1) in
			(match u with
				| List lst -> (List (d :: lst), p2)
				| _ -> runtime "can only cons to a list")
		| List lst -> (e, o)

		(* Comparison operators *)
		| Eq (e1, e2) ->
			let (p1,t) = eval (e1,o) in
			let (p2,r) = eval (e2,t) in
			(match (p1,p2) with
			| (Int i, Int j) -> (Bool (i=j), r)
			| (Bool b, Bool c) -> (Bool (b=c), r)
			| (Strg s, Strg t) -> (Bool (s=t), r)
			| (List a, List b) ->
				(match (a,b) with
					| ([],[]) -> (Bool true, r)
					| ([],_) -> (Bool false, r)
					| (_,[]) -> (Bool false, r)
					| (ah::at,bh::bt) ->
						let (cmp,p) = eval (Eq (ah, bh), r) in
						(match cmp with
							| Bool b -> if b then eval (Eq (List at, List bt), p) else (cmp,p)
							| _ -> raise (Fatal "Eq did not return Bool")))
			| _ -> runtime "invalid types for comparison")
		| Neq (e1, e2) ->
			let (d,t) = eval (Eq (e1, e2), o) in
			(match d with
				| Bool b -> (Bool (not b), t)
				| _ ->  raise (Fatal "Eq did not return Bool"))
		| Lt (e1, e2) -> eval_comp (<) (<) e1 e2 o
		| Le (e1, e2) -> eval_comp (<=) (<=) e1 e2 o
		| Gt (e1, e2) -> eval_comp (>) (>) e1 e2 o
		| Ge (e1, e2) -> eval_comp (>=) (>=) e1 e2 o

		(* Arithmetic operators on integers *)
		| Plus (e1, e2) -> eval_arith (+) e1 e2 o
		| Minus (e1, e2) -> eval_arith (-) e1 e2 o
		| Mul (e1, e2) -> eval_arith ( * ) e1 e2 o
		| Div (e1, e2) -> eval_arith (/) e1 e2 o

		(* Arithmetic operators on floats *)
		| PlusF (e1, e2) -> eval_arithF (+.) e1 e2 o
		| MinusF (e1, e2) -> eval_arithF (-.) e1 e2 o
		| MulF (e1, e2) -> eval_arithF ( *. ) e1 e2 o
		| DivF (e1, e2) -> eval_arithF (/.) e1 e2 o

		(* String operators *)
		| Concat (e1, e2) -> eval_strg (^) e1 e2 o

		(* Boolean operators *)
		| Not e1 ->
	    (match eval (e1, o) with
	      | (Bool b, t) -> (Bool (not b), t)
	      | _ -> runtime "can only use negation on booleans"
	    )	 
		| And (e1, e2) ->
       (match eval (e1, o) with
			   | (Bool b, t) as d ->
             if b then
      		   (match eval (e2, t) with
      		     | (Bool b, _) as d -> d
      		     | _ -> runtime "can only use && on booleans")
             else d
         | _ -> runtime "can only use && on booleans")
		| Or (e1, e2) ->
       (match eval (e1, o) with
			   | (Bool b, t) as d ->
             if b then d
             else
      		   (match eval (e2, t) with
      		     | (Bool b, _) as d -> d
      		     | _ -> runtime "can only use && on booleans")
         | _ -> runtime "can only use && on booleans")
		| Unit -> (e, o)
		| Native _ -> (e, o)

		(* Dummy is used internally *)
		| Dummy -> (e, o)
      
and eval_comp (op_i:int->int->bool) (op_b:bool->bool->bool) (e1:expr) (e2:expr) (o:env) :state =
	let (p1,t) = eval (e1,o) in
	let (p2,r) = eval (e2,t) in
	(match (p1,p2) with
		| (Int i, Int j) -> (Bool (op_i i j), r)
		| (Bool b, Bool c) -> (Bool (op_b b c), r)
		| _ -> runtime "can only compare ints to ints and bools to bools")

and eval_arith (op:int->int->int) (e1:expr) (e2:expr) (o:env) :state =
	let (p1,t) = eval (e1,o) in
	let (p2,r) = eval (e2,t) in
	(match (p1,p2) with
		| (Int i, Int j) -> (Int (op i j), r)
		| _ -> runtime "can only do arithmetic on ints or floats")

and eval_arithF (op:float->float->float) (e1:expr) (e2:expr) (o:env) :state =
	let (p1,t) = eval (e1,o) in
	let (p2,r) = eval (e2,t) in
	(match (p1,p2) with
		| (Float i, Float j) -> (Float (op i j), r)
		| _ -> runtime "can only do arithmetic on ints or floats")

and eval_strg (op:string->string->string) (e1:expr) (e2:expr) (o : env) : state =
	let (p1,t) = eval (e1,o) in
	let (p2,r) = eval (e2,t) in
	(match (p1,p2) with
		| (Strg i, Strg j) -> (Strg (op i j), r)
		| _ -> runtime "can only do string operations on strings")
