
type id = string

(* typeNC = type NoCaml *)
type typeNC =
  | Integer
  | FloatType
  | String
  | Boolean
  | UnitType
  | Arrow of typeNC * typeNC
  | VarType of id
  | ListType of typeNC
  | DummyType
(* Null means no type specified. It is used in the parser to handle the cases
   when the programmer doesn't declare all the types.
   For example, "fun x (y : int) -> x" creates the AST node
   FunType ([Var x; Var y], [Null; Integer], Var x) *)
  | Null

type expr =
  | Var of id
  | Int of int
  | Float of float
  | Strg of string
  | Bool of bool
  | Fun of expr list * expr
  | FunType of expr list * typeNC list * expr
  | Let of id * expr list * expr * expr
  | LetType of id * typeNC * expr list * typeNC list* expr * expr
  | Letrec of id * expr list * expr * expr
  | LetrecType of id * typeNC * expr list * typeNC list * expr * expr
  | App of expr * expr
  | Assg of id * expr
  | Seq of expr * expr
  | If of expr * expr * expr
  | While of expr * expr
  | For of id * expr * expr * expr
  (* Lists *)
  | ListMake of expr
  | Cons of expr * expr
  | List of expr list
  (* Comparison operators *)
  | Eq of expr * expr
  | Neq of expr * expr
  | Lt of expr * expr
  | Le of expr * expr
  | Gt of expr * expr
  | Ge of expr * expr
  (* Arithmetic operators on integers*)
  | Plus of expr * expr
  | Minus of expr * expr
  | Mul of expr * expr
  | Div of expr * expr
  (* Arithmetic operators on floats *)
  | PlusF of expr * expr
  | MinusF of expr * expr
  | MulF of expr * expr
  | DivF of expr * expr
  (* String operators *)
  | Concat of expr * expr
  (* Boolean operators *)
  | Not of expr
  | And of expr * expr
  | Or of expr * expr
  (* Unit *)
  | Unit
  (* Used internally *)
  | Dummy
  | Native of id * (expr -> expr) * typeNC

type data = expr

type directive =
  | Load of string
  | Chdir of string
  | Pwd
  | Ls
  | Env
  | Scope
  | Type_inf
  | Quit

type declaration =
  | Def of id * expr list * expr
  | DefType of id * typeNC * expr list * typeNC list * expr
  | Defrec of id * expr list * expr
  | DefrecType of id * typeNC * expr list * typeNC list * expr

type toplevel =
  | Directive of directive
  | Declarations of declaration list
  | Expression of expr

(* character encoding *)
let utf8 = ref false

let precedence = function
	| Dummy -> failwith "unexpected dummy"
	| Native _ -> 9
	| Var _ -> 9
	| Int _ -> 9
	| Float _ -> 9
	| Strg _ -> 9
	| Bool _ -> 9
	| Unit -> 9
	| List _ -> 9
	| ListMake _ -> 9
	| Not _ -> 8
	| App _ -> 7
	| Cons _ -> 6
	| Eq _ -> 5
	| Neq _ -> 5
	| Lt _ -> 5
	| Le _ -> 5
	| Gt _ -> 5
	| Ge _ -> 5
	| Plus _ -> 4
	| Minus _ -> 4
	| PlusF _ -> 4
	| MinusF _ -> 4
	| Concat _ -> 4
	| Or _ -> 4
	| Mul _ -> 3
	| Div _ -> 3
	| MulF _ -> 3
	| DivF _ -> 3
	| And _ -> 3
	| Assg _ -> 2
	| Seq _ -> 1
	| If _ -> 0
	| While _ -> 0
	| For _ -> 0
	| Fun _ -> 0
	| FunType _ -> 0
	| Let _ -> 0
	| LetType _ -> 0
	| Letrec _ -> 0
	| LetrecType _ -> 0
	

type assoc = LEFT | RIGHT | PREFIX | SUFFIX | ASSOC | NONASSOC

let assoc = function
  | App _ -> LEFT
  | Assg _ -> RIGHT
  | Cons _ -> RIGHT
  | Seq _ -> ASSOC
  | If _ -> PREFIX
  | Fun _ -> PREFIX
  | Let _ -> PREFIX
  | Not _ -> PREFIX
  | _ -> NONASSOC

let rec to_string (e : expr) : string =
  (* parenthesize based on associativity and precedence *)
  let protect_left e e1 : string =
    let (p, p1) = (precedence e, precedence e1) in
    let s = to_string e1 in
    if (p1 < p && assoc e1 <> SUFFIX) || (p = p1 && assoc e1 = RIGHT) then Printf.sprintf "(%s)" s else s in
  let protect_right e e1 : string =
    let (p, p1) = (precedence e, precedence e1) in
    let s = to_string e1 in
    if (p1 < p && assoc e1 <> PREFIX) || (p = p1 && assoc e1 = LEFT) then Printf.sprintf "(%s)" s else s in
	let binop_print op e e1 e2 = Printf.sprintf "%s %s %s" (protect_left e e1) op (protect_right e e2) in
  let arglist x = String.concat " " (List.map to_string x) in
  match e with
    | Var x -> x
    | Int x -> string_of_int x
    | Float x -> string_of_float x
    | Strg x -> "\"" ^ String.escaped x ^ "\""
    | Bool b -> string_of_bool b
    | Fun (x, e1) ->
        Printf.sprintf "fun %s -> %s" (arglist x) (protect_left e e1)
    | FunType(x, _, e1) -> to_string (Fun(x,e1))
    | If (e1, e2, e3) ->
        Printf.sprintf "if %s then %s else %s" (protect_left e e1) (protect_left e e2) (protect_left e e3)
    | While (e1, e2) ->
        Printf.sprintf "while %s do %s done" (to_string e1) (to_string e2)
    | For (x, e1, e2, e3) ->
        Printf.sprintf "for %s = %s to %s do %s done" x (to_string e1) (to_string e2) (to_string e3)
    | Let (x, [], e1, e2) ->
        Printf.sprintf "let %s = %s in %s" x (protect_left e e1) (protect_left e e2)
    | Let (x, y, e1, e2) ->
		  	Printf.sprintf "let %s %s = %s in %s" x (arglist y) (protect_left e e1) (protect_left e e2)
    | LetType (x ,_,y,_,e1,e2) -> to_string (Let(x,y,e1,e2))
    | Letrec (x, [], e1, e2) ->
  			Printf.sprintf "let rec %s = %s in %s" x (protect_left e e1) (protect_left e e2)
    | Letrec (x, y, e1, e2) ->
			  Printf.sprintf "let rec %s %s = %s in %s" x (arglist y) (protect_left e e1) (protect_left e e2)
    | LetrecType (x ,_,y,_,e1,e2) -> to_string (Letrec(x,y,e1,e2))
    | App (e1, e2) -> Printf.sprintf "%s %s" (protect_left e e1) (protect_right e e2)
    | Assg (x, e1) -> Printf.sprintf "%s := %s" x (protect_right e e1)
    | Seq (e1, e2) -> Printf.sprintf "%s; %s" (protect_left e e1) (protect_right e e2)

    (* Lists *)
    | ListMake e1 -> Printf.sprintf "[%s]" (to_string e1)
    | Cons (e1, e2) -> Printf.sprintf "%s :: %s" (protect_left e e1) (protect_right e e2)
    | List l -> Printf.sprintf "[%s]" (String.concat "; " (List.map (protect_left e) l))

    (* Comparison operators *)
    | Eq (e1, e2) -> binop_print "=" e e1 e2
    | Neq (e1, e2) -> binop_print "!=" e e1 e2
    | Le (e1, e2) -> binop_print "<=" e e1 e2
    | Lt (e1, e2) -> binop_print "<" e e1 e2
    | Gt (e1, e2) -> binop_print ">" e e1 e2
    | Ge (e1, e2) -> binop_print ">=" e e1 e2

    (* Arithmetic operators on integers*)
    | Plus (e1, e2) -> binop_print "+" e e1 e2
    | Minus (e1, e2) -> binop_print "-" e e1 e2
    | Mul (e1, e2) -> binop_print "*" e e1 e2
    | Div (e1, e2) -> binop_print "/" e e1 e2

    (* Arithmetic operators on floats *)
    | PlusF (e1, e2) -> binop_print "+." e e1 e2
    | MinusF (e1, e2) -> binop_print "-." e e1 e2
    | MulF (e1, e2) -> binop_print "*." e e1 e2
    | DivF (e1, e2) -> binop_print "/." e e1 e2

    (* String operators *)
    | Concat (e1, e2) -> binop_print "^" e e1 e2

    (* Boolean operators *)
    | Not e1 -> Printf.sprintf "not %s" (protect_left e e1)
    | And (e1,e2) -> binop_print "&&" e e1 e2
    | Or (e1,e2) ->  binop_print "||" e e1 e2
    | Unit -> "()"
    | Dummy -> "<dummy>"
    | Native _ -> "<native>"
  
(* Collect all free variables in an expression *)
let free_vars (fv : id Util.HashSet.t) (e : expr) : unit =
  (* exclude bound vars in scope *)
  let free (bv : id list) (x : id) : unit =
    if not (List.mem x bv) then Util.HashSet.add fv x in
  let args s = List.filter ((!=) "()") (List.map (function (Var x) -> x | _ -> "()") s) in
  let rec free_vars (bv : id list) (e : expr) : unit =
    match e with
    (* Binding operators *)
    | Fun (x, e) -> free_vars (args x @ bv) e
    | FunType (x, _, e) -> free_vars bv (Fun (x, e))
    | Let (f, x, e1, e2) -> free_vars (args x @ bv) e1; free_vars (f :: bv) e2
    | LetType (f, _, x, _, e1, e2) -> free_vars bv (Let (f, x, e1, e2))
    | Letrec (f, x, e1, e2) -> free_vars (f :: args x @ bv) e1; free_vars (f :: bv) e2
    | LetrecType (f, _, x, _, e1, e2) -> free_vars bv (Letrec (f, x, e1, e2))
    (* Non-binding operators *)
    | Var x -> free bv x
    | Assg (x, e) -> free bv x; free_vars bv e
    | For (x, e1, e2, e3) -> free bv x; List.iter (free_vars bv) [e1; e2; e3]
    | (Int _ | Float _ | Strg _ | Bool _ | Dummy | Native _ | Unit) -> ()
    | (Eq (e1, e2) | Neq (e1, e2) | Lt (e1, e2) | Le (e1, e2) | Gt (e1, e2) | 
       Ge (e1, e2) | Plus (e1, e2) | Minus (e1, e2) | Mul (e1, e2) | Div (e1, e2) |
       PlusF (e1, e2) | MinusF (e1, e2) | MulF (e1, e2) | DivF (e1, e2) |
       Concat (e1, e2) | Cons (e1, e2) | App (e1, e2) | Seq (e1, e2) | While (e1, e2) |
       And (e1, e2) | Or (e1, e2)) -> free_vars bv e1; free_vars bv e2
    | If (e1, e2, e3) -> List.iter (free_vars bv) [e1; e2; e3]
    | ListMake e1 -> free_vars bv e1
    | List lst -> List.iter (free_vars bv) lst
    | Not e1 -> free_vars bv e1 in
  free_vars [] e
