
open Ast

(* A type schema is made up of two elements:
   - a list of (universally quantified) type variables, and
   - a type
For example, fun x -> x has the schema "for all t1, t1 -> t1" represented 
by Schema ([t1], Arrow (t1, t1)) *)

type schema = Schema of id list * typeNC
type data = schema

(* Collect all free variables in a type expression *)
let free_vars (fv : id Util.HashSet.t) (Schema (_, t) : schema) : unit =
  let rec free_vars (e : typeNC) : unit =
  match e with
    | (Integer | FloatType | String | Boolean | DummyType | UnitType | Null) -> ()
    | Arrow (t1, t2) -> free_vars t1; free_vars t2
    | VarType x -> Util.HashSet.add fv x
    | ListType t -> free_vars t in
  free_vars t

let to_string (t : typeNC) : string =
  (* convert to fresh variables *)
  let ls = Util.LexStream.make() in
  let vars = Util.HashSet.make() in
  free_vars vars (Schema ([], t));
  let fresh_vars = Hashtbl.create 11 in
  Util.HashSet.iter (fun x -> Hashtbl.add fresh_vars x (Util.LexStream.next ls)) vars;
  let rec to_string (t : typeNC) : string =
    match t with
      | Integer -> "int"
      | FloatType -> "float"
      | String -> "string"
      | Boolean -> "bool"
      | UnitType -> "unit"
      | Arrow (Arrow _ as t1, t2) ->
          Printf.sprintf "(%s) -> %s" (to_string t1) (to_string t2)
      | Arrow (t1, t2) ->
          Printf.sprintf "%s -> %s" (to_string t1) (to_string t2)
      | VarType x ->
          (try Hashtbl.find fresh_vars x
          with Not_found -> raise (Util.Fatal "unbound variable"))
      | ListType (Arrow _ as t) -> Printf.sprintf "(%s) list" (to_string t)
      | ListType t -> Printf.sprintf "%s list" (to_string t)
      | DummyType -> "internal: dummy"
      | Null -> "null" in
  to_string t

(* For debugging purposes *)
(* Prints an equation system *)
let print_aux l =
  let f (x, y) = Printf.printf "%s = %s\n" (to_string x) (to_string y) in
  print_endline "Equations start:";
  List.iter f l;
  print_endline "Equations end"
