module C = Common

type padic = int * int C.ilist * int list
(* ...13131345.234 in base 7 is represented as 
   ( 7, [ 4; 5; 3; 1; 3; 1; 3 ... ], [ 2; 3; 4 ] ) *)

module Corecursive_Equal          = Corecursive.Corecursive(Equal)
module Corecursive_From_rational  = Corecursive.Corecursive(From_rational)
module Corecursive_To_string      = Corecursive.Corecursive(To_string) (* obsolete *)
module Corecursive_To_lists      = Corecursive.Corecursive(To_lists)
module Corecursive_Addi           = Corecursive.Corecursive(Addi)
module Corecursive_Multi          = Corecursive.Corecursive(Multi)
module Corecursive_From_list      = Corecursive.Corecursive(From_list)
module Corecursive_Divi           = Corecursive.Corecursive(Divi)
module Corecursive_To_float       = Corecursive.Corecursive(To_float)
module Corecursive_Divzeros       = Corecursive.Corecursive(Divzeros)
module Corecursive_Normalize      = Corecursive.Corecursive(Normalize)
let multi p i j = Corecursive_Multi.main (p, i, j, C.Nil)
let divi p i j = Corecursive_Divi.main (p, i, j)

let (*co*)rec naive_equal = function (* default true *)
  | C.Nil, C.Nil -> true
  | C.Cons(h1, t1), C.Cons(h2, t2) -> 
    if h1 = h2 then naive_equal(!t1, !t2) else false
  | C.Nil, C.Cons(0, t) -> naive_equal(C.Nil, !t)
  | C.Cons(0, t), C.Nil -> naive_equal(C.Nil, !t)
  | _ -> false
let rec equal (p1, i1, j1) (p2, i2, j2) =
  if p1 != p2 then
    failwith "Cannot compare p-adic numbers in two different bases"
  else
    (i1 = i2) && (Corecursive_Equal.main (j1, j2))

let (*co*)rec naive_normalize = function
  | C.Nil -> C.Nil
  | i when Corecursive_Equal.main (i, C.Nil) -> C.Nil
  | C.Cons(h, t) -> C.Cons(h, t)
let rec normalize (p, i, j) =
  (p, Corecursive_Normalize.main i, j)

let rec shift n i = (* multiplies i by p^n *)
  if n = 0 then i
  else if n > 0 then 
  (match i with
   | (p, l, [ ] ) -> shift (n-1) (p, C.Cons(0, ref l), [ ])
   | (p, l, hr :: tr) -> shift (n-1) (p, C.Cons(hr, ref l), tr))
  else (* n < 0 *) 
  (match i with
   | (p, C.Nil, r) -> shift (n+1) (p, C.Nil, 0 :: r)
   | (p, C.Cons(hl, tl), r) -> shift (n+1) (p, !tl, hl :: r))

let (*co*)rec naive_from_rational (p, a, b) =
  (* supposes that b is not dividable by p *)
  if a = 0 then C.Nil
  else let d = C.euclid p a b in
       C.Cons (d, ref (naive_from_rational (p, (a - b * d) / p, b)))
let rec from_rational p a b = (* creates a/b in base p *)
  let rec divide_by_p b =
    if b mod p = 0 then let resb, resn = divide_by_p (b/p) in resb, resn + 1
    else b, 0 in
  let rec shift (p, i, j) n = (* essentially divides by p n times *)
    if n = 0 then (p, i, j) else match i with
    | C.Nil -> shift (p, C.Nil, 0 :: j) (n-1)
    | C.Cons(hi, ti) -> shift (p, !ti, hi :: j) (n-1)
  in
    let b', n = divide_by_p b in
    let res = Corecursive_From_rational.main (p, a, b') in
    shift (p, res, [ ]) n

let (*co*)rec naive_from_list = function
 | [ ] -> C.Nil
 | h :: t -> C.Cons(h, ref (naive_from_list t))
let from_lists p (i : int list) (j : int list) =
(* creates a p-adic number with left part i and right part j, in base p *)
  (p, Corecursive_From_list.main i, j)

let (*co*)rec naive_to_string (p, i, j) = failwith "TODO"
let old_to_string (p, i, j) = 
(* buggy for p > 35, but keeping it for archive purposes *)
(* for details look at module To_string.ml               *)
  let r, nr = Corecursive_To_string.main i in
  let f = String.concat "" (List.map C.string_of_int2 j) in
  (if r = "" then (if nr = "" then "0" else nr) ^ "." ^ f
   else "..." ^ r ^ r ^ r ^ nr ^ "." ^ f ^ " = " ^
        "(" ^ r ^ ")" ^ nr ^ "." ^ f)
  ^ " (base " ^ (string_of_int p) ^ ")"

let to_string (p, i, j) =
  let f = String.concat "" (List.map C.string_of_int2 j) in
  let nrl, rl = Corecursive_To_lists.main i in
  let nr = String.concat "" (List.rev (List.map C.string_of_int2 nrl)) in
  let r  = String.concat "" (List.rev (List.map C.string_of_int2 rl)) in
  (if r = "" then (if nr = "" then "0" else nr) ^ "." ^ f
   else "..." ^ r ^ r ^ r ^ nr ^ "." ^ f ^ " = " ^
        "(" ^ r ^ ")" ^ nr ^ "." ^ f)
  ^ " (base " ^ (string_of_int p) ^ ")"

let print p = print_string (to_string p); print_newline()

let (*co*)rec naive_to_float (p, i) = match i with
  | C.Nil -> 0.
  | C.Cons(h, t) -> 
      (float_of_int h) +. (float_of_int p) *. (naive_to_float (p, !t))
let to_float (p, i, j) =
  let pf = float_of_int p in
  let rec list_to_float = function 
    | [] -> 0.
    | h :: t -> ((float_of_int h) +. (list_to_float t)) /. pf in
  (Corecursive_To_float.main (p, i)) +. (list_to_float j)

let (*co*)rec naive_addi (p, i, j, c) = match i, j with
  | C.Nil, C.Nil -> if c = 0 then C.Nil 
                else C.Cons (c mod p, ref (naive_addi (p, C.Nil, C.Nil, c/p)))
  | C.Cons(h, t), C.Nil | C.Nil, C.Cons(h, t) ->
      naive_addi (p, C.Cons(h, t), C.Cons(0, ref C.Nil), c)
  | C.Cons(hi, ti), C.Cons(hj, tj) -> let res = hi + hj + c in
      C.Cons ( res mod p, ref (naive_addi (p, !ti, !tj, res / p) ) )
let add (p1, i1, j1) (p2, i2, j2) =
  if p1 != p2 then
    failwith "Cannot add p-adic numbers in two different bases"
  else
    let p = p1 in
    let rec add_float_part j1 j2 = match j1, j2 with
      | [], [] -> [], 0
      | h :: t, [] | [], h :: t -> h :: t, 0
      | h1 :: t1, h2 :: t2 ->
        let t, c = add_float_part t1 t2 in
        let d = h1 + h2 + c in
        (d mod p) :: t, d/p in
    let j, c = add_float_part j1 j2 in
    let i = Corecursive_Addi.main (p, i1, i2, c) in
    (p, i, j)

let (*co*)rec naive_mult1 (p, i, j, c) = match i with
  | C.Nil -> if c = 0 then C.Nil 
             else C.Cons (c mod p, ref (naive_mult1 (p, C.Nil, j, c/p)))
  | C.Cons(hi, ti) -> let res = hi * j + c in
      C.Cons (res mod p, ref (naive_mult1 (p, !ti, j, res / p)))
let (*co*)rec naive_multi (p, i, j, c) = match j with
  | C.Nil -> c
  | C.Cons(hj, tj) ->
    match (naive_addi (p, (naive_mult1 (p, i, hj, 0)), c, 0)) with
    | C.Nil -> C.Nil (* WRONG *)
    | C.Cons(hres, tres) -> C.Cons(hres, ref (naive_multi (p, i, !tj, !tres)))
let mult (p1, i1, j1) (p2, i2, j2) =
  if p1 != p2 then
    failwith "Cannot multiply p-adic numbers in two different bases"
  else
    let p = p1 in
    let l1 = List.length j1 and l2 = List.length j2 in
    (* remove the commas then add them again *)
    match shift l1 (p1, i1, j1), shift l2 (p2, i2, j2) with
     | (_, i3, [ ]), (_, i4, [ ]) ->
        shift (- (l1 + l2)) (p, multi p i3 i4, [ ])
     | _ -> assert false

let sub (p1, i1, j1) (p2, i2, j2) =
(* to substract n1 - n2, just do n1 + n2 * (-1) *)
  if p1 != p2 then
    failwith "Cannot substract p-adic numbers in two different bases"
  else
    add (p1, i1, j1) (mult (p2, i2, j2) (from_rational p2 (-1) 1))

let (*co*)rec naive_divzeros = function (* default None *)
  | C.Nil -> None
  | C.Cons(0, t) ->
    (match naive_divzeros !t with 
     | None -> None
     | Some(n, i) -> Some(n+1, i))
  | i -> Some(0, i)
let (*co*)rec naive_divi (p, i, j) = 
  let minus_one p = Corecursive_From_rational.main (p, -1, 1) in
  let minusi p i j = (* of p-adic integers *)
    naive_addi (p, i, multi p (minus_one p) j, 0) in
  match i, j with
  | _, C.Nil -> failwith "Cannot divide by zero"
  | C.Nil, _ -> C.Nil
  | C.Cons(hi, ti), C.Cons(hj, tj) ->
    let d = C.euclid p hi hj in
    match minusi p i (naive_mult1 (p, j, d, 0)) with
    | C.Nil -> C.Cons(d, ref (naive_divi (p, C.Nil, j)))
    | C.Cons(0, t) -> C.Cons(d, ref (naive_divi (p, !t, j)))
    | _ -> assert false
let div (p1, i1, j1) (p2, i2, j2) =
  if p1 != p2 then
    failwith "Cannot divide p-adic numbers in two different bases"
  else
    let p = p1 in
    let l1 = List.length j1 and l2 = List.length j2 in
    (* remove the commas then add them again *)
    match shift l1 (p1, i1, j1), shift l2 (p2, i2, j2) with
     | (_, i3, [ ]), (_, i4, [ ]) ->
        (match Corecursive_Divzeros.main i4 with
           | None -> failwith "Cannot divide by zero"
           | Some(l4, i5) -> 
               shift (- (l1 - l2) - l4) (p, divi p i3 i5, [ ]))
     | _ -> assert false