  type variable = string
  type equation = ( (variable * float) list * float )

  let rec sum_aux c1 eq1 c2 eq2 = 
    (* calculates c2 * eq2 - c1 * eq1 without the constants *)
    match eq1, eq2 with
     | [], [] -> []
     | (v1, f1) :: t1, (v2, f2) :: t2 when v1 = v2 ->
       let res2 = sum_aux c1 t1 c2 t2 in 
       let total = c2 *. f2 -. c1 *. f1 in
       if total = 0.0 then res2 
                      else (v1, total) :: res2
     | (v1, f1) :: t1, [] ->
         let res2 = sum_aux c1 t1 c2 eq2 in
         (v1, -. c1 *. f1) :: res2
     | (v1, f1) :: t1, (v2, _) :: _ when v1 < v2 ->
         let res2 = sum_aux c1 t1 c2 eq2 in
         (v1, -. c1 *. f1) :: res2
     | [], (v2, f2) :: t2 | _ :: _, (v2, f2) :: t2 ->
         let res2 = sum_aux c1 eq1 c2 t2 in
         (v2, c2 *. f2) :: res2

  let sum c1 eq1 c2 eq2 (* calculates c2 * eq2 - c1 * eq1 *) =
    let (e1, f1) = eq1 and (e2, f2) = eq2 in
    let res2 = sum_aux c1 e1 c2 e2 in
    (res2, c2 *. f2 -. c1 *. f1)

  let eliminate v f (eq1, f1) (eq2, f2) =
  (* eliminates v in eq2 using eq1, where (v,f) was in front of eq1 
     returns the changed eq2 *)
    try
      let c2 = List.assoc v eq2 in
      let res = sum f (eq2, f2) c2 ((v,f) :: eq1, f1) in
      try 
        let (eq, f) = res in
        let epsilon = List.assoc v eq in (* for floating point errors *)
        print_string ("epsilon = " ^ v ^ " " ^ (string_of_float epsilon) ^ "\n");
        (List.remove_assoc v eq, f)
      with Not_found -> res
    with
      Not_found -> (eq2, f2)

  let rec solve_aux (x:variable) (eqs:equation list) (isol: equation list) =
    match eqs with
      | [] -> isol
      | eq :: tl -> match eq with
          | [], 0.0 -> failwith "Infinite number of solutions" (* check *)
          | [], _   -> failwith "No solution"
          | (v, f) :: t, i -> 
              if (f = 0.0) then failwith "This should not be zero"
              else solve_aux x (List.map (eliminate v f (t, i)) tl) (eq :: isol)

  let rec compute_solutions (sols : (variable * float) list) 
      (isol : equation list) = match isol with
    | [] -> sols
    | ((v,f) :: eq, c) :: tl -> let val_v =
        (-. 1. /. f) *.
        (List.fold_left
           (fun acc (v, coeff) -> acc +. coeff *. (List.assoc v sols))
           c eq) in
        compute_solutions ((v, val_v) :: sols) tl
    | _ -> failwith "Impossible"

  let rec eliminate_doubles (* in a sorted list of pairs *) = function
   | [] -> []
   | [ p ] -> [ p ]
   | (v1, f1) :: (v2, f2) :: t when v1 = v2 ->
       let total = f1 +. f2 in
       if total = 0.0 
         then eliminate_doubles t
         else eliminate_doubles ((v1,total) :: t)
   | (v1, f1) :: (v2, f2) :: t ->
       (v1, f1) :: (eliminate_doubles ((v2, f2) :: t))

  let solve (x:variable) (eqs0:equation list) =
    (* each equation is supposed sorted *)
    let eqs1 = List.map 
      (fun eq ->
        List.sort (fun (v1, _) (v2, _) -> String.compare v1 v2) (fst eq), snd eq)
      eqs0 in
    (* and with no doubles and no zeroes *)
    let eqs = List.map (fun (eq, f) -> eliminate_doubles eq, f) eqs1 in
    let isol = solve_aux x eqs [] in
    let sols = compute_solutions [] isol in
    List.assoc x sols