(* Abstract interpretation for loops *)
(* Application to CFG with cycles    *)
(*************************************)

module Naive = struct
  type boolean = True | False | Var of string | And of boolean * boolean
  type command = Skip | Assign of string * boolean | Cons of command * command
               | If of boolean * command * command | While of boolean * command

  type aboolean = ATrue | AFalse | ATop
  type astore   = (string * aboolean) list

  let string_of_aboolean = function
    | ATrue -> "ATrue" | AFalse -> "AFalse" | ATop -> "ATop"
  let string_of_astore s = "[ " ^
    ( String.concat " ; " 
        (List.map (fun (a, b) -> a ^ " : " ^ (string_of_aboolean b)) s) ) ^ " ]"  
  let print_astore s = print_string (string_of_astore s); print_newline()
    
  let rec interpret_b (b:boolean) (s:astore) : aboolean = match b with
   | True -> ATrue | False -> AFalse
   | Var x -> List.assoc x s
   | And(b1, b2) -> match interpret_b b1 s, interpret_b b2 s with
      | AFalse, _ | _, AFalse -> AFalse
      | ATrue, ATrue -> ATrue
      | _ -> ATop

  let rec insert (a, b:'a * 'b) (l: ('a * 'b) list) = 
    (* list sorted from smallest to largest *)
    match l with
    | [] -> [a, b]
    | (c, d) :: t when a = c -> (a, b) :: t
    | (c, d) :: t when a < c -> (a, b) :: l
    | (c, d) :: t -> (c, d) :: (insert (a, b) t)

  let join b1 b2 = match b1, b2 with
   | ATrue, ATrue -> ATrue | AFalse, AFalse -> AFalse
   | _ -> ATop

  let rec join_env e1 e2 = match e1, e2 with
   | [], _ -> e2
   | _, [] -> e1
   | (x1, b1) :: t1, (x2, b2) :: t2 when x1 = x2 ->
       (x1, join b1 b2) :: (join_env t1 t2)
   | (x1, b1) :: t1, (x2, b2) :: t2 when x1 < x2 ->
       (x1, b1) :: (join_env t1 e2)
   |  (x1, b1) :: t1, (x2, b2) :: t2 ->
       (x2, b2) :: (join_env e1 t2)

  let rec interpret (c:command) (s:astore) : astore = match c with
   | Skip -> s | Cons(c1, c2) -> interpret c2 (interpret c1 s)
   | Assign(x, b) -> insert (x, interpret_b b s) s
   | If(b, c1, c2) -> ( match interpret_b b s with
      | ATrue -> interpret c1 s
      | AFalse -> interpret c2 s
      | ATop -> join_env (interpret c1 s) (interpret c2 s) )
   | While(b, c) -> failwith "While loop not allowed"
   (* Naive, corec part *)
   (* ( match interpret_b b s with
      | AFalse -> s
      | ATrue -> interpret (While(b, c)) (interpret c s)
      | ATop -> join_env s (interpret (While(b, c)) (interpret c s)) ) *)
end

(* module Ai = struct *)
  open Naive
  let b = Var "x"
  let c = Assign("x", False)

  type variable = string

  type 'b f = IFalse of astore | ITrue of 'b | ITop of astore * 'b
  let fh  (h: 'c * 'e -> 'a * 'e) 
      : 'c f * 'e -> 'a f * 'e = function
   | IFalse s, e -> IFalse s, e
   | ITrue b, e -> let b1, e1 = h(b, e) in ITrue b1, e1
   | ITop(s, b), e -> let b1, e1 = h(b, e) in ITop(s, b1), e1
  let fh_simple (h: 'c -> 'a) (c : 'c f) =
    fst (fh (fun (c0, ()) -> h c0, ()) (c, ()))

  type equation = variable * variable f
  type coalgebra = astore
  type algebra   = astore
 
  let equal = (=) (* not == this time! *)

  let gamma s = match interpret_b b s with
   | AFalse -> IFalse s
   | ATrue -> ITrue (interpret c s)
   | ATop -> ITop(s, interpret c s)

  let alpha = function
   | IFalse s -> s
   | ITrue s -> s
   | ITop(s1, s2) -> join_env s1 s2

  let string_of_equation = function
   | (v1, IFalse s) -> v1 ^ " = " ^ (string_of_astore s)
   | (v1, ITrue v2) -> v1 ^ " = " ^ v2
   | (v1, ITop(s, v2)) -> v1 ^ " = " ^ (string_of_astore s) ^ " U " ^ v2 

  let rec solve_aux (eqs:equation list) (guesses:(variable * algebra) list) =
    let rec eval_rhs : variable f -> algebra =
      fun e -> alpha (fh_simple (fun v0 -> List.assoc v0 guesses) e)
      (* function | I1 z -> VarSet.singleton z
         | I2(v1, v2) -> VarSet.union (List.assoc v1 guesses) 
         (List.assoc v2 guesses) *)
    in
    let new_guesses = List.map (fun (x, eq) -> (x, eval_rhs eq)) eqs in
    if List.fold_left2 (fun b (_, g) (_, ng) -> 
      (* tests if guesses and new_guesses are observationally the same *)
      b && (g = ng)) (* difference 1 with FV *)
        true guesses new_guesses 
      then guesses
      else solve_aux eqs new_guesses

  let solve (name:variable) (eqs:equation list) : algebra =
    List.assoc name
      (solve_aux eqs (List.map (fun (x, _) -> x, [ ]) eqs))
    (* difference 2 with FV, empty environment is smaller than any other *)
(* end *)
