/* 
 * Author: Ranveer Chandra
 * File: SSCH.c
 *
 * This file implements some functionality of SSCH.  To start with
 * we implement the queueing disciplines
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "api.h"
#include "partition.h"
#include "mac_802_11.h"
#include "ip.h"
#include "mpls_shim.h"
   
#define DEBUG_MAINTENANCE_BUFFER  0
#define DEBUG_ERROR              1

static clocktype sschSkewStat = 0;
static int sschNumSwitchesStat = 0;

//------------------------------------------------------------------------
// Functions to handle the SSCH retransmit buffer

//-------------------------------------------------------------------------
// FUNCTION: SSCHInitRexmtBuffer
// PURPOSE:  Initializing SSCH retransmit buffer, where packets will wait
//           for next hop confirmation
// ARGUMENTS: Pointer to Retransmit buffer
// RETURN:   None
//-------------------------------------------------------------------------

void SSCHInitRexmtBuffer(SSCHRexmtBuffer *rexmtBuffer)
{
    // Initialize message buffer
    rexmtBuffer->head = NULL;
    rexmtBuffer->sizeInPacket = 0;
}


//-------------------------------------------------------------------------
// FUNCTION: SSCHInsertRexmtBuffer
// PURPOSE:  Inserting a packet in the retransmit buffer
// ARGUMENTS: Pointer to Retransmit buffer, the packet pointer, next hop for
//            the packet
// RETURN:   None
//-------------------------------------------------------------------------

void SSCHInsertRexmtBuffer(
			   Node* node,
			   Message* msg,
			   NodeAddress nextHop,
			   SSCHRexmtBuffer* rexmtBuffer)
{
  SSCHRexmtBufferEntry* newEntry = NULL;
  SSCHRexmtBufferEntry* current = rexmtBuffer->head;
  SSCHRexmtBufferEntry* previous = NULL;
  // char* addrId = (char *) MEM_malloc(sizeof(NodeAddress) +
  //    sizeof(unsigned short));
  
  IpHeaderType* ipHdr = (IpHeaderType *) MESSAGE_ReturnPacket(msg);
  // unsigned short ipId = 0;
  
  // if the buffer exceeds silently drop first packet
  // if no buffer size is specified in bytes it will only check for
  // number of packet.
  if (rexmtBuffer->sizeInPacket == SSCH_REXMT_BUFFER_SIZE)
    {
      MESSAGE_Free(node, msg);
      return;
    }
  
  // Find Insertion point.  Insert after all address matches.
  // This is to maintain a sorted list in ascending order of the
  // destination address
  
  while ((current) && (current->nextHop <= nextHop))
    {
      previous = current;
      current = current->next;
    }
  
  // Allocate space for the new message
  newEntry = (SSCHRexmtBufferEntry *)
    MEM_malloc(sizeof(SSCHRexmtBufferEntry));
  
  // Store the allocate message along with the destination number and the
  // the time at which the packet has been inserted
  if(DEBUG_ERROR)
    //if (DEBUG_MAINTENANCE_BUFFER)
    {
      char timeStr[100];
      ctoa(getSimTime(node), timeStr);
      printf("Node %u Inserting packet in the rexmt buffer with: "
	     "nextHop: %u, timestamp %s\n", node->nodeId, nextHop, timeStr);
    }
  
  newEntry->nextHop = nextHop;
  newEntry->perSlotRetryCount = 0;
  newEntry->acrossSlotRetryCount = 0;
  newEntry->msgId = (unsigned short) ipHdr->ip_id;
  newEntry->msg = msg;
  newEntry->timeStamp = getSimTime(node);
  newEntry->next = current;
  
  // Increase the size of the buffer
  ++(rexmtBuffer->sizeInPacket);
  
  // Here we are adding one message in the buffer. So set the timeout
  // of it.
  // memcpy(addrId, &srcAddr, sizeof(NodeAddress));
  
  // ipId = ipHdr->ip_id;
  
  // memcpy(addrId + sizeof(NodeAddress), &ipId, sizeof(unsigned short));
  
  // The timer below is not necessary now. But can be required if
  // underlying MAC did not have any mechanism for acknowledgement
  
  // SSCHSetTimer(
  //     node,
  //     addrId,
  //     sizeof(NodeAddress) + sizeof(unsigned short),
  //     MSG_NETWORK_RexmtTimeout,
  //     SSCH_MAIN_THOLDOFF_TIME);
  
  //MEM_free(addrId);
  
  // Got the insertion point
  if (previous == NULL)
    {
      // The is the first message in the buffer
      rexmtBuffer->head = newEntry;
    }
  else
    {
      // This is an intermediate node in the list
      previous->next = newEntry;
    }
}


//-------------------------------------------------------------------------
// FUNCTION: SSCHDeleteRexmtBufferByNextHop
// PURPOSE:  Deleting all entries from the retransmit buffer which are sent
//           to a particular next hop
// ARGUMENTS: Pointer to Retransmit buffer, next hop address
// RETURN:   None
//-------------------------------------------------------------------------

void SSCHDeleteRexmtBufferByNextHop(
				    Node* node,
				    SSCHRexmtBuffer* rexmtBuffer,
				    NodeAddress nextHop)
{
  SSCHRexmtBufferEntry* current = rexmtBuffer->head;
  SSCHRexmtBufferEntry* prev = NULL;
  SSCHRexmtBufferEntry* toFree;
  
  while (current)
    {
      if (current->nextHop == nextHop)
        {
	  toFree = current;
	  
	  if (prev == NULL)
            {
	      rexmtBuffer->head = current->next;
            }
	  else
            {
	      prev->next = current->next;
            }
	  
	  if (DEBUG_MAINTENANCE_BUFFER)
	    {
	      printf("Deleted rexmt buffer entry with:\n"
		     "nextHop: %u\n"
		     "ipId: %u\n",
		     current->nextHop,
		     current->msgId);
	    }
	  current = current->next;
	  
	  --(rexmtBuffer->sizeInPacket);

	  MESSAGE_Free(node, toFree->msg);
	  MEM_free(toFree);
	  break;
        }
      else
        {
	  prev = current;
	  current = current->next;
        }
    }
}

//-------------------------------------------------------------------------
// FUNCTION: SSCHDeleteRexmtBufferByNextHop
// PURPOSE:  Deleting all entries from the retransmit buffer which are sent
//           to a particular next hop
// ARGUMENTS: Pointer to Retransmit buffer, next hop address
// RETURN:   None
//-------------------------------------------------------------------------

void SSCHDeleteAllRexmtBufferByNextHop(
				    Node* node,
				    SSCHRexmtBuffer* rexmtBuffer,
				    NodeAddress nextHop)
{
  SSCHRexmtBufferEntry* current = rexmtBuffer->head;
  SSCHRexmtBufferEntry* prev = NULL;
  SSCHRexmtBufferEntry* toFree;
  
  while (current)
    {
      if (current->nextHop == nextHop)
        {
	  toFree = current;
	  
	  if (prev == NULL)
            {
	      rexmtBuffer->head = current->next;
            }
	  else
            {
	      prev->next = current->next;
            }
	  
	  if (DEBUG_MAINTENANCE_BUFFER)
	    {
	      printf("Deleted rexmt buffer entry with:\n"
		     "nextHop: %u\n"
		     "ipId: %u\n",
		     current->nextHop,
		     current->msgId);
	    }
	  current = current->next;
	  
	  --(rexmtBuffer->sizeInPacket);

	  MESSAGE_Free(node, toFree->msg);
	  MEM_free(toFree);
	  //break;
        }
      else
        {
	  prev = current;
	  current = current->next;
        }
    }
}

//-------------------------------------------------------------------------
// FUNCTION: SSCHResetPerSlotCounter
// PURPOSE:  Reser the perSlotRetryCount to 0 on each channel switch
// ARGUMENTS: Pointer to Retransmit buffer
// RETURN:   None
//-------------------------------------------------------------------------

void SSCHResetPerSlotCounter(
			     Node* node,
			     SSCHRexmtBuffer* rexmtBuffer)
{
  SSCHRexmtBufferEntry* current = rexmtBuffer->head;
  SSCHRexmtBufferEntry* prev = NULL;
  
  if (DEBUG_ERROR)
    {
      //printf("Resetting the per slot counters on a switch\n");
    }
  while (current)
    {
      current->perSlotRetryCount = 0;
      current = current->next;
    }
}


//-------------------------------------------------------------------------
// FUNCTION: SSCHDeleteRexmtBufferByEntry
// PURPOSE:  Deleting a specific entry from the retransmit buffer
// ARGUMENTS: Pointer to Retransmit buffer, pointer to the entry
// RETURN:   None
//-------------------------------------------------------------------------

void SSCHDeleteRexmtBufferByEntry(
				  Node* node,
				  SSCHRexmtBuffer* rexmtBuffer,
				  SSCHRexmtBufferEntry* rexmtBufEntry)
{
  SSCHRexmtBufferEntry* current = rexmtBuffer->head;
  SSCHRexmtBufferEntry* prev = NULL;
  
  while (current && current != rexmtBufEntry)
    {
      prev = current;
      current = current->next;
    }
  
  ERROR_Assert(current,
	       "Deleting entry from RexmtBuffer that doesn't exist");
  
  if (DEBUG_ERROR)
    {
      char timeStr[100];
      ctoa(current->timeStamp, timeStr);
      
      printf("Deleted rexmt buffer entry with:\n"
	     "nextHop: %u, timeStamp %s\n", current->nextHop, timeStr);
    }
  
  if (prev == NULL)
    {
      rexmtBuffer->head = current->next;
    }
  else
    {
      prev->next = current->next;
    }
  --(rexmtBuffer->sizeInPacket);  
  
  MESSAGE_Free(node, current->msg);
  MEM_free(current);
}


//-------------------------------------------------------------------------
// FUNCTION: SSCHSearchRexmtBuffer
// PURPOSE:  Searching Retransmit buffer for a specific entry by
//           source address destination address and id of ip header
// ARGUMENTS: Pointer to Retransmit buffer, source address, destination
//            address and id of ip header
// RETURN:   None
//-------------------------------------------------------------------------

SSCHRexmtBufferEntry* SSCHSearchRexmtBuffer(
					    SSCHRexmtBuffer* rexmtBuffer,
					    NodeAddress nextHop)
{
  SSCHRexmtBufferEntry* current = rexmtBuffer->head;
  
  if (DEBUG_MAINTENANCE_BUFFER)
    {
      printf("Trying to find a packet with: "
	     "nextHop %u\n", nextHop);
    }
  
  while (current && current->nextHop <= nextHop)
    {
      if (DEBUG_MAINTENANCE_BUFFER)
        {
	  printf("Content of current packet:\n"
		 "\tnextHop %u"
		 "\tPSCount %u"
		 "\tASCount %u"
		 "\tsequence %u\n",
		 current->nextHop,
		 current->perSlotRetryCount,
		 current->acrossSlotRetryCount,
		 current->msg->sequenceNumber);
        }
      
      if (current->nextHop == nextHop)
        {
	  return current;
        }
      current = current->next;
    }
  return NULL;
}


//-------------------------------------------------------------------------
// FUNCTION: SSCHHasFailedDestinationAttemptInSlot
// PURPOSE:  Searching Retransmit buffer for a specific entry by
//           dest address to see if an attempt has failed in this slot
//           This is used to adjust the contention window in mac_80211.c
// ARGUMENTS: Mac data
// RETURN:   None
//-------------------------------------------------------------------------

BOOL SSCHHasFailedDestinationAttemptInSlot(
					   MacData802_11* M802)
{
  SSCHRexmtBuffer* rexmtBuffer = &(M802->sschRexmtBuffer);
  NodeAddress nextHop = M802->currentNextHopAddress;
  SSCHRexmtBufferEntry* current = rexmtBuffer->head;

  if (DEBUG_MAINTENANCE_BUFFER)
    {
      printf("Trying to find a packet with: "
	     "nextHop %u\n", nextHop);
    }

  while (current && current->nextHop <= nextHop)
    {
      if (DEBUG_MAINTENANCE_BUFFER)
        {
	  printf("Content of current packet:\n"
		 "\tnextHop %u"
		 "\tPSCount %u"
		 "\tASCount %u"
		 "\tsequence %u\n",
		 current->nextHop,
		 current->perSlotRetryCount,
		 current->acrossSlotRetryCount,
		 current->msg->sequenceNumber);
        }
      
      if (current->nextHop == nextHop)
        {
	  return (current->perSlotRetryCount > 0);
        }
      current = current->next;
    }
  return FALSE;
}


//-------------------------------------------------------------------------
// FUNCTION: SSCHSearchRexmtBufferWithMinRetryCount
// PURPOSE:  Searching Retransmit buffer for a specific entry 
//           that has been tried for the least number of times
// ARGUMENTS: Pointer to Retransmit buffer, source address, destination
//            address and id of ip header
// RETURN:   None
//-------------------------------------------------------------------------
SSCHRexmtBufferEntry* SSCHSearchRexmtBufferWithMinRetryCount(
							     Node *node,
							     MacData802_11 *M802,
							     SSCHRexmtBuffer* rexmtBuffer)
{
  SSCHRexmtBufferEntry* current = rexmtBuffer->head;
  SSCHRexmtBufferEntry* minRetryEntry = rexmtBuffer->head;
  int retryCount = current->perSlotRetryCount;
  NodeAddress nextHop = current->nextHop;
  clocktype minTimeStamp = current->timeStamp;
  clocktype minTimeBetweenRetries = 
    M802->sschSlotDuration/(M802->sschRetryCountPerSlot-1);
  
  if (DEBUG_ERROR)
    {
      char delayStr[100];
      ctoa(minTimeBetweenRetries, delayStr);
      printf("Searching buffer for min retry count, synched to %d, currentslot %d\n", 
	     M802->sschSyncInfo[M802->sschCurrentSlot-1],
	     M802->sschCurrentSlot-1);
    }
  while (current)
    {
      if (DEBUG_MAINTENANCE_BUFFER)
        {
	  printf("Content of current packet:\n"
		 "\tnextHop %u"
		 "\tPSCount %u"
		 "\tASCount %u"
		 "\tsequence %u\n",
		 current->nextHop,
		 current->perSlotRetryCount,
		 current->acrossSlotRetryCount,
		 current->msg->sequenceNumber);
        }
      
      if (current->nextHop != nextHop
	  || current->nextHop == ANY_DEST) {
	printf("Trying nextHop %d\n", current->nextHop);
	if (current->nextHop == M802->sschSyncInfo[M802->sschCurrentSlot-1]
	    //|| (M802->sschCurrentSlot == 0)
            || (getSimTime(node) - current->timeStamp) >= minTimeBetweenRetries) 
	{
	  printf("Further Trying nextHop %d\n", current->nextHop);
	  if ((current->nextHop == ANY_DEST 
	      && current->perSlotRetryCount == 0
	       && current->timeStamp < minTimeStamp)
	      || (current->nextHop != ANY_DEST 
		  && (current->perSlotRetryCount < retryCount
		      || current->perSlotRetryCount < M802->sschRetryCountPerSlot
		      || (current->perSlotRetryCount == retryCount 
			  && current->timeStamp < minTimeStamp))))
	    {
	      minRetryEntry = current;
	      if(current->nextHop != ANY_DEST) 
		retryCount = current->perSlotRetryCount;
	      minTimeStamp = current->timeStamp;
	    }
	}
	nextHop = current->nextHop;
      }
      current = current->next;
    }
  printf("Got packet for %d\n", minRetryEntry->nextHop);
  return minRetryEntry;
}

//-------------------------------------------------------------------------
// FUNCTION: SSCHGetOldestEntryFromRexmtBuffer
// PURPOSE:  Searching Retransmit buffer for the oldest packet
// ARGUMENTS: Pointer to Retransmit buffer
// RETURN:   None
//-------------------------------------------------------------------------

SSCHRexmtBufferEntry* SSCHGetOldestEntryFromRexmtBuffer(
							SSCHRexmtBuffer* rexmtBuffer)
{
  SSCHRexmtBufferEntry* current = rexmtBuffer->head;
  SSCHRexmtBufferEntry* minTimeStampEntry = rexmtBuffer->head;
  NodeAddress nextHop = ANY_DEST;
  clocktype minTimeStamp = current->timeStamp;
  
  if (DEBUG_ERROR)
    {
      printf("Searching buffer for entry with oldest timestamp\n");
    }
  while (current)
    {
      if (DEBUG_MAINTENANCE_BUFFER)
        {
	  printf("Content of current packet:\n"
		 "\tnextHop %u"
		 "\tPSCount %u"
		 "\tASCount %u"
		 "\tsequence %u\n",
		 current->nextHop,
		 current->perSlotRetryCount,
		 current->acrossSlotRetryCount,
		 current->msg->sequenceNumber);
        }
      
      if (current->nextHop != nextHop) 
	{
	  if (current->timeStamp < minTimeStamp)
	    {
	      minTimeStampEntry = current;
	      nextHop = current->nextHop;
	      minTimeStamp = current->timeStamp;
	    }
	  
	}	
      current = current->next;
    }
  return minTimeStampEntry;
}

void SSCHInitNbrInfo(MacData802_11 *M802)
{
  int i, j;

  for(i=0; i<MAX_NBRS; i++) {
    M802->nbrAddress[i] = ANY_DEST;
    M802->sschNbrBeaconTime[i] = (clocktype)0;
    for(j=0; j<SSCH_NUM_SLOTS; j++)
      M802->sschNbrSeeds[i][j] = -1;
    M802->numPacketsForAddress[i] = 0;
  }
}

BOOL SSCHExistsLessThanThreshPacketsToANbr(MacData802_11 *M802) 
{
  int i;
  for(i = 0; i < M802->numNbrs; i++) {
    if(M802->numPacketsForAddress[i] < SSCH_BUFFER_THRESH_PER_NBR) {
      return TRUE;
    }
  }  
  return FALSE;
}

void SSCHIncrementNumPacketsForAddress(
		    MacData802_11* M802,
		    NodeAddress nbrAddress,
		    BOOL toIncrement)
{
  int countSlot;
  int insertPos = 0;
  BOOL newEntry = TRUE;

  for(insertPos = 0; insertPos < M802->numNbrs; insertPos++) {
    if(M802->nbrAddress[insertPos] == nbrAddress) {
      newEntry = FALSE;
      break;
    }
  }
  assert(insertPos < MAX_NBRS);
  
  if(newEntry) {
    assert(toIncrement);
    assert(insertPos == M802->numNbrs);
    M802->numNbrs++;
    M802->nbrAddress[insertPos] = M802->currentNextHopAddress;
  }
  if(toIncrement)
    {
      M802->numPacketsForAddress[insertPos]++;
      printf("Increased numpackets for %d to %d, buffer size %d\n", 
	     M802->nbrAddress[insertPos],
	     M802->numPacketsForAddress[insertPos],
	     M802->sschRexmtBuffer.sizeInPacket);
    }
  else
    {
      M802->numPacketsForAddress[insertPos]--;
      printf("Decremented numpackets for %d to %d rexmt buffer %d\n", 
	     M802->nbrAddress[insertPos],
	     M802->numPacketsForAddress[insertPos],
	     M802->sschRexmtBuffer.sizeInPacket);
    }
  assert(M802->numPacketsForAddress[insertPos] >= 0);
}

void SSCHResetNumPacketsForAddress(
		    MacData802_11* M802,
		    NodeAddress nbrAddress,
		    BOOL toIncrement)
{
  int countSlot;
  int insertPos = 0;
  BOOL newEntry = TRUE;

  for(insertPos = 0; insertPos < M802->numNbrs; insertPos++) {
    if(M802->nbrAddress[insertPos] == nbrAddress) {
      newEntry = FALSE;
      break;
    }
  }
  assert(insertPos < MAX_NBRS);
  
  if(newEntry) {
    assert(toIncrement);
    assert(insertPos == M802->numNbrs);
    M802->numNbrs++;
    M802->nbrAddress[insertPos] = M802->currentNextHopAddress;
  }
  if(toIncrement)
    {
      assert(FALSE);
    }
  else
    {
      M802->numPacketsForAddress[insertPos] = 0;
      printf("Decremented numpackets for %d to %d rexmt buffer %d\n", 
	     M802->nbrAddress[insertPos],
	     M802->numPacketsForAddress[insertPos],
	     M802->sschRexmtBuffer.sizeInPacket);
    }
  assert(M802->numPacketsForAddress[insertPos] >= 0);
}
// End of Buffer Handling Routines
//---------------------------------------------------------------------------

//--------------------------------------------------------------------------
//  NAME:        MacHandleSSCHBeacon.
//  PURPOSE:     Handling routine for sending an SSCH Beacon.
//  PARAMETERS:  Node* node
//                  Pointer to node
//               MacData802_11* M802
//                  Pointer to 802.11 structure
//  RETURN:      None.
//  ASSUMPTION:  None.
//--------------------------------------------------------------------------
void MacHandleSSCHBeacon(
			 Node *node, 
			 MacData802_11 *M802)
{
  M802->sschBeaconIsDue = TRUE;
  // If there is a packet to be sent, we will give the beacon the highest priority
  // else attempt to go into difs or eifs
  if(M802->currentMessage == NULL) {
    if (M802->state == M802_11_S_IDLE) {
      if ((M802->useDvcs) &&
	  ((Mac802_11PhyStatus(node, M802) == PHY_IDLE) ||
	   (Mac802_11PhyStatus(node, M802) == PHY_SENSING)))
	{
	  Mac802_11SetPhySensingDirectionForNextOutgoingPacket(node, M802);
	}//if//
      
      if ((Mac802_11PhyStatus(node, M802) == PHY_IDLE) &&
	  (!Mac802_11WaitForNAV(node, M802)))
	{
	  char delayStr[100];
	  // Not holding for NAV and phy is idle,
	  // so no backoff.
	  ctoa(M802->BO, delayStr);
	  printf("Node %d, bo is %s\n", node->nodeId, delayStr);
	  ERROR_Assert(M802->BO == 0,
		       "MacHandleSSCHBeacon: "
		       "Backoff is non-zero in IDLE state.\n");
	}
      else
	{
	  char delayStr[100];
	  // Otherwise set backoff.
	  Mac802_11SetBackoffIfZero(node, M802);
	  ctoa(M802->BO, delayStr);
	  printf("Node %d, setting backoff of %s\n", node->nodeId, delayStr);
	}//if//
      
      Mac802_11AttemptToGoIntoWaitForDifsOrEifsState(node, M802);
      
    }//if//
  }
}

//--------------------------------------------------------------------------
//  NAME:        SSCHAdjustSeeds
//  PURPOSE:     Adjust the seeds of the node if it wants to follow a node
//               Currently, we make this decision in the parity slot
//  PARAMETERS:  Node* node
//                  Pointer to node
//               MacData802_11* M802
//                  Pointer to 802.11 structure
//  RETURN:      None.
//  ASSUMPTION:  None.
//--------------------------------------------------------------------------
void SSCHAdjustFutureSeeds(
			   Node *node,
			   MacData802_11 *M802)
{
  int countSlot;
  int insertPos = 0, maxPosition = -1, oldestPosition = -1;
  int maxPackets = 0;
  SSCHRexmtBufferEntry *rexmtBufferEntry = NULL;
  int maxSlot, oldSlot;

  
  for(insertPos = 0; insertPos < M802->numNbrs; insertPos++) {
    if(M802->numPacketsForAddress[insertPos] > maxPackets
       && M802->sschNbrBeaconTime[insertPos] > 0
       && (getSimTime(node) - M802->sschNbrBeaconTime[insertPos] < M802->sschNumChannels * M802->sschSlotDuration)
       && M802->sschNbrSeeds[insertPos][SSCH_PARITY_SLOT] > 0) {
      maxPackets = M802->numPacketsForAddress[insertPos];
      maxPosition = insertPos;
    }
  }
  
  printf("Node %d has %d packets for %d, rexmt size %d\n", node->nodeId, M802->numPacketsForAddress[maxPosition], M802->nbrAddress[maxPosition], M802->sschRexmtBuffer.sizeInPacket);
  
  printf("Seed %d\n", M802->sschNbrSeeds[maxPosition][SSCH_PARITY_SLOT]);

  if(M802->sschRexmtBuffer.sizeInPacket > 0)
    rexmtBufferEntry = SSCHGetOldestEntryFromRexmtBuffer(&M802->sschRexmtBuffer);

  if(rexmtBufferEntry != NULL) {
    for(insertPos = 0; insertPos < M802->numNbrs; insertPos++) {
      if(M802->nbrAddress[insertPos] == rexmtBufferEntry->nextHop
	 && M802->sschNbrBeaconTime[insertPos] > 0
	 && (getSimTime(node) - M802->sschNbrBeaconTime[insertPos] < M802->sschNumChannels * M802->sschSlotDuration)
	 && M802->sschNbrSeeds[insertPos][SSCH_PARITY_SLOT] > 0) {
	oldestPosition = insertPos;
	break;
      }
    }      
  }

  // Change seed to follow the node for whom you have the oldest packet
  if(oldestPosition != -1) {
    printf("Node %d has %d packets for %d, rexmt size %d\n", node->nodeId, M802->numPacketsForAddress[oldestPosition], M802->nbrAddress[oldestPosition], M802->sschRexmtBuffer.sizeInPacket);
    
    printf("Seed %d\n", M802->sschNbrSeeds[oldestPosition][SSCH_PARITY_SLOT]);

    oldSlot = SSCH_PARITY_SLOT;
    printf("Synching old to %d\n", M802->nbrAddress[oldestPosition]);
    M802->sschNextParitySeed = M802->sschNbrSeeds[oldestPosition][oldSlot];
    M802->sschNextParityChannel = M802->sschNbrChannel[oldestPosition][oldSlot];
    M802->sschNextParitySync = M802->nbrAddress[oldestPosition];
  } else  if(maxPosition != -1) {
    // Change seed to follow the node for whom you have max packets to send
    maxSlot = SSCH_PARITY_SLOT;
    printf("Synching max to %d\n", M802->nbrAddress[maxPosition]);
    M802->sschNextParitySeed = M802->sschNbrSeeds[maxPosition][maxSlot];
    M802->sschNextParityChannel = M802->sschNbrChannel[maxPosition][maxSlot];
    M802->sschNextParitySync = M802->nbrAddress[maxPosition];
  }


  printf("Node %d synched to: ", node->nodeId); 
  for(insertPos = 0; insertPos < SSCH_NUM_SLOTS; insertPos++) {
    printf(" %d", M802->sschSyncInfo[insertPos]);
  }  
  printf("\n");

  printf("Seeds: ", node->nodeId); 
  for(insertPos = 0; insertPos < SSCH_NUM_SLOTS; insertPos++) {
    printf(" %d", M802->sschSeed[insertPos]);
  }  
  printf("\n");

  printf("Channels: ", node->nodeId); 
  for(insertPos = 0; insertPos < SSCH_NUM_SLOTS; insertPos++) {
    printf(" %d", M802->sschChannel[insertPos]);
  }  
  printf("\n");
}		     

//--------------------------------------------------------------------------
//  NAME:        MacHandleSSCHSwitchChannel.
//  PURPOSE:     Handle switching of channel depending on state.
//  PARAMETERS:  Node* node
//                  Pointer to node
//               MacData802_11* M802
//                  Pointer to 802.11 structure
//  RETURN:      None.
//  ASSUMPTION:  None.
//--------------------------------------------------------------------------
void MacHandleSSCHSwitchChannel(
				Node *node, 
				MacData802_11 *M802)
{
  char statStr[100];
  
  if (DEBUG_ERROR)
    {
      if(M802->sschDelayedSwitch)
	printf("Delayed switch is true\n");
      else
	printf("Delayed switch is false\n");
    }
  M802->IsInExtendedIfsMode = FALSE;
  M802->sschDelayedSwitch = FALSE;
  M802->sschDelayedSwitchTimeout = FALSE;
  
  // The card should not switch when it is transmitting or receiving a packet
  if( Mac802_11PhyStatus(node, M802) == PHY_TRANSMITTING 
      //|| Mac802_11PhyStatus(node, M802) == PHY_RECEIVING
      ) {
    //printf("Phy state is %d, Mac state %d\n", Mac802_11PhyStatus(node, M802), M802->state);
    ERROR_ReportError("MacHandleSSCHSwitchChannel: "
		      "Unexpected state at PHY Layer.\n");
  }
  
  // Switch the channel
  if(M802->sschNumSlotSwitches > M802->sschNumChannels) {
    // Switch to the Parity Channel
    int newChannel = 0;
    //int countSlot;
    //for(countSlot=0; countSlot<SSCH_NUM_SLOTS; countSlot++)
    //  newChannel += M802->sschSeed[countSlot];

    // In the parity slot, switch to the parity seed
    // This ensures that nodes intersect at least once
    PHY_SSCHSwitchToGivenChannel(node, M802->sschSeed[SSCH_PARITY_SLOT], M802->myMacData->phyNumber);
    M802->sschNumSlotSwitches = 1;
    M802->sschCurrentSlot = 0;
  } else {
    int countNbr;

    // Adjust the channel number of all your neighbors
    for(countNbr = 0; countNbr < M802->numNbrs; countNbr++) {
      if(M802->sschNbrBeaconTime[countNbr] > 0)
	M802->sschNbrChannel[countNbr][M802->sschCurrentSlot] = 
	  (M802->sschNbrChannel[countNbr][M802->sschCurrentSlot] +
	   M802->sschNbrSeeds[countNbr][M802->sschCurrentSlot])%
	  M802->sschNumChannels;
    }

    PHY_SSCHSwitchChannel(node, M802->sschSeed[M802->sschCurrentSlot], M802->sschChannel[M802->sschCurrentSlot], M802->myMacData->phyNumber);
    PHY_GetTransmissionChannel(node, M802->myMacData->phyNumber, &M802->sschChannel[M802->sschCurrentSlot]);

    if(M802->sschChangeSeedInNextSlot) {
      M802->sschSeed[M802->sschCurrentSlot] = (pc_nrand(node->seed) % SSCH_NUM_SEEDS) + 1;	
      M802->sschChannel[M802->sschCurrentSlot] = (pc_nrand(node->seed) % M802->sschNumChannels);
      M802->sschChangeSeedInNextSlot = FALSE;
    }

    M802->sschNumPacketsReceivedInSlot[M802->sschCurrentSlot] = 0;
    M802->sschNumDistinctSendersInSlot[M802->sschCurrentSlot] = 0;
    M802->sschNumPacketsSentInSlot[M802->sschCurrentSlot] = 0;
    M802->sschNumPacketsSentSuccessfullyInSlot[M802->sschCurrentSlot] = 0;
    M802->sschCurrentSlot++;
    if(M802->sschCurrentSlot == SSCH_NUM_SLOTS) {
      M802->sschCurrentSlot = 0;
      M802->sschNumSlotSwitches++;
    }

  }

  SSCHResetPerSlotCounter(node, &M802->sschRexmtBuffer);
  
  M802->sschNumSwitches++;
  M802->sschLastSwitchTime = getSimTime(node);

  // These two lines are just for some stats
  //sschNumSwitchesStat++;
  //sschSkewStat = (sschSkewStat*(sschNumSwitchesStat-1) 
  //	  + (M802->sschNumSwitches*M802->sschSlotDuration - getSimTime(node)))/sschNumSwitchesStat;
  if((getSimTime(node) - (M802->sschNumSwitches-1)*M802->sschSlotDuration) > sschSkewStat) 
    sschSkewStat = getSimTime(node) - (M802->sschNumSwitches-1)*M802->sschSlotDuration;
  ctoa(sschSkewStat, statStr);
  printf("The skew is %s\n", statStr);
 

  Mac802_11StartTimerOfGivenType(node, M802, 
				 M802->sschNumSwitches*M802->sschSlotDuration 
				 - getSimTime(node), MSG_MAC_SSCHSwitchChannel);

  Mac802_11StartTimerOfGivenType(node, M802, 
		  		 M802->sschSlotDuration/8 + pc_nrand(node->seed) % (M802->sschSlotDuration/4) , 
				 MSG_MAC_SSCHBeacon);
  
  // Decide action based on current 802.11 state
  switch (M802->state) {
  case M802_11_S_IDLE:
    M802->BO = 0;
    if(M802->currentMessage != NULL)
      Mac802_11AttemptToGoIntoWaitForDifsOrEifsState(node, M802);
    break;
    
  case M802_11_S_WF_DIFS_OR_EIFS:
    //if (M802->BO == 0) {
    //    Mac802_11TransmitFrame(node, M802);
    //}
    //else {
    ERROR_ReportError("MacHandleSSCHSwitchChannel: "
		      "Should not be in difs/eifs state.\n");
    //printf("Backoff was %d ", M802->BO);
    //Mac802_11SetBackoffIfZero(node, M802);
    //Mac802_11ContinueBackoff(node, M802);
    //Mac802_11SetState(node, M802, M802_11_S_BO);
    //printf("and it is %d\n", M802->BO);
    //Mac802_11StartTimer(node, M802, M802->BO);
    //}//if//
    
    break;
    
  case M802_11_S_BO: 
    ERROR_Assert(M802->BO != 0,
		 "MacHandleSSCHSwitchChannel: Backoff should not be 0.\n");
    //printf("I am backed off by %d\n", M802->BO);
    if(M802->dataRateInfo == NULL) {
      M802->dataRateInfo =
	Mac802_11GetDataRateEntry(node, M802, M802->currentNextHopAddress);
    }
    M802->dataRateInfo->numAcksFailed++;
    M802->retxDueToAckDcf++;

  case M802_11_S_WFACK: 
    if (M802->state == M802_11_S_WFACK)
      printf("Phy state is %d\n", Mac802_11PhyStatus(node, M802));
    M802->BO = 0;
    M802->NAV = 0;
    M802->CW = M802->cwMin;
    M802->IsInExtendedIfsMode = TRUE;
    //Mac802_11AttemptToGoIntoWaitForDifsOrEifsState(node, M802);
    //Mac802_11TransmitFrame(node, M802);
    //Mac802_11Retransmit(node, M802);
    
    //break;
    
    if ((M802->dataRateInfo->numAcksFailed ==
	 M802_11_NUM_ACKS_FOR_RATE_DECREASE) ||
	(M802->dataRateInfo->numAcksFailed == 1 &&
	 M802->dataRateInfo->firstTxAtNewRate))
      {
	M802->dataRateInfo->dataRateType =
	  MAX(M802->dataRateInfo->dataRateType - 1,
	      M802->lowestDataRateType);
	
	M802->dataRateInfo->numAcksFailed = 0;
	M802->dataRateInfo->numAcksInSuccess = 0;
	M802->dataRateInfo->firstTxAtNewRate = FALSE;
	M802->dataRateInfo->dataRateTimer =
	  getSimTime(node) + M802->rateAdjustmentTime;
      }
    
    Mac802_11Retransmit(node, M802);
    
    break;
  
  case M802_11_S_NAV_RTS_CHECK_MODE:
  case M802_11_S_WFNAV:
    {
      M802->NAV = 0;
      if ((M802->currentMessage != NULL) ||
	  (!MAC_OutputQueueIsEmpty(node,
				   M802->myMacData->interfaceIndex)))
	{
	  Mac802_11AttemptToGoIntoWaitForDifsOrEifsState(node, M802);
	}
      else {
	M802->BO = 0;
	Mac802_11SetState(node, M802, M802_11_S_IDLE);
      }
      break;
    }
    
  case M802_11_S_WFCTS: {
    //ERROR_ReportError("MacHandleSSCHSwitchChannel: "
    //    "Should not be in difs/eifs state.\n");
    M802->dataRateInfo->numAcksFailed++;
    
    if ((M802->dataRateInfo->numAcksFailed ==
	 M802_11_NUM_ACKS_FOR_RATE_DECREASE) ||
	(M802->dataRateInfo->numAcksFailed == 1 &&
	 M802->dataRateInfo->firstTxAtNewRate))
      {
	M802->dataRateInfo->dataRateType =
	  MAX(M802->dataRateInfo->dataRateType - 1,
	      M802->lowestDataRateType);
	
	M802->dataRateInfo->numAcksFailed = 0;
	M802->dataRateInfo->numAcksInSuccess = 0;
	M802->dataRateInfo->firstTxAtNewRate = FALSE;
	M802->dataRateInfo->dataRateTimer =
	  getSimTime(node) + M802->rateAdjustmentTime;
      }
    
    M802->retxDueToCtsDcf++;
    Mac802_11Retransmit(node, M802);
    
    break;
  }
  
  case M802_11_S_WFDATA: {
    Mac802_11CheckForOutgoingPacket(node, M802);
    break;
  }
  
  case M802_11_S_HEADER_CHECK_MODE: {
    const M802_11LongControlFrame* hdr =
      (const M802_11LongControlFrame*)
      MESSAGE_ReturnPacket(M802->potentialIncomingMessage);
    
    BOOL frameHeaderHadError;
    clocktype endSignalTime;
    
    BOOL packetForMe =
      ((hdr->destAddr == node->nodeId) ||
       (hdr->destAddr == ANY_DEST));
    
    ERROR_Assert((hdr->frameType != M802_11_CTS) &&
		 (hdr->frameType != M802_11_ACK),
		 "Frame type should not be CTS or ACK");
    
    PHY_TerminateCurrentReceive(
				node, M802->myMacData->phyNumber, packetForMe,
				&frameHeaderHadError, &endSignalTime);
    
    if (Mac802_11PhyStatus(node, M802) == PHY_RECEIVING) {
      // Keep Receiving (wait).
      Mac802_11SetState(node, M802, M802_11_S_IDLE);
    }
    else {
      M802->noOutgoingPacketsUntilTime = endSignalTime;
      
      if (DEBUG_ERROR) {
	if (packetForMe) {
	  printf("%d : %I64d For Me/Error\n",
		 node->nodeId, getSimTime(node));
	}
	else if (frameHeaderHadError) {
	  printf("%d : %I64d For Other/Error\n",
		 node->nodeId, getSimTime(node));
	}
	else if (M802->currentMessage != NULL) {
	  printf("%d : %I64d For Other/Have Message\n",
		 node->nodeId, getSimTime(node));
	}
	else {
	  printf("%d : %I64d For Other/No Message\n",
		 node->nodeId, getSimTime(node));
	}//if//
      }//if//
      
      if (!frameHeaderHadError) {
	clocktype headerCheckDelay =
	  PHY_GetFrameDuration(
			       node,
			       M802->myMacData->phyNumber,
			       PHY_GetRxDataRateType(
						     node, M802->myMacData->phyNumber),
			       M802_11_SHORT_CTRL_FRAME_SIZE);
	
	
	// Extended IFS mode if for backing off more when a
	// collision happens and a packet is garbled.
	// We are forcably terminating the packet and thus
	// we should not use extended IFS delays.
	
	M802->IsInExtendedIfsMode = FALSE;
	
	// It should be able to learn AOA, but temporarily disabled.
	// UpdateDirectionCacheWithCurrentSignal(
	//     node, M802, hdr->sourceAddr, TRUE);
	
	Mac802_11ProcessNotMyFrame(
				   node, M802,
				   (Mac802_11MicroToNanosecond(hdr->duration)
				    - headerCheckDelay),
				   (hdr->frameType == M802_11_RTS));
      }
      else {
	// Header had error.  Assume no information received.
      }//if//
      
      Mac802_11CheckForOutgoingPacket(node, M802);
    }//if//
    
    break;
  }
  
  case M802_11_CFP_START: 
  case M802_11_S_WF_CFP_BEACON: 
  default: {
    printf("MAC_802_11: Node %u got unknown state type %d",
	   node->nodeId, M802->state);
    
    ERROR_ReportError("MacHandleSSCHSwitchChannel: "
		      "Switch in unexpected state.\n");
  }
  }
}

//--------------------------------------------------------------------------
//  NAME:        MacHandleSSCHTransmitFailure.
//  PURPOSE:     Handle frame since CTS or ACK was never received.
//  PARAMETERS:  Node* node
//                  Pointer to node that is resetting its contention window
//               MacData802_11* M802
//                  Pointer to 802.11 structure
//  RETURN:      None.
//  ASSUMPTION:  None.
//--------------------------------------------------------------------------
void MacHandleSSCHTransmitFailure(
				  Node* node,
				  MacData802_11* M802)
{
  
  SSCHRexmtBufferEntry *rexmtBufferEntry = NULL;
  int insertPos = 0;
  
  rexmtBufferEntry = SSCHSearchRexmtBuffer(&M802->sschRexmtBuffer, M802->currentNextHopAddress);		
  printf("Searching for nextHop %d\n", M802->currentNextHopAddress); 
  
  if (DEBUG_ERROR)
    {
      printf("Handling a retransmit failure\n");
    }
  
  if(M802->currentMessage != NULL) {
    if(rexmtBufferEntry == NULL) {
      SSCHInsertRexmtBuffer(node, MESSAGE_Duplicate(node, M802->currentMessage), 
			    M802->ipNextHopAddr, &M802->sschRexmtBuffer);
      SSCHIncrementNumPacketsForAddress(M802, M802->ipNextHopAddr, TRUE);
      rexmtBufferEntry = SSCHSearchRexmtBuffer(&M802->sschRexmtBuffer, M802->ipNextHopAddr);		
    }
    
    ERROR_Assert(rexmtBufferEntry != NULL,
		 "MacHandleSSCHTransmitFailure: "
		 "RexmtBufferEntry should not be NULL.\n");
    
    if(rexmtBufferEntry->perSlotRetryCount == 0) {
      rexmtBufferEntry->acrossSlotRetryCount++;
    }    
    rexmtBufferEntry->perSlotRetryCount++;
    rexmtBufferEntry->timeStamp = getSimTime(node);
    printf("At %d, for %d increased perSlot to %d, acrossSlot %d\n", node->nodeId, rexmtBufferEntry->nextHop, rexmtBufferEntry->perSlotRetryCount, rexmtBufferEntry->acrossSlotRetryCount);
    
    //ERROR_Assert(M802->currentMessage != rexmtBufferEntry->msg,
    //	"MacHandleSSCHTransmitFailure: "
    //	"The current message is inconsistent with buffer entry.\n");
    
    if(rexmtBufferEntry->acrossSlotRetryCount > SSCH_NUM_SLOTS*(M802->sschNumChannels+1)
       || (rexmtBufferEntry->nextHop == ANY_DEST 
	   && rexmtBufferEntry->acrossSlotRetryCount > 5)
       //&& rexmtBufferEntry->perSlotRetryCount > M802->sschRetryCountPerSlot
       ) {
      // Drop the packet and indicate it to the upper layers
      M802->pktsDroppedDcf++;
      if (DEBUG_ERROR)
	{
	  printf("Dropping a packet from the retransmit buffer, acrossSlot %d\n", rexmtBufferEntry->acrossSlotRetryCount);
	}
      // Drop all packets
      // First find the position of this nbr
      for(insertPos = 0; insertPos < M802->numNbrs; insertPos++) {
	if(M802->nbrAddress[insertPos] == rexmtBufferEntry->nextHop) {
	  break;
	}
      }
      assert(insertPos != M802->numNbrs);

      // delete this nbr
      //while(M802->numPacketsForAddress[insertPos] > 0) {
      Mac802_11InformNetworkOfPktDrop(node, M802, M802->currentMessage);
      //Mac802_11InformNetworkOfPktDrop(node, M802, rexmtBufferEntry->msg);
      SSCHResetNumPacketsForAddress(M802, rexmtBufferEntry->nextHop, FALSE);
      SSCHDeleteAllRexmtBufferByNextHop(node, &M802->sschRexmtBuffer, rexmtBufferEntry->nextHop);
      //rexmtBufferEntry = SSCHSearchRexmtBuffer(&M802->sschRexmtBuffer, M802->nbrAddress[insertPos]);
      //if(rexmtBufferEntry == NULL) 
      //break;
      //}
    } 
  }

  M802->sschBeaconIsDue = FALSE;
  Mac802_11ResetCurrentMessageVariables(M802);
  Mac802_11SetState(node, M802, M802_11_S_IDLE);
  //Mac802_11DecreaseCW(node, M802);
  Mac802_11CheckForOutgoingPacket(node, M802);
}

//--------------------------------------------------------------------------
//  NAME:        SSCHMoveAPacketFromRexmtBufferToTheLocalBuffer.
//  PURPOSE:     Move a packet from the retransmit buffer to M802->currentmessage.
//  PARAMETERS:  Node* node
//                  Pointer to node that is resetting its contention window
//               MacData802_11* M802//                  Pointer to 802.11 structure
//  RETURN:      None.
//  ASSUMPTION:  None.
//--------------------------------------------------------------------------
void SSCHMoveAPacketFromRexmtBufferToTheLocalBuffer(
						    Node *node, 
						    MacData802_11 *M802)
{
  SSCHRexmtBufferEntry *rexmtBufferEntry = NULL;
  IpHeaderType* ipHeader;
  int interfaceIndex = M802->myMacData->interfaceIndex;
  
  rexmtBufferEntry = SSCHSearchRexmtBufferWithMinRetryCount(node, M802, &M802->sschRexmtBuffer);		

  ERROR_Assert(rexmtBufferEntry != NULL,
	       "SSCHMoveAPacketFromRexmtBufferToTheLocalBuffer: "
	       "RexmtBufferEntry should not be NULL.\n");
  
  if(rexmtBufferEntry->perSlotRetryCount > M802->sschRetryCountPerSlot) {
    printf("PS Retry Count %d\n", rexmtBufferEntry->perSlotRetryCount);
    return;
  }
  
  if(rexmtBufferEntry->nextHop == ANY_DEST
     && rexmtBufferEntry->perSlotRetryCount > 0) {
    printf("PS Retry Count for bcast %d\n", rexmtBufferEntry->perSlotRetryCount);
    return;
  }
  
  if (DEBUG_ERROR)
    {
      //printf("Moving a packet from the ssch rexmt buffer to M802\n");
    }
  M802->currentMessage = MESSAGE_Duplicate(node, rexmtBufferEntry->msg);
  M802->currentNextHopAddress = rexmtBufferEntry->nextHop;
  
  M802->pktsToSend++;
  
  // For a BSS station, change next hop address to be that of the AP
  M802->ipNextHopAddr = M802->currentNextHopAddress;
  
  // Extract the source/dest address from the packet
  // Note: Only IP4 and MPLS assumed here.
  if (M802->myMacData->mplsVar == NULL)  {
    // Treat as an IP message
    ipHeader =
      (IpHeaderType*)MESSAGE_ReturnPacket(M802->currentMessage);
  }
  else {
    // Treat this packet as an MPLS message
    char *aHeader;
    aHeader = (MESSAGE_ReturnPacket(M802->currentMessage)
	       + sizeof(Mpls_Shim_LabelStackEntry));
    ipHeader = (IpHeaderType*)aHeader;
  }
  M802->ipSourceAddr = ipHeader->ip_src;
  M802->ipDestAddr = ipHeader->ip_dst;
  
  // Ensure that network broadcast addresses are treated correctly
  if (M802->currentNextHopAddress ==
      NetworkIpGetInterfaceBroadcastAddress(node, interfaceIndex))
    {
      M802->currentNextHopAddress = ANY_DEST;
    }
  
  M802->dataRateInfo = Mac802_11GetDataRateEntry(node, M802, M802->currentNextHopAddress);
  
  ERROR_Assert(M802->dataRateInfo->ipAddress ==
	       M802->currentNextHopAddress,
	       "Address does not match");
  
  if ((M802->dataRateInfo->ipAddress != ANY_ADDRESS &&
       M802->dataRateInfo->dataRateTimer < getSimTime(node)) ||
      (M802->dataRateInfo->numAcksInSuccess ==
       M802_11_NUM_ACKS_FOR_RATE_INCREASE))
    {
      //if (M802->dataRateInfo->dataRateType != M802->highestDataRateType) {
      //    printf("%lld: node %d increase rate to %d to node %d\n",
      //           getSimTime(node), node->nodeId,
        //           M802->dataRateInfo->dataRateType + 1,
      //           M802->dataRateInfo->ipAddress);
      //}
      
      M802->dataRateInfo->dataRateType =
	MIN(M802->dataRateInfo->dataRateType + 1,
	    M802->highestDataRateType);
      
      M802->dataRateInfo->firstTxAtNewRate = TRUE;
      M802->dataRateInfo->numAcksFailed = 0;
      M802->dataRateInfo->numAcksInSuccess = 0;
      M802->dataRateInfo->dataRateTimer =
            getSimTime(node) + M802->rateAdjustmentTime;
    }
}

//--------------------------------------------------------------------------
//  NAME:        SSCHShouldIChangeMySeed.
//  PURPOSE:     Change seed in this slot under two conditions. If 
//               the node received a thresh number of packets in this slot
//               OR the node received a thresh number of packets in each slot
//               and received the min in my slot
//  PARAMETERS:  MacData802_11* M802
//                  Pointer to 802.11 structure
//  RETURN:      None.
//  ASSUMPTION:  None.
//--------------------------------------------------------------------------
BOOL SSCHShouldIChangeMySeed(MacData802_11* M802)
{
  int minReceivedPackets = M802->sschNumPacketsReceivedInSlot[M802->sschCurrentSlot];
  int i, count = 0;

  if(M802->sschNumPacketsReceivedInSlot[M802->sschCurrentSlot] < SSCH_SYNC_THRESH) 
    return TRUE;
  
  for(i=0; i<SSCH_NUM_SLOTS; i++) {
    printf("Slot %d, received %d\n", i, M802->sschNumPacketsReceivedInSlot[i]);
    if(M802->sschNumPacketsReceivedInSlot[i] < minReceivedPackets
       && i != SSCH_PARITY_SLOT) {
      count ++;
    }
  }
  if (count > 1) {
    printf("Cannot change\n");
    return FALSE;
  }

  return TRUE;

} 

int SSCHFindNbrsWithSameSeedAsMe(MacData802_11* M802)
{
  int countNbr;
  int numNbrsWithSameSeed = 0;
  // Adjust the channel number of all your neighbors
  for(countNbr = 0; countNbr < M802->numNbrs; countNbr++) {
    if(M802->sschNbrChannel[countNbr][M802->sschCurrentSlot] == M802->sschChannel[M802->sschCurrentSlot]
       && M802->sschNbrChannel[countNbr][M802->sschCurrentSlot] == M802->sschSeed[M802->sschCurrentSlot])
      numNbrsWithSameSeed++;
  }
  return numNbrsWithSameSeed;
}

int SSCHFindNbrsWithSameParitySeedAsMe(MacData802_11* M802)
{
  int countNbr;
  int numNbrsWithSameSeed = 0;
  // Adjust the channel number of all your neighbors
  for(countNbr = 0; countNbr < M802->numNbrs; countNbr++) {
    if(M802->sschNbrChannel[countNbr][M802->sschCurrentSlot] == M802->sschNextParityChannel
       && M802->sschNbrChannel[countNbr][M802->sschCurrentSlot] == M802->sschNextParitySeed)
      numNbrsWithSameSeed++;
  }
  return numNbrsWithSameSeed;
}

//--------------------------------------------------------------------------
//  NAME:        MacSSCHTransmitBeacon.
//  PURPOSE:     Initiate transmit of SSCH Beacon frame.
//  PARAMETERS:  Node* node that wants to send a data frame.
//                  Pointer to node
//               MacData802_11* M802
//                  Pointer to 802.11 structure
//  RETURN:      None.
//  ASSUMPTION:  None.
//--------------------------------------------------------------------------
void MacSSCHTransmitBeacon(
			   Node* node,
			   MacData802_11* M802)
{
  NodeAddress destAddr;
  int dataRateType;
  M802_11LongControlFrame* hdr;
  Message* pktToPhy;
  int i;
  BOOL sendCurrentInfo = TRUE;
  
  ERROR_Assert(Mac802_11PhyStatus(node, M802) == PHY_IDLE,
	       "MacSSCHTransmitBeacon: "
	       "Physical status is not IDLE.\n");
  
  ERROR_Assert(M802->sschBeaconIsDue,
	       "MacSSCHTransmitBeacon: "
	       "There is no beacon due.\n");

  M802->dataRateInfo =
    Mac802_11GetDataRateEntry(node, M802, ANY_DEST);

  PHY_SetTxDataRateType(node, M802->myMacData->phyNumber,
			M802->dataRateInfo->dataRateType);
  dataRateType = PHY_GetTxDataRateType(node, M802->myMacData->phyNumber);
  
  pktToPhy = MESSAGE_Alloc(node, 0, 0, 0);
  
  MESSAGE_PacketAlloc(node,
		      pktToPhy,
		      M802_11_LONG_CTRL_FRAME_SIZE,
		      TRACE_802_11);
  
  hdr = (M802_11LongControlFrame*)MESSAGE_ReturnPacket(pktToPhy);
  
  hdr->frameType = M802_11_SSCH_BEACON;
  
  hdr->sourceAddr = M802->selfAddr;
  hdr->destAddr = ANY_DEST;
  
  hdr->duration = getSimTime(node) - M802->sschNumSwitches*M802->sschSlotDuration; 

  // Sync to the node for whom you have the max packets to send. 
  // Do not sync, if either you are in a slot whose seed  will be modified in 
  // a parity slot, or someone else is synched with you on this slot, or 
  // this is the parity slot itself
  // NOTE: sschCurrentSlot pertains to the next slot number
  if(DEBUG_ERROR) {
    printf("Node %d, Taking Sync Decision, seed %d, channel %d, synched %d, slot %d\n", 
	   node->nodeId,
	   M802->sschSeed[M802->sschCurrentSlot],
	   M802->sschChannel[M802->sschCurrentSlot],
	   M802->sschSyncInfo[M802->sschCurrentSlot],
	   M802->sschCurrentSlot);

    printf("Attempted send %d packets, success %d\n", 
	   M802->sschNumPacketsSentInSlot[M802->sschCurrentSlot], 
	   M802->sschNumPacketsSentSuccessfullyInSlot[M802->sschCurrentSlot]);

    printf("Received %d packets\n", 
	   M802->sschNumPacketsReceivedInSlot[M802->sschCurrentSlot]);
  }

  if((M802->sschCurrentSlot) != SSCH_PARITY_SLOT 
     && M802->sschNumSlotSwitches <= M802->sschNumChannels) {
    int insertPos = 0, maxPosition = -1;
    int maxPackets = 0;
    
    if(M802->sschNumPacketsSentInSlot[M802->sschCurrentSlot] > 2 
       && M802->sschRexmtBuffer.sizeInPacket > 0
       && M802->sschNumPacketsSentSuccessfullyInSlot[M802->sschCurrentSlot] == 0) {
      printf("I have incorrect info for node %d\n", M802->sschSyncInfo[M802->sschCurrentSlot]);
      // Reset the seed info for the node
      for(insertPos = 0; insertPos < M802->numNbrs; insertPos++) {
	if(M802->nbrAddress[insertPos] == M802->sschSyncInfo[M802->sschCurrentSlot]) {
	  if(M802->sschSeed[M802->sschCurrentSlot] == M802->sschNbrSeeds[insertPos][M802->sschCurrentSlot] && 
	     M802->sschChannel[M802->sschCurrentSlot] == M802->sschNbrChannel[insertPos][M802->sschCurrentSlot]) {
	    M802->sschNbrSeeds[insertPos][M802->sschCurrentSlot] = -1;
	    M802->sschNbrChannel[insertPos][M802->sschCurrentSlot] = -1;
	    printf("Resetting for node %d is %d\n", M802->sschSyncInfo[M802->sschCurrentSlot], insertPos);
	  } else {
	    printf("Seed info got updated\n");
	  }
	}
      }      
      M802->sschSyncInfo[M802->sschCurrentSlot] = -1;
    }
    
    // If the number of packets received in this slot over the last iteration 
    // was less than a thresh, then change the seed immediately
    if(SSCHShouldIChangeMySeed(M802)) {
      
      SSCHRexmtBufferEntry *rexmtBufferEntry;

      // Find the oldest packet
      if(M802->sschRexmtBuffer.sizeInPacket > 0) {
	rexmtBufferEntry = SSCHGetOldestEntryFromRexmtBuffer(&M802->sschRexmtBuffer);
	
	if(rexmtBufferEntry != NULL) {
	  for(insertPos = 0; insertPos < M802->numNbrs; insertPos++) {
	    if(M802->nbrAddress[insertPos] == rexmtBufferEntry->nextHop
	       && M802->sschNbrBeaconTime[insertPos] > 0
	       && (getSimTime(node) - M802->sschNbrBeaconTime[insertPos] < M802->sschNumChannels * M802->sschSlotDuration)
	       && M802->sschNbrSeeds[insertPos][M802->sschCurrentSlot] > 0) {
	      maxPosition = insertPos;
	      printf("Found oldest entry %d\n", M802->nbrAddress[maxPosition]);
	      break;
	    }
	  }      
	}
	
	if(maxPosition == -1 || (pc_nrand(node->seed)%5) > 3) {	
	  // find the nbr for whom you have max packets to send
	  for(insertPos = 0; insertPos < M802->numNbrs; insertPos++) {
	    if(M802->numPacketsForAddress[insertPos] > maxPackets
	       && M802->sschNbrBeaconTime[insertPos] > 0
	       && M802->sschNbrSeeds[insertPos][M802->sschCurrentSlot] > 0) {
	      maxPackets = M802->numPacketsForAddress[insertPos];
	      maxPosition = insertPos;
	    }  
	  }
	}


	if(maxPosition != -1) {
	  printf("Node %d, maxpos %d syncing to Node %d, seed %d, channel %d\n",
		 node->nodeId, 
		 maxPosition, 
		 M802->nbrAddress[maxPosition], 
		 M802->sschNbrSeeds[maxPosition][M802->sschCurrentSlot],  
		 M802->sschNbrChannel[maxPosition][M802->sschCurrentSlot]);
	  
	  
	  // If this is a sender slot, but a node is following you, 
	  // then do not send the current information
	  if((M802->sschSyncInfo[M802->sschCurrentSlot] == M802->nbrAddress[maxPosition]
	      && M802->sschNumPacketsReceivedInSlot[M802->sschCurrentSlot] > SSCH_SYNC_THRESH)) {
	    printf("Already synched though\n");
	    sendCurrentInfo = FALSE;
	  }
	  
	  M802->sschSeed[M802->sschCurrentSlot] = M802->sschNbrSeeds[maxPosition][M802->sschCurrentSlot];
	  M802->sschChannel[M802->sschCurrentSlot] = M802->sschNbrChannel[maxPosition][M802->sschCurrentSlot];
	  M802->sschSyncInfo[M802->sschCurrentSlot] = M802->nbrAddress[maxPosition];
	} else if (M802->sschSyncInfo[M802->sschCurrentSlot] != -1) {
	  // Consider the case 1->2->3, and at node 2. Node 2 wants to send to 
	  // node 3, but without interference from node 1. So, we want node 1
	  // to not follow 2 here.
	  printf("Node %d, resetting sync info for %d\n", node->nodeId, M802->sschSyncInfo[M802->sschCurrentSlot]);
	  M802->sschSeed[M802->sschCurrentSlot] = (pc_nrand(node->seed) % SSCH_NUM_SEEDS) + 1;	
	  M802->sschChannel[M802->sschCurrentSlot] = (pc_nrand(node->seed) % M802->sschNumChannels);
	  M802->sschSyncInfo[M802->sschCurrentSlot] = -1;
	}
      }
    } else {
      if(DEBUG_ERROR) {
	printf("Node %d, This is a receiver slot\n", node->nodeId);
      }

      // If I am the receiver, then I will check to see how many other nodes
      // have the same seed as mine.  If 2 or more of non-senders have the
      // same seed, I will modify my seed and slot with probability 0.5

      if((SSCHFindNbrsWithSameSeedAsMe(M802) - M802->sschNumDistinctSendersInSlot[M802->sschCurrentSlot]) > 1) {
	 // && (pc_nrand(node->seed)%6) > 2) {
	M802->sschChangeSeedInNextSlot = TRUE;
	printf("Node %d, %d are synched to me, but sending %d, changing seed\n", 
	       node->nodeId, 
	       SSCHFindNbrsWithSameSeedAsMe(M802), 
	       M802->sschNumDistinctSendersInSlot[M802->sschCurrentSlot]);
      }
    } 
  }

  if(M802->sschCurrentSlot == SSCH_PARITY_SLOT 
     && M802->sschRexmtBuffer.sizeInPacket > 0) {
    if(M802->sschNumPacketsSentInSlot[M802->sschCurrentSlot] > 0 
       && M802->sschNumPacketsSentSuccessfullyInSlot[M802->sschCurrentSlot] == 0) {
      int insertPos = 0;
      printf("I have incorrect info for node %d\n", M802->sschSyncInfo[M802->sschCurrentSlot]);
      // Reset the seed info for the node
      for(insertPos = 0; insertPos < M802->numNbrs; insertPos++) {
	if(M802->nbrAddress[insertPos] == M802->sschSyncInfo[M802->sschCurrentSlot]) {
	  M802->sschNbrSeeds[insertPos][M802->sschCurrentSlot] = -1;
	  M802->sschNbrChannel[insertPos][M802->sschCurrentSlot] = -1;
	  printf("InsertPos for node %d is %d\n", M802->sschSyncInfo[M802->sschCurrentSlot], insertPos);
	}
      }      
      M802->sschSyncInfo[M802->sschCurrentSlot] = -1;
    }
    
    if((SSCHFindNbrsWithSameParitySeedAsMe(M802) - M802->sschNumDistinctSendersInSlot[M802->sschCurrentSlot]) > 1
       && M802->sschNumPacketsReceivedInSlot[M802->sschCurrentSlot] >= SSCH_SYNC_THRESH) {
      //       && (pc_nrand(node->seed)%6) > 2) {
      M802->sschNextParitySeed = (pc_nrand(node->seed) % SSCH_NUM_SEEDS) + 1;	
      M802->sschNextParityChannel = (pc_nrand(node->seed) % M802->sschNumChannels);
      printf("Readjusting parity seeds at node %d\n", node->nodeId);
    }
    SSCHAdjustFutureSeeds(node, M802);
  }

  if(M802->sschRexmtBuffer.sizeInPacket > 0
     && M802->sschNumSlotSwitches > M802->sschNumChannels) {
    M802->sschSeed[SSCH_PARITY_SLOT] = M802->sschNextParitySeed;
    M802->sschChannel[SSCH_PARITY_SLOT] = M802->sschNextParityChannel;
    M802->sschSyncInfo[SSCH_PARITY_SLOT] = M802->sschNextParitySync;
  }

  assert(SSCH_NUM_SLOTS <= 4);
  for(i=0; i<SSCH_NUM_SLOTS; i++) {
    int tempInt = 0;
    if(i != SSCH_PARITY_SLOT) {
      if(!sendCurrentInfo && i == M802->sschCurrentSlot) 
	tempInt = 0 + M802->sschChannel[i];
      else
	tempInt = (M802->sschSeed[i]<<4) + M802->sschChannel[i];
    }
    else 
      tempInt = (M802->sschNextParitySeed<<4) + M802->sschNextParityChannel;

    hdr->FCS[i] = (tempInt);
    //(unsigned char)tempInt;
    printf("Sending Seed %d [int %d] from %d is %d, channel %d\n", i, 
    tempInt, hdr->sourceAddr, M802->sschSeed[i], M802->sschChannel[i]);
    
  }
  
  Mac802_11SetState(node, M802, M802_11_X_SSCH_BEACON);
  
  if (!M802->useDvcs) {
    Mac802_11StartTransmittingPacket(
				     node, M802, pktToPhy, M802->delayUntilSignalAirborn);
  }
  else {
    BOOL isCached = FALSE;
    double directionToSend;
    
    AngleOfArrivalCache_GetAngle(
				 &M802->directionalInfo->angleOfArrivalCache,
				 destAddr,
				 getSimTime(node),
				 &isCached,
				 &directionToSend);
    
    if (isCached) {
      Mac802_11StartTransmittingPacketDirectionally(
						    node,
						    M802,
						    pktToPhy, M802->delayUntilSignalAirborn,
						    directionToSend);
      PHY_LockAntennaDirection(node, M802->myMacData->phyNumber);
    }
    else {
      // Don't know which way to send, send omni.
      
      Mac802_11StartTransmittingPacket(node, M802, pktToPhy,
				       M802->delayUntilSignalAirborn);
    }//if//
  }//if//
}


void SSCHAddNbrInfo(
		    Node *node,
		    MacData802_11* M802,
		    NodeAddress nbrAddr,
		    char sschInfo[4])
{
  int countSlot;
  int receivedSeed;
  int insertPos = 0;
  BOOL newEntry = TRUE;

  for(insertPos = 0; insertPos < M802->numNbrs; insertPos++) {
    if(M802->nbrAddress[insertPos] == nbrAddr) {
      newEntry = FALSE;
      break;
    }
  }
  assert(insertPos < MAX_NBRS);
  if(newEntry) {
    assert(insertPos == M802->numNbrs);
    M802->numNbrs++;
    M802->nbrAddress[insertPos] = nbrAddr;
  }

  M802->sschNbrBeaconTime[insertPos] = getSimTime(node);

  for(countSlot=0; countSlot<SSCH_NUM_SLOTS; countSlot++) {
    receivedSeed = 0;
    receivedSeed = (int)(sschInfo[countSlot]);
    if (receivedSeed < 0)
      receivedSeed += 256;
    if(receivedSeed > 0) {
      M802->sschNbrSeeds[insertPos][countSlot] = receivedSeed>>4;
      M802->sschNbrChannel[insertPos][countSlot] = 
	receivedSeed - (M802->sschNbrSeeds[insertPos][countSlot]<<4);
      if(M802->sschNbrSeeds[insertPos][countSlot] == 0) {
	//printf("At %d, Resetting Seeds for Node %d, countSlot %d\n", node->nodeId, nbrAddr, countSlot);
	M802->sschNbrSeeds[insertPos][countSlot] = -1;
	M802->sschNbrChannel[insertPos][countSlot] = -1;
	if (M802->sschSyncInfo[M802->sschCurrentSlot] == nbrAddr) {
	  // Consider the case 1->2->3, and at node 2. Node 2 wants to send to 
	  // node 3, but without interference from node 1. So, we want node 1
	  // to not follow 2 here.
	  printf("Node %d, resetting sync info for %d\n", node->nodeId, M802->sschSyncInfo[M802->sschCurrentSlot]);
	  M802->sschSeed[M802->sschCurrentSlot] = (pc_nrand(node->seed) % SSCH_NUM_SEEDS) + 1;	
	  M802->sschChannel[M802->sschCurrentSlot] = (pc_nrand(node->seed) % M802->sschNumChannels);
	  M802->sschSyncInfo[M802->sschCurrentSlot] = -1;
	}
      }
    }
    /*
    printf("Node %d, Received Seed %d from %d is %d, channel %d, synched to %d\n", 
	   node->nodeId, countSlot, 
	   nbrAddr, M802->sschNbrSeeds[insertPos][countSlot], 
	   M802->sschNbrChannel[insertPos][countSlot],
	   M802->sschSyncInfo[countSlot]);
    */
  } 
}


