-----------------------------------------------------------------------------
-- |
-- Module      :  Control.Concurrent.TxEvent.CML.Simple
-- Copyright   :  (c) Kevin Donnelly & Matthew Fluet 2006
-- License     :  BSD-style
-- Maintainer  :  mfluet@acm.org
-- Stability   :  experimental
-- Portability :  non-portable (requires TxEvent)
--
-- Concurrent ML library.
-- 
-- This library provides /first-class synchronous events/ in the style
-- of CML (<http://cml.cs.uchicago.edu/>).  This library implements a
-- simplified CML interface, eliding the @withNack@ combinator.
--
-----------------------------------------------------------------------------
module Control.Concurrent.TxEvent.CML.Simple (
  -- * TxEvent type
    Evt           -- abstract

  -- * TxEvent combinators
  , alwaysEvt     -- :: a -> Evt a
  , wrapEvt       -- :: Evt a -> (a -> IO b) -> Evt b
  , guardEvt      -- :: IO (Evt a) -> Evt a
  , neverEvt      -- :: Evt a
  , chooseEvt     -- :: Evt a -> Evt a -> Evt a
  , sync          -- :: Evt a -> IO a

  -- * Synchronous channels
  , SChan         -- abstract
  , newSChan      -- :: IO (SChan a)
  , sendEvt       -- :: SChan a -> a -> Evt ()
  , recvEvt       -- :: SChan a -> Evt a

  -- * Time delays
  , timeOutEvt    -- :: Int -> Evt
  ) where

import Prelude
import Control.Monad
import qualified Control.Concurrent.TxEvent as TxEvent

----------------------------------------------------------------------
----------------------------------------------------------------------

type Evt a = IO (TxEvent.Evt (IO a))

sync :: Evt a -> IO a
sync iei = do
  ei <- iei
  i <- TxEvent.sync ei
  i

lift :: TxEvent.Evt a -> Evt a
lift ev = return (fmap return ev)

alwaysEvt :: a -> Evt a
alwaysEvt x = lift (TxEvent.alwaysEvt x)

wrapEvt :: Evt a -> (a -> IO b) -> Evt b
wrapEvt iei f = fmap (fmap (>>= f)) iei

guardEvt :: IO (Evt a) -> Evt a
guardEvt iiei = join iiei

neverEvt :: Evt a
neverEvt = lift (TxEvent.neverEvt)

chooseEvt :: Evt a -> Evt a -> Evt a
chooseEvt iei1 iei2 = do
  ei1 <- iei1
  ei2 <- iei2
  return (ei1 `TxEvent.chooseEvt` ei2)

type SChan a = TxEvent.SChan a

newSChan :: IO (TxEvent.SChan a)
newSChan = TxEvent.sync (TxEvent.newSChan)

recvEvt :: TxEvent.SChan a -> Evt a
recvEvt ch = lift (TxEvent.recvEvt ch)

sendEvt :: TxEvent.SChan a -> a -> Evt ()
sendEvt ch x = lift (TxEvent.sendEvt ch x)

timeOutEvt :: Int -> Evt ()
timeOutEvt n = lift (TxEvent.timeOutEvt n)
