-----------------------------------------------------------------------------
-- |
-- Module      :  Control.Concurrent.TxEvent.CML.Full
-- Copyright   :  (c) Kevin Donnelly & Matthew Fluet 2006
-- License     :  BSD-style
-- Maintainer  :  mfluet@acm.org
-- Stability   :  experimental
-- Portability :  non-portable (requires TxEvent)
--
-- Concurrent ML library.
-- 
-- This library provides /first-class synchronous events/ in the style
-- of CML (<http://cml.cs.uchicago.edu/>).  This library implements the
-- full CML interface, including the @withNack@ combinator.
--
-----------------------------------------------------------------------------
module Control.Concurrent.TxEvent.CML.Full (
  -- * TxEvent type
    Evt           -- abstract

  -- * TxEvent combinators
  , alwaysEvt     -- :: a -> Evt a
  , wrapEvt       -- :: Evt a -> (a -> IO b) -> Evt b
  , guardEvt      -- :: IO (Evt a) -> Evt a
  , withNackEvt   -- :: (Evt () -> IO (Evt a)) -> Evt a
  , neverEvt      -- :: Evt a
  , chooseEvt     -- :: Evt a -> Evt a -> Evt a
  , sync          -- :: Evt a -> IO a

  -- * Synchronous channels
  , SChan         -- abstract
  , newSChan      -- :: IO (SChan a)
  , sendEvt       -- :: SChan a -> a -> Evt ()
  , recvEvt       -- :: SChan a -> Evt a

  -- * Time delays
  , timeOutEvt    -- :: Int -> Evt
  ) where

import Prelude
import Control.Monad
import qualified Control.Concurrent.TxEvent as TxEvent
import qualified Control.Concurrent.TxEvent.AckVar as AckVar

----------------------------------------------------------------------
----------------------------------------------------------------------

type Evt a = IO ([AckVar.AckVar], TxEvent.Evt ([AckVar.AckVar], IO a))

sync :: Evt a -> IO a
sync iei = do
  (_, ei) <- iei
  (acks, i) <- TxEvent.sync ei
  mapM_ AckVar.setAckVar acks
  i

lift :: TxEvent.Evt a -> Evt a
lift ev = return ([], fmap (\ x -> ([], return x)) ev)

alwaysEvt :: a -> Evt a
alwaysEvt x = lift (TxEvent.alwaysEvt x)

wrapEvt :: Evt a -> (a -> IO b) -> Evt b
wrapEvt iei f = 
    fmap (\ (acks, ei) -> (acks, fmap (\ (acks, i) -> (acks, i >>= f)) ei)) iei
  

withNackEvt :: (Evt () -> IO (Evt a)) -> Evt a
withNackEvt f = do
  ack <- AckVar.newAckVar
  (acks, ei) <- join (f (lift (AckVar.getAckVarEvt ack)))
  return (ack:acks, ei)

guardEvt :: IO (Evt a) -> Evt a
guardEvt iiei = withNackEvt (\ _ -> iiei)

neverEvt :: Evt a
neverEvt = lift (TxEvent.neverEvt)

chooseEvt :: Evt a -> Evt a -> Evt a
chooseEvt iei1 iei2 = do
  (acks1, ei1) <- iei1
  (acks2, ei2) <- iei2
  return (acks1 ++ acks2, 
          (fmap (\ (acks, i) -> (acks2 ++ acks, i)) ei1) 
          `TxEvent.chooseEvt` 
          (fmap (\ (acks, i) -> (acks1 ++ acks, i)) ei2))

type SChan a = TxEvent.SChan a

newSChan :: IO (TxEvent.SChan a)
newSChan = TxEvent.sync (TxEvent.newSChan)

recvEvt :: TxEvent.SChan a -> Evt a
recvEvt ch = lift (TxEvent.recvEvt ch)

sendEvt :: TxEvent.SChan a -> a -> Evt ()
sendEvt ch x = lift (TxEvent.sendEvt ch x)

timeOutEvt :: Int -> Evt ()
timeOutEvt n = lift (TxEvent.timeOutEvt n)
