-----------------------------------------------------------------------------
-- |
-- Module      :  Control.Concurrent.TxEvent.ASChan
-- Copyright   :  (c) Kevin Donnelly & Matthew Fluet 2006
-- License     :  BSD-style
-- Maintainer  :  mfluet@acm.org
-- Stability   :  experimental
-- Portability :  non-portable (requires TxEvent)
--
-- Asynchronous channels.
--
-- This library provides asynchronous (i.e., buffered) channels.
--
-----------------------------------------------------------------------------

module Control.Concurrent.TxEvent.ASChan (
  -- * Asynchronous channels
    ASChan         -- abstract
  , newASChan      -- :: IO (AckVar a)
  , recvASChanEvt  -- :: AckVar a -> Evt a
  , sendASChanEvt  -- :: AckVar a -> a -> Evt ()
  ) where

import Control.Concurrent
import Control.Concurrent.TxEvent

import qualified Data.Queue as Queue
import Data.Queue (Queue)

-- Misc
forkIO_ :: IO () -> IO ()
forkIO_ act = forkIO act >> return ()

----------------------------------------------------------------------
----------------------------------------------------------------------

newtype ASChan a = ASChan (SChan a, SChan a)
{- ^
The type of asynchronous channels.
-}

buffer :: SChan a -> SChan a -> IO ()
buffer inC outC = 
    let recv q = do 
          x <- recvEvt inC
          return (Queue.addToQueue q x) in
    let send q = 
          case Queue.deQueue q of
            Nothing -> neverEvt
            Just (x, q') -> do
              sendEvt outC x
              return q' in
    let loopIO q = do
          q' <- sync ((recv q) `chooseEvt` (send q))
          loopIO q' in
    loopIO Queue.emptyQueue

{-|
Create a new asynchronous channel.
-}
newASChan :: IO (ASChan a)
newASChan = do 
  inC <- sync $ newSChan
  outC <- sync $ newSChan
  forkIO_ $ buffer inC outC
  return (ASChan (inC, outC))

{-|
Receive a value on the channel.
-}
recvASChanEvt :: ASChan a -> Evt a
recvASChanEvt (ASChan (_, c)) = recvEvt c

{-|
Send a value on the channel.
-}
sendASChanEvt :: ASChan a -> a -> Evt ()
sendASChanEvt (ASChan (c, _)) x = sendEvt c x
