/* prototypes.h

   Prototypes for all functions shared between files.

 */
#ifndef __PROTOTYPES_H
#define __PROTOTYPES_H

#include "macros.h"
#include "dataStructures.h"

/* External routines ....................................... (external.c) */

/* These are routines that the user is expected to invoke. */

/* The user should call the macro GC_INIT first not this function. */
void gcInit(unsigned int initialHeapSize); 

/* Plug-in replacement for malloc. */
void *gcMalloc(int bytes); 

/* Plug-in replacement for malloc but assumes the object contains no
   pointers. */
void *gcMallocPtrFree(int bytes); 

/* Sets the global variable gcAllocPtr.  Invokes a collection if
   necessary, grows the heap, or finds more space. Also cleans up whatever
   remains in the heap. */
void gcAllocSpace();

/* Allocate an object with single header word. Returns a pointer to where the
   object should reside.(Header word is already in place. */
Word *gcAllocHeader(Word header);

/* Used to alloc large objects. Size > page.  User is responsible for
   putting in header words. The size of the object including header words is
   passed to the routine.*/
Word *gcAllocLarge(unsigned int words);


/* Heap routines ........................................... (heap.c) */

/* Allocate a new heap block. Assumes variables in gcInfo are consistent. */
void allocateHeapBlock(void);

/* Page routines ........................................... (page.c) */

/* Get a particular page.  We need this for example for blackListing. */
void getPage(PageInfo *pi);

/* Reserves a single page, and returns a pointer to it.  Sets the page to have
   header = (header | PAGE_RESERVED).  Preferentially choose preferredPi 
   if possible. */
PageInfo *reservePage(Word header, PageInfo *preferredPi);

/* Tries to reserve  n consecutive pages.  Intended for allocating large
   objects.  Returns NULL on failure.  Will grow the heap if necessary. */
PageInfo *reserveNPages(unsigned int n, Word header);

/* Frees a single page. */
void freePage(PageInfo *pi); 

/* Frees a circular doubly linked list of pages. */
void freeListOfPages(PageInfo *head);

/* Used to request a page for system use.  Marks the page as being a System
   page. Will grow the heap if necessary */
PageInfo *internalPageRequest();

/* Frees a page for system use.  */
void internalPageFree(PageInfo *pi);

/* Clears a pinned page of all untraversed objects, so that no misleading
   headers will confuse the pointerQuery routine. */
void fixupPage(PageInfo *pi);

/* Pins all the objects on the page that have not been forwarded.  Does not pin
   the page. */
void pinPagesObjects(PageInfo *pi);

/* Updates the bitmask on a page. */
void updatePageBitmask(PageInfo *pi);

/* Scans a page and calls processObject on each object found. */
void scanPage(PageInfo *pi);

PageInfo *pageToPI(Word *p);

/* Deque routines .......................................... (deque.c) */

/* Returns a pointer to a newQueue.  */
Deque *newDeque(void);

/* Enqueues the element on the q. */
void enqueue(Deque *q, Word element);

/* Dequeues the next element on the q. */
Word dequeue(Deque *q);

/* Pops an element off the end of the q. */
Word pop(Deque *q);

/* Frees all the pages associated with the q. */
void freeDeque(Deque *q);

/* Resets a queue as though it had never been traversed. */
void resetDeque(Deque *q);

/* Merges the second queue into the first. The victim is freed upon being
   devoured. */
void mergeDeque(Deque *qCannibal, Deque *qVictim);

/* Header routines ......................................... (header.c) */
/* static int objectSize(Word header); */ 
/* return the size of an object with the given header word.  
Size does NOT include the header word or words */

/* Conservative routines ................................... (conservative.c) */
Word* pointerQueryEarly(Word p); /* Returns NULL if p is not a valid pointer.
			       Otherwise returns a pointer to the object p
			       refers to. */

Word *pointerQueryLate(Word p); /* See the file for the distinction between
				   these two. */

void scanRoots(void); /* Scans the stack/static area pinning all objects found
			 and enqueuing them on gcInfo.explicitQueue and
			 gcInfo.pinnedObjects */

void freeBlackList(); /* Frees the list of BlackListed pages. */

void blackListScanRoots(void); /* Same as above but we blackList pages that
				  don't pan out. */

void pinChildren(Word *object); /* Pins all the children of C-object object */

/* ImplicitQueue ........................................... (implicitQueue.c) */
/* Copies object into the queue.  We pass the header word because it happens to
   be lying around at the only callsite of iqEnqueue. Returns pointer to the
   object new location.  NULL on failure. */
#ifndef INLINE_INNER_LOOP
Word *iqEnqueue(Word *object, Word header); 

Word *iqDequeue(Word **iqStart); /* Returns the address of the
						  next object on the queue.
						  NULL if there is no next
						  object. */
#endif

/* Object .................................................. (object.c) */

/* Enqueue the children of object marking them appropriately. */
void processObject(Word *object, Deque *explicitQueue);

/* Called by processObject.  Enqueues/Pins the children of cObject as
   appropriate. Uses a "late" pointerQuery to determine if something is a
   pointer. */
void processCObject(Word *cObject, Word header, Deque *explicitQueue);

/* Same as above but used an "early" pointerQuery.  This will be useful for
   generations when we treat children of C-Objects in older generations as
   roots. */
void processCObjectEarly(Word *cObject, Word header, Deque *explicitQueue);

/* Pins all the pages object resides on. */
void pinObjectPages(Word *object, Word header);

/* Collect ................................................. (collect.c) */

/* Performs the actual garbage collection. */
void collect(void);

/* Resets the collection boundaries. Takes the number of pages used, before a
   collection, and determines where the new limits should be. */
void resetCollectionBoundaries(unsigned int pages);
/* Machine Dependent ....................................... (machine.c) */

/* Set the static/global area boundary */
void setStaticBoundary(void);

/* Clears words worth of stack space to NULL. Returns arg unchanged. */
void clearStackWords(unsigned int words);

/* Does stack clearing appropriate to what it has seen. */
void clearStack(void);

/* Returns the desired size of the heap in words based on the environment
   default FAC_HEAP. */
unsigned int getHeapSizeFromEnv(void);

#endif
/* EOF: prototypes.h */


/* (c) Frederick Smith, Greg Morrisett.
 *     October 1998, all rights reserved.
 *
 *
 *              Copyright 1990-1993 Digital Equipment Corporation
 *                         All Rights Reserved
 *
 * Permission to use, copy, and modify this software and its documentation is
 * hereby granted only under the following terms and conditions.  Both the
 * above copyright notice and this permission notice must appear in all copies
 * of the software, derivative works or modified versions, and any portions
 * thereof, and both notices must appear in supporting documentation.
 *
 * Users of this software agree to the terms and conditions set forth herein,
 * and hereby grant back to Digital a non-exclusive, unrestricted, royalty-free
 * right and license under any changes, enhancements or extensions made to the
 * core functions of the software, including but not limited to those affording
 * compatibility with other hardware or software environments, but excluding
 * applications which incorporate this software.  Users further agree to use
 * their best efforts to return to Digital any such changes, enhancements or
 * extensions that they make and inform Digital of noteworthy uses of this
 * software.  Correspondence should be provided to Digital at:
 * 
 *                       Director of Licensing
 *                       Western Research Laboratory
 *                       Digital Equipment Corporation
 *                       250 University Avenue
 *                       Palo Alto, California  94301  
 * 
 * This software may be distributed (but not offered for sale or transferred
 * for compensation) to third parties, provided such third parties agree to
 * abide by the terms and conditions of this notice.  
 * 
 * THE SOFTWARE IS PROVIDED "AS IS" AND DIGITAL EQUIPMENT CORP. DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS.   IN NO EVENT SHALL DIGITAL EQUIPMENT
 * CORPORATION BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */
