/* machine.c

   Machine dependent routines.(In this case the SPARC)

*/
#include "dataStructures.h"
#include "machine.h"
#include <stdlib.h>
#include <signal.h>
#include <setjmp.h>	
#include <string.h>
#include <alloca.h>

typedef void (*handler)();
static jmp_buf faultBuffer;

void faultHandler(int sig)
{
  longjmp(faultBuffer,1);
}

/* For up!=0 find the address nearest upperBound that doesn't cause a memory
   fault. Otherwise find the address nearest lowerBound ... */
Word *locateBoundary(Word* lowerBound, Word* upperBound, int up)
{        
  static handler oldSIGSEGVHandler;
  static handler oldSIGBUSHandler;
  static Word *p;

  if(up)
    p = lowerBound;
  else
    p = upperBound;

  /* Setup the fault Handler */
  oldSIGSEGVHandler = signal(SIGSEGV,faultHandler);
  oldSIGBUSHandler = signal(SIGBUS,faultHandler);
  /* Done setting up the fault Handler */

  if(setjmp(faultBuffer)==0)
    {
      for(;;)
	{
	  if(up)
	    {
	      *p = *p;
	      if(p<upperBound)
		p++;
	      else
		return upperBound;	      
	    }
	  else
	    {
	      *p = *p;
	      if(p>lowerBound)
		p--;
	      else
		return lowerBound;
	    }
	}
    }

  /* reset the fault Handler */
  signal(SIGSEGV, oldSIGSEGVHandler);
  signal(SIGBUS,oldSIGBUSHandler);

  return (up ? p - 1 : p + 1);
}

void setStaticBoundary(void)
{
  Word *staticBase,*staticTop;
  /* staticBase < staticTop */

  if(&end<&etext)
    {
      staticBase = &end;
      staticTop = locateBoundary(&end,&etext,1);
      gcInfo.staticBoundary = staticTop;
    }
  else
    {
      staticBase = locateBoundary(&etext,&end,0);
      staticTop = &end;
      gcInfo.staticBoundary = staticBase;
    }
}

/* I found this confusing when I looked at it a second time.  Here is how it is 
   supposed to work:
                         current location (p)
                            |
   stack----------^---------v---------^-----------| --->increasing
                  |                   |
		  clearedStackTop(ct) clearedStackBottom(cb)

		  range to clear      cb               ct
    ct < p < cb    0                  -                -
   (p < ct)        p-ct + slop        -                p
   (p > cb)        clearance          p+clearance      p

The order of the less thans and what should be subtracted from what varies
depending on the way the stack is increasing, so the code gets pretty messy.  To
combat this I define the following macros
   */

#ifdef STACK_INCREASING
#define      ABOVE(P,T) ((P) < (T))
#define      BELOW(P,B) ((P) > (B))
#define      ABOVE_CLEARANCE(P,T) ((P) - (T) + STACK_SLOP)
#define      NEW_CB(P,C) (P+C)
#else
#define      ABOVE(P,CT) ((P) > (CT))
#define      BELOW(P,CB) ((P) < (CB))
#define      ABOVE_CLEARANCE(P,CT) ((CT) - (P) + STACK_SLOP)
#define      NEW_CB(P,C) (P - C)
#endif

void clearStack(void)
{
 Word clearance;
 Word *p;
 
 p = (Word *)&p;
 clearance = 0;

 if(BELOW(p,gcInfo.clearedStackBottom))
    {
      clearance = STACK_CLEARANCE;
      gcInfo.clearedStackTop = p;
      gcInfo.clearedStackBottom = NEW_CB(p,clearance);
    }

 if(ABOVE(p,gcInfo.clearedStackTop))
   {
     clearance = ABOVE_CLEARANCE(p,gcInfo.clearedStackTop);
     gcInfo.clearedStackTop = p;
   }

 gcInfo.clearStackCount = 0;
 if(clearance == 0) return;

 p=alloca(clearance * sizeof(Word));
 memset(p,0,clearance*sizeof(Word));
 p=NULL;
 clearance = 0;

}

void clearStackWords(unsigned int words)
{
 Word *p;

 p=alloca(words * sizeof(Word));
 memset(p,0,words*sizeof(Word));
 p=NULL;

}

unsigned int getHeapSizeFromEnv(void)
{
 char *desiredHeapSize;
 unsigned int heapSize;

 desiredHeapSize = getenv("FAC_HEAP"); /* FAC_HEAP should be an integer
					  representing the desired size of the
					  heap in BYTES. */
 if(desiredHeapSize)
   {
     heapSize = atoi(desiredHeapSize)/sizeof(Word);
   }
 else
   {
     heapSize = (0x1 << 20); /* Default heapSize is 4 Mb. */
   }

 return  heapSize;

}


/* EOF: machine.c */

/* (c) Frederick Smith, Greg Morrisett.
 *     October 1998, all rights reserved.
 *
 *
 *              Copyright 1990-1993 Digital Equipment Corporation
 *                         All Rights Reserved
 *
 * Permission to use, copy, and modify this software and its documentation is
 * hereby granted only under the following terms and conditions.  Both the
 * above copyright notice and this permission notice must appear in all copies
 * of the software, derivative works or modified versions, and any portions
 * thereof, and both notices must appear in supporting documentation.
 *
 * Users of this software agree to the terms and conditions set forth herein,
 * and hereby grant back to Digital a non-exclusive, unrestricted, royalty-free
 * right and license under any changes, enhancements or extensions made to the
 * core functions of the software, including but not limited to those affording
 * compatibility with other hardware or software environments, but excluding
 * applications which incorporate this software.  Users further agree to use
 * their best efforts to return to Digital any such changes, enhancements or
 * extensions that they make and inform Digital of noteworthy uses of this
 * software.  Correspondence should be provided to Digital at:
 * 
 *                       Director of Licensing
 *                       Western Research Laboratory
 *                       Digital Equipment Corporation
 *                       250 University Avenue
 *                       Palo Alto, California  94301  
 * 
 * This software may be distributed (but not offered for sale or transferred
 * for compensation) to third parties, provided such third parties agree to
 * abide by the terms and conditions of this notice.  
 * 
 * THE SOFTWARE IS PROVIDED "AS IS" AND DIGITAL EQUIPMENT CORP. DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS.   IN NO EVENT SHALL DIGITAL EQUIPMENT
 * CORPORATION BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */
