/* conservative.c

 */

#include "prototypes.h"
#include "machine.h"
#include "pageHeader.h"
#include "header.h"

#include <assert.h>
#include <setjmp.h>
#include <stdlib.h>

/* Functions ................................................................ */

#ifdef PAGE_BITMASKS
/* Returns the address of the header word associated with the pointer p, p must
   point onto this page somewhere. */
Word *headerOfPointer(PageInfo *pi,Word *p)
{
 HeapBlock *hb;
 Word *header;
 Word pageHeader;
 Word *pageBitmask;
 Word offset;
 Word bitOffset;
 Word wordOffset;
 int large;

 assert(0); /* This functions is most likely is completely broken. */

 large = 0;

 hb = (HeapBlock *)BLOCK(pi);
 while(PAGE_OFFSET_HEADER(pi->header) >= PAGE_SIZE)
   {
     large = 1;
     if((pi-1) < (PageInfo *)hb) return NULL;
     pi--;
   }

 pageHeader = pi->header;

 if(large)
   {
     p = PI_TO_PAGE(pi) + PAGE_SIZE - 1;
   }

 if((pageHeader & PAGE_BITMASK_UPDATED) == 0) updatePageBitmask(pi);
  assert((pi->header & PAGE_BITMASK_UPDATED) != 0);

/* At this point the bitmask is assumed to be valid. */
  offset = p - PI_TO_PAGE(pi);
  pageBitmask = pi->bitmask;

  wordOffset = BITMASK_WORD_OFFSET(offset);
  bitOffset  = BITMASK_BIT_OFFSET(offset);

/* If p points to a header word then it is not valid. */
  if((*(pageBitmask + wordOffset) & (0x1 << bitOffset)) != 0)
     return NULL;

  do {
       offset--;
       wordOffset = BITMASK_WORD_OFFSET(offset);
       bitOffset  = BITMASK_BIT_OFFSET(offset);
  }while(((*(pageBitmask + wordOffset) & (0x1 << bitOffset)) == 0));

  assert(offset>=0);

  return (PI_TO_PAGE(pi) + offset);

}
#else
/* Returns the address of the header word associated with the pointer p, p is
   guaranteed to point onto this page somewhere.  The page must be filled with
   zeroes to the end, after the last object on the page.  */
/* There are two versions of headerOfPointer.  An "early" and "late" version.
   See the comments around pointerQuery for details.  */
Word *headerOfPointerEarly(PageInfo *pi,Word *p)
{
 Word *header, *nextHeader;
 int size;

 header = PI_TO_PAGE(pi) + PAGE_OFFSET_HEADER(pi->header);

 if(header>p)
    {
      HeapBlock *hb;
      hb = (HeapBlock *)BLOCK(pi);
      while(PAGE_OFFSET_HEADER(pi->header) >= PAGE_SIZE)
	{
	  if((pi-1) < (PageInfo *)hb) return NULL;
	  pi--;
	}
      header = PI_TO_PAGE(pi) + PAGE_OFFSET_HEADER(pi->header);
    }


 assert(GC_EXTRACT_STATUS(*header)!=GC_MOVED || (*header==GC_EMPTY));

 size = GC_OBJECT_SIZE(*header);

 nextHeader = header+size+1;

 while(p>(nextHeader))
   {
    header = nextHeader; 

    assert(GC_EXTRACT_STATUS(*header)!=GC_MOVED || (*header==GC_EMPTY));
    size = GC_OBJECT_SIZE(*header);

    nextHeader += size+1;
   }

 if(p>header && !GC_JUNK(*header) && !GC_END_LARGE(*header)) 
   return header;
 else 
   return NULL;

 }

/* For this version, the object must be pinned to be valid.  But
   we allow forwarded objects. */
Word *headerOfPointerLate(PageInfo *pi,Word *p)
{
 Word *header, *nextHeader;
 Word indirectHeader;
 int size;

 header = PI_TO_PAGE(pi) + PAGE_OFFSET_HEADER(pi->header);

 if(header>p)
   {
     HeapBlock *hb;
     hb = (HeapBlock *)BLOCK(pi);
     while(PAGE_OFFSET_HEADER(pi->header) >= PAGE_SIZE)
       {
	 if((pi-1) < (PageInfo *)hb) return NULL;
	 pi--;
       }
     header = PI_TO_PAGE(pi) + PAGE_OFFSET_HEADER(pi->header);
   }

 if(GC_EXTRACT_STATUS(*header)!=GC_MOVED || (*header == GC_EMPTY))
   {
     size = GC_OBJECT_SIZE(*header);
   }
 else
   {
     indirectHeader = INDIRECT_HEADER(*header);
     size = GC_OBJECT_SIZE(indirectHeader); 
   }

 nextHeader = header+size+1;
 while(p>(nextHeader))
   {
    header = nextHeader; 

    if((GC_EXTRACT_STATUS(*header) != GC_MOVED) || (*header == GC_EMPTY))
      {
	size = GC_OBJECT_SIZE(*header);
      }
    else
      {
	indirectHeader = INDIRECT_HEADER(*header);
	size = GC_OBJECT_SIZE(indirectHeader);
      }

    nextHeader += size+1;
   }

 if(p>header && GC_EXTRACT_STATUS(*header)==GC_PINNED &&
    !GC_END_LARGE(*header) && !GC_JUNK(*header)) 
   return header;
 else
   return NULL;
 }

#endif

/* There are two versions of pointerQuery.  The "early" version assumes that
   garbage collection has not started yet so there are no objects that have been
   forwarded, and there are no pages in toSpace.

   The "late" versions assumes that all objects pointed to by valid conservative
   pointers have been pinned, but allows for objects being forwarded too. 

   Returns NULL if p is not a valid pointer.  Otherwise it returns a pointer to
   the object p refers to.
  */
Word* pointerQueryEarly(Word p)
{
 PageInfo *pi;
 Word *pp;
 Word *header;
 Word blockNumber;

 pp = (Word *)p;

 blockNumber = BLOCK_NUMBER(pp);

 if((gcInfo.blockBitmask[BITMASK_WORD_OFFSET(blockNumber)] & 
    (0x1 << BITMASK_BIT_OFFSET(blockNumber))) == 0)
   {
     return NULL; /* Not a block we own. */
   }
 
 pi = PAGE_TO_PI(pp);

 if((pi->header & PAGE_USER) == 0) return NULL;

 GEN(if(PAGE_AGED(pi->header)) return NULL);

 STATS(gcInfo.nDeviousRoots++);
 STATS(gcInfo.nFalseRoots--);

 header = headerOfPointerEarly(pi,pp);

 if(header==NULL) return NULL;

 if(*header == GC_EMPTY) /* empty header */
   {
     return NULL;
   }  

 return (header+1);
}

Word* pointerQueryLate(Word p)
{
 PageInfo *pi;
 Word *pp;
 Word *header;
 Word blockNumber;

 pp = (Word *)p;

 blockNumber = BLOCK_NUMBER(pp);

 if((gcInfo.blockBitmask[BITMASK_WORD_OFFSET(blockNumber)] & 
    (0x1 << BITMASK_BIT_OFFSET(blockNumber))) == 0)
   {
     return NULL; /* Not a block we own. */
   }
 
 pi = PAGE_TO_PI(pp);

 if((pi->header & PAGE_USER) == 0) return NULL;

 GEN(if(PAGE_AGED(pi->header)) return NULL);

 header = headerOfPointerLate(pi,pp);

 if(header==NULL) return NULL;

 if(*header == GC_EMPTY) /* empty header */
   {
     return NULL;
   }  

 return (header+1);
}

#ifdef PAGE_BLACKLISTING
inline void freeBlackList()
{
 freeListOfPages(&gcInfo.blackList);
}
#endif

#ifdef PAGE_BLACKLISTING
inline static void blackList(Word p)
{
 Word *pp = (Word *)p;
 Word blockNumber;
 PageInfo *pi;

 blockNumber = BLOCK_NUMBER(pp);

 if((gcInfo.blockBitmask[BITMASK_WORD_OFFSET(blockNumber)] & 
    (0x1 << BITMASK_BIT_OFFSET(blockNumber))) == 0)
   {
     return; /* Not a block we own. */
   }
 
 pi = PAGE_TO_PI(pp);

 if(pi->header == PAGE_FREE)
   {
     getPage(pi);
     pi->header = PAGE_BLACKLISTED | PAGE_RESERVED;
     gcInfo.nBlackListedPages++;
     LIST_INSERT_BEGIN(&gcInfo.blackList,pi);
   }
}
#endif

/* This macro processes a potential root off the stack or static area. */
#define BL_PROCESS_POTENTIAL_ROOT(X)                                 \
  if(SIMPLE_PTR_TEST(X))                                             \
    {                                                                \
      STATS(gcInfo.nFalseRoots++);                                   \
      object = pointerQueryEarly(X);                                 \
      if(object!=NULL)                                               \
	{  /* ! NULL */                                              \
	 if(!OBJECT_TRAVERSED(object))                               \
	   {  /* ! Traversed */                                      \
  	    STATS(if(!OBJECT_PINNED(object)) gcInfo.nPinnedObjects++); \
	    STATS(gcInfo.nTraversedObjects++);                       \
	    STATS(gcInfo.nRoots++);                                  \
            STATS(gcInfo.nDeviousRoots--);                           \
								     \
	    OBJECT_TRAVERSE(object);                                 \
	    enqueue(gcInfo.explicitQueue,(Word)object);              \
	                                                             \
	    pinObjectPages(object,OBJECT_HEADER(object));            \
	   }                                                         \
    STATS(else { gcInfo.nDuplicates++; });                           \
	}                                                            \
    BLACKLIST(else { blackList(X);} );                               \
    }                                                                

void scanRoots(void)
{
 Word *p;
 Word *limit;
 Word *object;
 SIMPLE_PTR_TEST_SETUP;

 assert(gcInfo.explicitQueue != NULL);

 for(p=STATIC_BASE;p<(Word *)&gcInfo;p++)
   {
     BL_PROCESS_POTENTIAL_ROOT(*p);
   }
 for(p=(Word *)((char *)&gcInfo + sizeof(gcInfo));p<STATIC_TOP;p++)
   {
     BL_PROCESS_POTENTIAL_ROOT(*p);
   }

#ifdef STACK_INCREASING
     p = gcInfo.stackBottom;
     limit = gcInfo.stackTop;
#else
     p = gcInfo.stackTop;
     limit = gcInfo.stackBottom;
#endif

 for(;p<limit;p++)
   {
     BL_PROCESS_POTENTIAL_ROOT(*p);
   } 
}

/* Pins all the children of C-object p. Do nothing else to them!!! */
void pinChildren(Word *object)
{
  Word header = OBJECT_HEADER(object);
  int size = GC_OBJECT_SIZE(header);
  Word *p, *pend;
  Word *pFound;
  SIMPLE_PTR_TEST_SETUP;

  assert(OBJECT_VALID(object));
  for(p=object, pend = object+size;p<pend;p++)
    {
      if(SIMPLE_PTR_TEST(*p))
	 {
	   pFound = pointerQueryEarly((Word)*p);

	   if(pFound != NULL)
	     {
	       STATS(if(!OBJECT_PINNED(pFound)) gcInfo.nPinnedObjects++);
	       OBJECT_PIN(pFound);
	     }
	 }
    }
 
}
/* EOF: conservative.c */

/* (c) Frederick Smith, Greg Morrisett.
 *     October 1998, all rights reserved.
 *
 *
 *              Copyright 1990-1993 Digital Equipment Corporation
 *                         All Rights Reserved
 *
 * Permission to use, copy, and modify this software and its documentation is
 * hereby granted only under the following terms and conditions.  Both the
 * above copyright notice and this permission notice must appear in all copies
 * of the software, derivative works or modified versions, and any portions
 * thereof, and both notices must appear in supporting documentation.
 *
 * Users of this software agree to the terms and conditions set forth herein,
 * and hereby grant back to Digital a non-exclusive, unrestricted, royalty-free
 * right and license under any changes, enhancements or extensions made to the
 * core functions of the software, including but not limited to those affording
 * compatibility with other hardware or software environments, but excluding
 * applications which incorporate this software.  Users further agree to use
 * their best efforts to return to Digital any such changes, enhancements or
 * extensions that they make and inform Digital of noteworthy uses of this
 * software.  Correspondence should be provided to Digital at:
 * 
 *                       Director of Licensing
 *                       Western Research Laboratory
 *                       Digital Equipment Corporation
 *                       250 University Avenue
 *                       Palo Alto, California  94301  
 * 
 * This software may be distributed (but not offered for sale or transferred
 * for compensation) to third parties, provided such third parties agree to
 * abide by the terms and conditions of this notice.  
 * 
 * THE SOFTWARE IS PROVIDED "AS IS" AND DIGITAL EQUIPMENT CORP. DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS.   IN NO EVENT SHALL DIGITAL EQUIPMENT
 * CORPORATION BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */
