package JavaGroups;


import java.util.*;

/**
   Elements are added at the tail and removed from the head. Class is thread-safe in that
   1 producer and 1 consumer may add/remove elements concurrently. The class is not
   explicitely designed for multiple producers or consumers. Implemented as a linked
   list, so that removal of an element at the head does not cause a right-shift of the
   remaining elements (as in a Vector-based implementation).  */
public class Queue {
    Element   head=null, tail=null;
    boolean   closed=false;
    int       size=0;
    Object    mutex=new Object();
    int       num_markers=0;


    class _EndMarker {}

    class Element {
	Object  obj=null;
	Element next=null;

	Element(Object o) {obj=o;}

	public String toString() {
	    return obj != null ? obj.toString() : "null";
	}
    }

    
    public Queue() {
    }



    public void Add(Object obj) throws QueueClosed {
	if(closed)
	    throw new QueueClosed();

	Element el=new Element(obj);
	synchronized(mutex) {
	    if(head == null) {
		head=el;
		tail=head;
		size=1;
	    }
	    else {
		tail.next=el;
		tail=el;
		size++;	    
	    }
	    mutex.notify();
	}	
    }





    public void AddAtHead(Object obj) throws QueueClosed {
	if(closed)
	    throw new QueueClosed();
	
	Element el=new Element(obj);
	synchronized(mutex) {
	    if(head == null) {
		head=el;
		tail=head;
		size=1;
	    }
	    else {
		el.next=head;
		head=el;
		size++;
	    }
	    mutex.notify();
	}
    }



    // Always called with mutex locked (we don't have to lock mutex ourselves)
    Object RemoveInternal() {
	Element retval=null;

	if(head == null)
	    return null;

	retval=head;
	if(head == tail) { // last element
	    head=null; tail=null;
	}
	else
	    head=head.next;
	size--;
	return retval.obj;
    }



    /** Removes 1 element from head or blocks until next element has been added */
    public Object Remove() throws QueueClosed {
	Object retval=null;

	synchronized(mutex) {
	    while(size == 0) {
		if(closed)
		    throw new QueueClosed();
		try {
		    mutex.wait();
		}
		catch(IllegalMonitorStateException ex) {
		    throw ex;
		}
		catch(Exception ex) {
		    System.err.println("Queue: " + ex);
		}
	    }
	    if(closed)
		throw new QueueClosed();

	    retval=RemoveInternal();
	}

	if(retval == null)
	    return null;
	
	if(retval instanceof _EndMarker) {
	    Close(false); // mark queue as closed
	    throw new QueueClosed();
	}
	
	return retval;
    }




    public Object Remove(long timeout) throws QueueClosed, Timeout {
	Object retval=null;

	synchronized(mutex) {
	    if(size == 0) {
		if(closed)
		    throw new QueueClosed();
		try {
		    mutex.wait(timeout);
		}
		catch(IllegalMonitorStateException ex) {
		    throw ex;
		}
		catch(IllegalArgumentException ex2) {
		    throw ex2;
		}
		catch(Exception e) {
		    System.err.println("Queue: " + e);
		}
	    }
	    if(closed)
		throw new QueueClosed();

	    retval=RemoveInternal();
	    if(retval == null) throw new Timeout();

	    if(retval instanceof _EndMarker) {
		Close(false);
		throw new QueueClosed();
	    }
	    return retval;
	}
    }



    public void RemoveElement(Object obj) throws QueueClosed {
	Element el, tmp_el;

	synchronized(mutex) {
	    el=head;	

	    if(el == null) return;  // no elements present

    
	    if(el.obj.equals(obj)) {
		head=el.next;
		el.next=null;
		if(size == 1)
		    tail=head;  // null
		size--;
		return;
	    }

	    while(el.next != null) {
		if(el.next.obj.equals(obj)) {
		    tmp_el=el.next;
		    el.next=el.next.next;  // point to the el past the next one. can be null.
		    tmp_el.next=null;
		    size--;
		    break;
		}
		el=el.next;
	    }
	}
    }




    public Object Peek() throws QueueClosed {
	Object retval=null;
	
	synchronized(mutex) {
	    while(size == 0) {
		if(closed)
		    throw new QueueClosed();
		try {
		    mutex.wait();
		}
		catch(IllegalMonitorStateException ex) {
		    throw ex;
		}
		catch(Exception ex) {
		    System.err.println("Queue: " + ex);
		}
	    }
	    if(closed)
		throw new QueueClosed();

	    retval=head != null? head.obj : null;
	}

	if(retval == null)
	    return null;
	
	if(retval instanceof _EndMarker) {
	    Close(false); // mark queue as closed
	    throw new QueueClosed();
	}
	
	return retval;
    }




    /** Doesn't remove element */
    public Object Peek(long timeout) throws QueueClosed, Timeout {
	Object retval=null;

	synchronized(mutex) {
	    if(size == 0) {
		if(closed)
		    throw new QueueClosed();
		try {
		    mutex.wait(timeout);
		}
		catch(IllegalMonitorStateException ex) {
		    throw ex;
		}
		catch(IllegalArgumentException ex2) {
		    throw ex2;
		}
		catch(Exception e) {
		    System.err.println("Queue: " + e);
		}
	    }
	    if(closed)
		throw new QueueClosed();

	    retval=head != null? head.obj : null;
	    if(retval == null) throw new Timeout();

	    if(retval instanceof _EndMarker) {
		Close(false);
		throw new QueueClosed();
	    }
	    return retval;
	}
    }



    /**
       Marks the queues as closed. When an <code>Add</code> or <code>Remove</code> operation is
       attempted on a closed queue, an exception is thrown.
       @param flush_entries When true, a end-of-entries marker is added to the end of the queue.
                            Entries may be added and removed, but when the end-of-entries marker
			    is encountered, the queue is marked as closed. This allows to flush
			    pending messages before closing the queue.
     */
    public void Close(boolean flush_entries) {
	if(flush_entries) {
	    try {
		Add(new _EndMarker()); // add an end-of-entries marker to the end of the queue
		num_markers++;
	    }
	    catch(QueueClosed closed) {}
	    return;
	}

	synchronized(mutex) {
	    closed=true;
	    try {
		mutex.notifyAll();
	    }
	    catch(Exception e) {
		System.err.println("Queue: " + e);
	    }
	}
    }



    public void Reset() {
	num_markers=0;
	if(!closed)
	    Close(false);

	synchronized(mutex) {
	    size=0;
	    head=null;
	    tail=null;
	}
	closed=false;
    }



    public int Size() {
	return size - num_markers;
    }

    public String toString() {
	return "Queue (" + Size() + ") messages";
    }



    public Vector GetContents() {
	Vector   retval=new Vector();
	Element  el;

	synchronized(mutex) {
	    el=head;
	    while(el != null) {
		retval.addElement(el.obj);
		el=el.next;
	    }
	}
	return retval;
    }



    public static void main(String[] args) {
	Queue q=new Queue();
	
	try {
	    q.Add("Bela");
	    q.Add("Janet");
	    System.out.println(q.GetContents());
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }




//      public static void main(String[] args) {
//  	Queue q=new Queue();
	
//  	try {

//  	    for(int i=0; i < 5; i++)
//  		q.Add("Element #" + i);

//  	    for(int i=0; i < 5; i++)
//  		System.out.println(q.Remove());

	    
//  	    q.Add(new Integer(1));
//  	    q.Add(new Integer(2));
//  	    q.Add(new Integer(3));
//  	    System.out.println(q); System.out.println(q.GetContents());
//  	    q.RemoveElement(new Integer(2));
//  	    System.out.println(q); System.out.println(q.GetContents());
	    
	    
//  	    q.Add("Bela");
//  	    q.Add("Janet");
//  	    System.out.println(q); System.out.println(q.GetContents());

//  	    q.RemoveElement("Bela");
//  	    System.out.println(q); System.out.println(q.GetContents());


//  	    for(int i=0; i < 5; i++)
//  		q.Add(new Integer(i));

//  	    System.out.println(q); System.out.println(q.GetContents());


//  	    System.out.println("Peeking: " + q.Peek());
//  	    System.out.println("Peeking: " + q.Peek());
//  	    System.out.println("Removing: " + q.Remove());
//  	    System.out.println("Peeking: " + q.Peek());
//  	    System.out.println("Peeking: " + q.Peek());



//  	    q.Close(true);
	    
//  	    try {
//  		while(true) {
//  		    System.out.println(q.Remove());
//  		    System.out.println(q);
//  		}
//  	    }
//  	    catch(QueueClosed closed) {
//  		System.out.println("The queue has been closed");
//  		return;
//  	    }

//  	}
//  	catch(Exception e) {
//  	    System.err.println(e);
//  	}

//      }

}
