package JavaGroups;

import java.io.*;
import java.util.*;


/**
 * Allows a client of <em>Channel</em> to be notified when messages have been received
 * instead of having to actively poll the channel for new messages. Typically used in the
 * client role (Receive()). As this class does not implement interface 
 * <code>Transport</code>, but <b>uses</b> it for receiving messages, an underlying object
 * has to be used to send messages (e.g. the channel on which an object of this class relies).
 */
public class PullPushAdapter implements Runnable {
    protected Transport           transport=null;
    protected MessageListener     listener=null;
    protected MembershipListener  membership_listener=null;
    protected Thread              receiver_thread=null;
    
    
    public PullPushAdapter(Transport transport) {
	this.transport=transport;
	Start();
    }
    
    public PullPushAdapter(Transport transport, MessageListener l) {
	this.transport=transport;
	SetListener(l);
	Start();
    }


    public PullPushAdapter(Transport transport, MembershipListener ml) {
	this.transport=transport;
	SetMembershipListener(ml);
	Start();
    }


    public PullPushAdapter(Transport transport, MessageListener l, MembershipListener ml) {
	this.transport=transport;
	SetListener(l);
	SetMembershipListener(ml);
	Start();
    }



    public void Start() {
	receiver_thread=new Thread(this, "PullPushAdapterThread");
	receiver_thread.start();
    }

    public void Stop() {
	if(receiver_thread != null) {
	    receiver_thread.stop();
	    receiver_thread=null;
	}
    }
    
    public void SetListener(MessageListener l) {
	listener=l;
    }


    public void SetMembershipListener(MembershipListener ml) {
	membership_listener=ml;
    }
    

    /**
     * Reentrant run(): message reception is serialized, then the listener is notified of the 
     * message reception
     */
    public void run() {
	Object           obj;
	MessageListener  l;

	while(true) {
	    try {
		obj=transport.Receive(0);
		if(obj == null)
		    continue;

		if(obj instanceof Message) {
		    if(listener != null)
			listener.Receive((Message)obj);
		}
		else if(obj instanceof GetStateEvent) {
		    if(listener != null) {
			if(transport instanceof Channel)
			    ((Channel)transport).ReturnState(listener.GetState());
			else {
			    System.err.println("PullPushAdapter.run(): underlying transport is " +
					       "not a Channel, but a " + 
					       transport.getClass().getName() + ": cannot return " +
					       "state using ReturnState() !");
			    continue;
			}
		    }
		}
		else if(obj instanceof SetStateEvent) {
		    if(listener != null) {
			try {
			    listener.SetState(((SetStateEvent)obj).GetArg());
			}
			catch(ClassCastException cast_ex) {
			    System.err.println("PullPushAdapter.run(): received SetStateEvent, " +
					       "but argument " + ((SetStateEvent)obj).GetArg() + 
					       " is not serializable ! Discarding message.");
			    continue;
			}
		    }
		}
		else if(obj instanceof View) {
		    if(membership_listener != null)
			membership_listener.ViewAccepted((View)obj);
		}
		else if(obj instanceof SuspectEvent) {
		    if(membership_listener != null)
			membership_listener.Suspect(((SuspectEvent)obj).GetMember());
		}
		else if(obj instanceof BlockEvent) {
		    if(membership_listener != null)
			membership_listener.Block();
		}
	    }
	    catch(ChannelNotConnected conn) {
		System.err.println(conn);
		break;
	    }
	    catch(Exception e) {
		System.err.println(e);
	    }
	}
    }


    



}
