package JavaGroups;

import java.net.*;
import java.util.Vector;
import java.util.Enumeration;
import JavaGroups.*;
import JavaGroups.JavaStack.Address;



// Todo: linear backoff in time for retransmissions


/**
   Keeps track of messages according to their sequence numbers. Allows messages to be added out of
   order, and with gaps between sequence numbers. Method <code>Remove</code> removes the first
   message with a sequence number that is 1 higher than <code>next_to_remove</code> (this variable
   is then incremented), or it returns null if no message is present, or if no message's sequence
   number is 1 higher.<p>
   Started out as a copy of SlidingWindow. Main diff: RetransmitCommand is different, and
   retransmission thread is only created upon detection of a gap. 

   @author Bela Ban May 27 1999
 */
public class NakReceiverWindow {


    
    /**
       Retransmit command (see Gamma et al.) used by the sliding window table to retrieve
       missing messages. It tries <em>twice</em> to retransmit by sending a pt2pt message to
       the sender of the message. If this does not furnish the message, a multicast message
       is sent to the group (<em>just once</em>). If this does still not furnish the message,
       the sender of the message is suspected, and subsequently removed, and its entry
       removed from the sliding window table.  
    */
    public interface RetransmitCommand {
	
	/**
	   Get the missing messages between sequence numbers <code>first_seqno</code> 
	   and <code>last_seqno</code> by sending a retransmit message to destination 
	   <code>dest</code>. <code>first_seqno</code> may be == <code>last_seqno</code>, but
	   <em>not</em> greater !
	   @param first_seqno The sequence number of the first missing message
	   @param last_seqno  The sequence number of the last missing message
	   @param sender The destination of the object to which the retransmit request is sent.
	*/
	void Retransmit(long first_seqno, long last_seqno, Object sender);
    }
    
    



    class Entry {	
	long      seqno=0;
	Message   msg;
	
	Entry(long seqno, Message msg) {
	    this.seqno=seqno;
	    this.msg=msg;
	}

	public Entry Copy() {
	    return new Entry(seqno, msg != null ? msg.Copy() : null);
	}

	public String toString() {
	    StringBuffer ret=new StringBuffer();
	    ret.append(seqno);
	    if(msg == null)
		ret.append("-");
	    else
		ret.append("+");
	    return ret.toString();
	}
    }



    /**
       Maintains a pool of messages that need to be retransmitted. Messages are aged and 
       retransmission requests sent according to age (linear backoff used).
     */
    class Retransmitter implements Runnable {
	Object              sender=null;
	RetransmitCommand   command=null;
	Thread              retransmitter=null;
	Vector              msgs=new Vector();


	class RetrEntry {
	    long seqno=0, timestamp=System.currentTimeMillis();

	    RetrEntry(long seqno) {this.seqno=seqno;}
	}


	public Retransmitter(Object sender, RetransmitCommand c) {
	    this.sender=sender;
	    command=c;
	}


	// Does not check for duplicates !
	void Add(long first_seqno, long last_seqno) {
	    RetrEntry entry;

	    if(first_seqno > last_seqno) {
		long tmp=first_seqno;
		first_seqno=last_seqno;
		last_seqno=tmp;
	    }

	    synchronized(msgs) {
		for(long i=first_seqno; i <= last_seqno; i++) {
		    entry=new RetrEntry(i);
		    msgs.addElement(entry);
		}
	    }	    
	}




	public void Remove(long seqno) {
	    RetrEntry entry;

	    synchronized(msgs) {
		for(int i=0; i < msgs.size(); i++) {
		    entry=(RetrEntry)msgs.elementAt(i);
		    if(entry.seqno == seqno) {
			msgs.removeElementAt(i);
			return;
		    }
		}
	    }
	}



	public void Reset() {
	    synchronized(msgs) {
		msgs.removeAllElements();
	    }
	}


	
	public void run() {
	    RetrEntry  e, next_entry;
	    long       current_time=0, lower_bound=0, upper_bound=0, prev_upper_bound=0;
	    Vector     retransmit_entries=new Vector();
	    
	    while(msgs.size() > 0) {
		retransmit_entries.removeAllElements();
		synchronized(msgs) {
		    for(int i=0; i < msgs.size(); i++) {
			e=(RetrEntry)msgs.elementAt(i);
			current_time=System.currentTimeMillis();
			if(current_time - e.timestamp > retransmit_period) {
			    retransmit_entries.addElement(e);
			    e.timestamp=current_time;
			}
		    }
		}

		// detect ranges for retransmission, e.g. [4-11]
		while(retransmit_entries.size() > 0) {
		    e=(RetrEntry)retransmit_entries.elementAt(0);
		    retransmit_entries.removeElementAt(0);
		    lower_bound=e.seqno; upper_bound=lower_bound;

		    while(retransmit_entries.size() > 0) {
			next_entry=(RetrEntry)retransmit_entries.elementAt(0);
			if(upper_bound +1 == next_entry.seqno) {
			    retransmit_entries.removeElementAt(0);
			    upper_bound=next_entry.seqno;
			}
			else
			    break;
		    }
		    command.Retransmit(lower_bound, upper_bound, sender);
		}

		
		synchronized(retransmitter) {
		    try {
			retransmitter.wait(retransmit_period);
		    }
		    catch(Exception ex) {
			System.err.println(ex);
		    }
		}
	    }
	}



	public void Notify() {
	    if(retransmitter == null) {
		retransmitter=new Thread(this, "NakReceiverWindow.RetransmitterThread");
		retransmitter.start();
	    }
	    else {
		if(retransmitter.isAlive()) {
		    synchronized(retransmitter) {
			retransmitter.notify();
		    }
		}
		else {
		    retransmitter.stop();
		    retransmitter=new Thread(this, "NakReceiverWindow.RetransmitterThread");
		    retransmitter.start();
		}
	    }
	}


	void Stop() {
	    if(retransmitter != null) {
		retransmitter.stop();
		retransmitter=null;
	    }
	}
	

    }









    Object              sender=null;
    long                head=0, tail=0;            // keep track of next seqno to remove and highest received
    List                msgs=new List();           // received msgs; list of Entries
    List                delivered_msgs=new List(); // msgs removed by Remove(); list of Entries
    RetransmitCommand   command=null;
    long                retransmit_period=3000;    // may be dynamically adjusted
    Retransmitter       retransmitter=null;



    public NakReceiverWindow(Object sender, RetransmitCommand command, long start_seqno) {
	this.sender=sender;
	this.command=command;
	head=start_seqno;
	tail=head;
	retransmitter=new Retransmitter(sender, command);
    }


    public void finalize() {retransmitter.Stop();}
    
    public void SetRetransmitPeriod(long t) {retransmit_period=t;}


    /**
       Adds a message according to its sequence number (ordered). Variables <code>head</code> and
       <code>tail</code> mark the start and end of the messages received, but not delivered yet.
       When a message is received, if its seqno is smaller than <code>head</code>, it is discarded
       (already received). If it is bigger than <code>tail</code>, we advance <code>tail</code> and 
       add empty elements. If it is between <code>head</code> and <code>tail</code>, we set the
       corresponding missing (or already present) element. If it is equal to <code>tail</code>,
       we advance the latter by 1 and add the message (default case).
     */
    public void Add(long seqno, Message msg) {
	Entry    current=null;
	boolean  inserted=false;
	long     old_tail=tail;
	
	if(seqno < head) {
	    //System.err.println("NakReceiverWindow.Add(): seqno " + seqno + " is smaller than " + 
	    //	       head + "); discarding message");
		return;
	}

	synchronized(msgs) {
	    if(seqno == tail) {
		msgs.Add(new Entry(seqno, msg));           // add at end (regular expected msg)
		tail++;
	    }
	    else if(seqno > tail) {                        // gap detected
		for(long i=tail; i < seqno; i++){          // add placeholders, creating gaps
		    msgs.Add(new Entry(i, null));
		    tail++;
		}
		msgs.Add(new Entry(seqno, msg));           // add real msg
		tail=seqno + 1;
		retransmitter.Add(old_tail, seqno-1);      // tell retransmitter to retrieve missing msgs
		retransmitter.Notify();
	    }
	    else if(seqno < tail) {                        // finally received missing message
		for(Enumeration en=msgs.Elements(); en.hasMoreElements();) {
		    current=(Entry)en.nextElement();
		    if(seqno == current.seqno) {
			if(current.msg == null) {
			    current.msg=msg;
			    // System.out.println("NakReceiverWindow: received missing msg #" + seqno);
			}
			retransmitter.Remove(seqno);       // no need to retransmit further
			return;
		    }
		}
	    }
	}
    }




    
    /**
       Find the entry with seqno <code>head</code>. If e.msg is != null -> return it and increment 
       <code>head</code>, else return null.
     */
    public Message Remove() {
	Entry    e;
	Message  retval=null;

	synchronized(msgs) {
	    e=(Entry)msgs.PeekAtHead();
	    if(e != null && e.msg != null) {
		retval=e.msg;
		msgs.RemoveFromHead();
		delivered_msgs.Add(e.Copy());
		head++;
		return retval;
	    }
	    return retval;
	}
    }

    
    /**
       Delete all messages <= seqno (they are stable, that is, have been received at all members).
       Stop when a number > seqno is encountered (all messages are ordered on seqnos).
    */
    public void Stable(long seqno) {
	Entry e;
	while((e=(Entry)delivered_msgs.PeekAtHead()) != null) {
	    if(e.seqno > seqno)
		break;
	    else
		delivered_msgs.RemoveFromHead();
	}
    }




    /** Deletes all entries */
    public void Reset() {
	retransmitter.Stop();
	retransmitter.Reset();
	msgs.RemoveAll();
	delivered_msgs.RemoveAll();
	head=tail=0;
    }




    /** Returns the highest sequence number of a message consumed by the application (by Remove()) */
    public long GetHighestDelivered() {
	return Math.max(head-1, -1);
    }


    /** Returns the highest sequence number received so far (which may be higher than the highest seqno
	<em>delivered</em> so far */
    public long GetHighestReceived() {
	return Math.max(tail-1, -1);
    }




    /** Return messages that are higher than <code>seqno</code> (excluding <code>seqno</code>) */
    public List GetMessagesHigherThan(long seqno) {
	List   retval=new List();
	Entry  entry;

	// check received messages
	for(Enumeration e=msgs.Elements(); e.hasMoreElements();) {
	    entry=(Entry)e.nextElement();
	    if(entry.seqno > seqno)
		retval.Add(entry.msg);
	}

	// check delivered messages (messages retrieved via Remove(), not *stable* messages !)
	for(Enumeration e=delivered_msgs.Elements(); e.hasMoreElements();) {
	    entry=(Entry)e.nextElement();
	    if(entry.seqno > seqno)
		retval.Add(entry.msg);
	}
	return retval;
    }




    /** 
	Return all messages m for which the following holds: m > lower && m <= upper (excluding lower,
	including upper). Check both <code>msgs</code> and <code>delivered_msgs</code>.
     */
    public List GetMessagesInRange(long lower, long upper) {
	List    retval=new List();
	Entry   entry;

	// check received messages
	for(Enumeration e=msgs.Elements(); e.hasMoreElements();) {
	    entry=(Entry)e.nextElement();
	    if(entry.seqno > lower && entry.seqno <= upper)
		retval.Add(entry.msg);
	}

	// check delivered messages (messages retrieved via Remove(), not *stable* messages !)
	for(Enumeration e=delivered_msgs.Elements(); e.hasMoreElements();) {
	    entry=(Entry)e.nextElement();
	    if(entry.seqno > lower && entry.seqno <= upper)
		retval.Add(entry.msg);
	}
	return retval.Size() == 0 ? null : retval;
    }



    public int    Size()     {return msgs.Size();}
    public String toString() {return msgs.toString();}






    



    public static void main(String[] args) {
	NakReceiverWindow win=new NakReceiverWindow(null, null, 1);
	Message           m=new Message();

	win.Add(1, m);
	win.Add(2, m);
	win.Add(3, m);
	win.Add(4, m);
	win.Add(6, m);
	win.Add(7, m);

	System.out.println("Highest seqno=" + win.GetHighestReceived());
	win.Add(5, m);
	System.out.println("Highest seqno=" + win.GetHighestReceived());



//  	class retr implements NakReceiverWindow.RetransmitCommand {
//  	    NakReceiverWindow w;

//  	    void SetWin(NakReceiverWindow w) {this.w=w;}


//  	    public void Retransmit(long first_seqno, long last_seqno, Object sender) {
//  		System.out.println("--> Retransmit [" + first_seqno + "," + last_seqno + "]");

//  		for(long i=first_seqno; i <= last_seqno; i++) {
//  		    w.Add(i, new Message());		    
//  		}


//  	    }
//  	}

//  	retr r=new retr();
//  	win=new NakReceiverWindow(null, r, 0);
//  	r.SetWin(win);
	

//  	for(int i=0; i < 10; i++) {

//  	    if(i == 3 || i == 7)
//  		continue;
	    
//  	    win.Add(i, m);
//  	}

//  	System.out.println(win + ", size is " + win.Size());
//  	Util.Sleep(5000);
//  	System.out.println(win + ", size is " + win.Size());

	
    }
    
}
