package JavaGroups;

import java.io.*;
import JavaGroups.JavaStack.*;

public class Message implements Externalizable {
    protected Object        dest_addr=null;
    protected Object        src_addr=null;
    protected byte[]        buf=null;
    protected Stack         headers=new Stack();

    

    /** Public constructor
     *  @param dest Address of receiver. If it is <em>null</em> or a <em>string</em>, then
     *              it is sent to the group (either to current group or to the group as given
     *              in the string). If it is a Vector, then it contains a number of addresses
     *              to which it must be sent. Otherwise, it contains a single destination.<p>
     *              Addresses are generally untyped (all are of type <em>Object</em>. A channel
     *              instance must know what types of addresses it expects and downcast
     *              accordingly.
     *  @param src  Address of sender
     *  @param buf  Message to be sent
     */
    public Message(Object dest, Object src, byte[] buf) {
	dest_addr=dest;
	src_addr=src;
	this.buf=buf;
    }



    /** Public constructor
     *  @param dest Address of receiver. If it is <em>null</em> or a <em>string</em>, then
     *              it is sent to the group (either to current group or to the group as given
     *              in the string). If it is a Vector, then it contains a number of addresses
     *              to which it must be sent. Otherwise, it contains a single destination.<p>
     *              Addresses are generally untyped (all are of type <em>Object</em>. A channel
     *              instance must know what types of addresses it expects and downcast
     *              accordingly.
     *  @param src  Address of sender
     *  @param obj  The object will be serialized into the byte buffer. <em>Object 
                    has to be serializable </em>!
     */
    public Message(Object dest, Object src, Object obj) {
	dest_addr=dest;
	src_addr=src;
	SetObject(obj);
    }





    // Only used for Externalization (creating an initial object)
    public Message() {}

    public Object  GetDest()                {return dest_addr;}
    public void    SetDest(Object new_dest) {dest_addr=new_dest;}
    public Object  GetSrc()                 {return src_addr;}
    public void    SetSrc(Object new_src)   {src_addr=new_src;}
    public byte[]  GetBuffer()              {return buf;}
    public void    SetBuffer(byte[] b)      {buf=b;}
    public Stack   GetHeaders()             {return headers;}
    public void    SetHeaders(Stack h)      {headers=h;}

    public void    SetObject(Object obj) {
	try {
	    buf=Util.ObjectToByteBuffer(obj);
	}
	catch(Exception e) {
	    System.err.println("Message.SetObject(): " + e);
	}
    }

    public Object  GetObject() {
	try {
	    return buf != null ? Util.ObjectFromByteBuffer(buf) : null;
	}
	catch(Exception e) {
	    System.err.println("Message.GetObject(): " + e);
	    return null;
	}
    }

    /*---------------------- Used by protocol layers ----------------------*/
    
    public void   AddHeader(Object hdr)  {
	headers.Push(new Header(hdr));
    }

    public Object RemoveHeader() {
	Header hdr=(Header)headers.Pop();
	return hdr != null? hdr.GetObject() : null;
    }

    public void RemoveHeaders() {headers.RemoveAll();}

    public Object PeekHeader() {
	Header hdr=(Header)headers.Peek();
	return hdr != null? hdr.GetObject() : null;
    }
    /*---------------------------------------------------------------------*/


    public Message Copy() {
	Message retval=new Message();
	retval.dest_addr=dest_addr;
	retval.src_addr=src_addr;
	retval.buf=buf;
	retval.headers=(Stack)headers.Copy();
	return retval;
    }


    public Message MakeReply() {
	return new Message(src_addr, null, null);
    }



    public String toString() {
	StringBuffer ret=new StringBuffer();
	ret.append("[Dst: ");
	if(dest_addr == null)
	    ret.append("<null>");
	else
	    ret.append(dest_addr);
	ret.append(", src: ");
	if(src_addr == null)
	    ret.append("<null>");
	else
	    ret.append(src_addr);

	if(!headers.Empty())
	    ret.append(" (" + headers.Size() + " headers)");

	ret.append(", size = ");
	if(buf != null && buf.length > 0)
	    ret.append(buf.length);
	else
	    ret.append("0");
	ret.append(" bytes");
	ret.append("]");
	return ret.toString();
    }


    /**
       Returns size of buffer plus size of all headers plus a fixed overhead (other fields of
       Message, marshaling code etc.
     */
    public long Size() {
	long              retval=buf != null ? buf.length : 0;
	java.util.Vector  contents=headers.GetContents();

	for(int i=0; i < contents.size(); i++)
	    retval+=((Header)contents.elementAt(i)).Size();
	return retval;
    }


    public String PrintObjectHeaders() {
	Object            obj;
	Header            hdr;
	StringBuffer      retval=new StringBuffer();
	java.util.Vector  contents=headers.GetContents();

	for(int i=0; i < contents.size(); i++) {
	    hdr=(Header)contents.elementAt(i);
	    if(hdr != null)
		retval.append("\n" + i + ": " + hdr.GetObject().toString());
	}
	return retval.toString();
    }



    public void writeExternal(ObjectOutput out) throws IOException {
 	out.writeObject(dest_addr);
 	out.writeObject(src_addr);
 	if(buf == null)
 	    out.writeInt(0);
 	else {
 	    out.writeInt(buf.length);
 	    out.write(buf);
 	}
  	out.writeObject(headers);
    }



    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException {
 	int tmp;

 	dest_addr=in.readObject();
 	src_addr=in.readObject();
 	tmp=in.readInt();
 	if(tmp != 0) {
 	    buf=new byte[tmp];
 	    in.readFully(buf);
 	}
  	headers=(Stack)in.readObject();
    }



//      public static void main(String[] args) {
//  	Message             m;
//  	FileOutputStream    outstream;
//  	ObjectOutputStream  os;

//  	try {
//  	    outstream=new FileOutputStream("messages.dat");
//  	    os=new ObjectOutputStream(outstream);
//  	    for(int i=0; i < 10; i++) {
//  		m=new Message(null, null, new String("Hello world" + i).getBytes());
//  		System.out.println("Size of message is " + m.Size());
//  		os.writeObject(m);
//  	    }
//  	}
//  	catch(Exception e) {
//  	    System.err.println(e);
//  	}	
//      }


    public static void main(String[] args) {
	Message m1, m2;
	byte[]  buf;

	try {
	    m1=new Message(null, null, "Bela Ban");
	    System.out.println(m1.GetObject());
	    buf=Util.ObjectToByteBuffer(m1);
	    m2=(Message)Util.ObjectFromByteBuffer(buf);
	    System.out.println(m2.GetObject());
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }



}
