package JavaGroups.JavaStack;

import java.io.*;
import java.net.*;
import JavaGroups.*;
import JavaGroups.JavaStack.Protocols.TunnelHeader;


public class RouterStub {
    String              router_host=null;       // name of the router host
    int                 router_port=0;          // port on which router listens on router_host
    Socket              sock=null;              // socket connecting to the router
    ObjectOutputStream  output=null;            // output stream associated with sock
    ObjectInputStream   input=null;             // input stream associated with sock
    Address             local_addr=null;        // address of group mbr. Once assigned, remains always the same
    final long          RECONNECT_TIMEOUT=5000; // milliseconds to wait until next connection retry attempt
    boolean             connected=false;





    /**
       Creates a stub for a remote Router object.
       @param router_host The name of the router's host
       @param router_port The router's port
     */
    public RouterStub(String router_host, int router_port) {
	this.router_host=router_host != null ? router_host : "localhost";
	this.router_port=router_port;
    }





    /**
       Establishes a connection to the router. The router will send my address (its peer address) back
       as an Address, which is subsequently returned to the caller. The reason for not using
       InetAddress.getLocalHost() or sock.getLocalAddress() is that this may not be permitted
       with certain security managers in case this code runs in an applet.
     */
    public Address Connect() {
	Address ret=null;

	try {
	    sock=new Socket(router_host, router_port);
	    sock.setSoLinger(true, 500);

	    // Retrieve our own address by reading it from the socket
	    input=new ObjectInputStream(sock.getInputStream());
	    ret=(Address)input.readObject();
	    output=new ObjectOutputStream(sock.getOutputStream());
	    connected=true;
	}
	catch(Exception e) {
	    connected=false;
	    System.err.println("RouterStub.Connect(): " + e);
	}

	// Address() uses InetAddress.getLocalHost() to find the host. May not be permitted in applets !
	if(ret == null && sock != null)
	    ret=new Address(sock.getLocalPort());

	// set local address; this is the one that we will use from now on !
	if(ret != null && local_addr == null)
	    local_addr=ret;

	return ret;
    }





    /** Closes the socket and the input and output streams associated with it */
    public void Disconnect() {
	if(output != null) {
	    try {
		output.close();
		output=null;
	    }
	    catch(Exception e) {}
	}

	if(input != null) {
	    try {
		input.close();
		input=null;
	    }
	    catch(Exception e) {}
	}

	if(sock != null) {
	    try {
		sock.close();
		sock=null;
	    }
	    catch(Exception e) {}
	}
	connected=false;
    }




    
    /**
       Register this process with the routerb under <code>groupname</code>.
       @param groupname The name of the group under which to register
       @return boolean False if connection down, true if registration successful.
     */
    public boolean Register(String groupname) {
	Request req=new Request(Request.REGISTER, groupname, local_addr);
	
	if(sock == null || output == null || input == null) {
	    System.err.println("RouterStub.Register(" + groupname + "): no connection to router !");
	    connected=false;
	    return false;
	}

	try {
	    output.writeObject(req);
	    output.flush();
	}
	catch(Exception e) {
	    System.err.println("RouterStub.Register(): " + e);
	    connected=false;
	    return false;
	}
	return true;
    }




    
    /**
       Retrieves the membership (list of Addresses) for a given group. This is mainly used by the PING
       protocol to obtain its initial membership. This is used infrequently, so don't maintain socket
       for the entire time, but create/delete it on demand.
     */
    public List Get(String groupname) {
	List                ret=null;
	Request             req=new Request(Request.GET, groupname);
	Socket              tmpsock;
	ObjectOutputStream  output;
	ObjectInputStream   input;

	try {
	    tmpsock=new Socket(router_host, router_port);
	    tmpsock.setSoLinger(true, 500);
	    input=new ObjectInputStream(tmpsock.getInputStream());
	    input.readObject();  // discard my own address (first thing returned by router when opening sock)
	    output=new ObjectOutputStream(tmpsock.getOutputStream());
	    output.writeObject(req);
	    ret=(List)input.readObject();
	    output.close();
	    input.close();
	    tmpsock.close();
	}
	catch(Exception e) {
	    System.err.println(e);
	}	
	return ret;
    }


    
    /** Sends a message to the router. Returns false if message cannot be sent (e.g. no connection to
	router, true otherwise. */
    public boolean Send(Message msg) {
	if(sock == null || output == null || input == null) {
	    System.err.println("RouterStub.Send(): no connection to router !");
	    connected=false;
	    return false;
	}
	try {
	    output.writeObject(msg);
	}
	catch(Exception e) {
	    System.err.println("RouterStub.Send(): " + e);
	    connected=false;
	    return false;
	}
	return true;
    }


    
    /** Receives a message from the router (blocking mode). If the connection is down, false is returned,
	otherwise true */
    public Message Receive() {
	Message ret=null;

	if(sock == null || output == null || input == null) {
	    System.err.println("RouterStub.Receive(): no connection to router !");
	    connected=false;
	    return null;
	}
	try {
	    ret=(Message)input.readObject();
	}
	catch(Exception e) {
	    connected=false;
	    System.err.println("RouterStub.Receive(): " + e);
	    return null;
	}
	return ret;
    }



    /** Tries to establish connection to router. Tries until router is up again. */
    public synchronized boolean Reconnect() {
	Address new_addr=null;

	if(connected) return false;

	Disconnect();
	while((new_addr=Connect()) == null) {
	    Util.Sleep(RECONNECT_TIMEOUT);
	}
	System.out.println("RouterStub.Reconnect(): new address is " + new_addr);
	return true;
    }



    public static void main(String[] args) {
	if(args.length != 2) {
	    System.out.println("RouterStub <host> <port>");
	    return;
	}
	RouterStub    stub=new RouterStub(args[0], new Integer(args[1]).intValue());
	Address       my_addr=stub.Connect();
	boolean       rc;
	final String  groupname="BelaGroup";
	Message       msg;
	List          mbrs;
	
	System.out.println("My address is " + my_addr);

	System.out.println("Registering under " + groupname);
	rc=stub.Register(groupname);
	System.out.println("Done, rc=" + rc);


	System.out.println("Getting members of " + groupname + ": ");
	mbrs=stub.Get(groupname);
	System.out.println("Done, mbrs are " + mbrs);


	for(int i=0; i < 10; i++) {
	    msg=new Message(null, my_addr, "Bela #" +i);
	    msg.AddHeader(new TunnelHeader(groupname));
	    rc=stub.Send(msg);
	    System.out.println("Sent msg, rc=" + rc);
	}

	for(int i=0; i < 10; i++) {
	    System.out.println("stub.Receive():");
	    msg=stub.Receive();
	    System.out.println("Received msg");
	}
	
	stub.Disconnect();
    }

    
}
