package JavaGroups.JavaStack;

import java.io.*;
import java.net.*;
import java.util.*;
import JavaGroups.*;
import JavaGroups.JavaStack.Protocols.TunnelHeader;






/** 
    Router for TCP based group comunication (using layer TCP instead of UDP). Instead of the TCP
    layer sending packets point-to-point to each other member, it sends the packet to the router
    which - depending on the target address - multicasts or unicasts it to the group / or single
    member.<p>
    This class is especially interesting for applets which cannot directly make connections
    (neither UDP nor TCP) to a host different from the one they were loaded from. Therefore,
    an applet would create a normal channel plus protocol stack, but the bottom layer would have
    to be the TCP layer which sends all packets point-to-point (over a TCP connection) to the 
    router, which in turn forwards them to their end location(s) (also over TCP). A centralized
    router would therefore have to be running on the host the applet was loaded from.<p>
    An alternative for running JavaGroups in an applet (IP multicast is not allows in applets as of
    1.2), is to use point-to-point UDP communication via the gossip server. However, then the appplet
    has to be signed which involves additional administrative effort on the part of the user.
*/
public class Router {
    Hashtable     groups=new Hashtable();  // groupname - vector of AddressEntry's
    int           port=8080;
    ServerSocket  srv_sock=null;





    public Router(int port) throws Exception {
	this.port=port;
	srv_sock=new ServerSocket(port, 50);  // backlog of 50 connections
    }



    public void Start() {
	Request             req;
	Socket              sock;
	SocketThread        socket_thread;
	AddressEntry        entry;
	ObjectInputStream   input;
	ObjectOutputStream  output;
	Address             peer_addr;

	System.out.println("\nRouter started at " + new Date().toGMTString());
	System.out.println("Listening on port " + port + "\n");

	while(true) {
	    try {
		sock=srv_sock.accept();
		sock.setSoLinger(true, 500);
		
		peer_addr=new Address(sock.getInetAddress(), sock.getPort());
		output=new ObjectOutputStream(sock.getOutputStream());
		output.writeObject(peer_addr);
		
		
		// We can have 2 kinds of messages at this point: GET requests or REGISTER requests.
		// GET requests are processed right here, REGISTRATION requests cause the spawning of
		// a separate thread handling it (long running thread as it will do the message routing
		// on behalf of that client for the duration of the client's lifetime).

		input=new ObjectInputStream(sock.getInputStream());
		req=(Request)input.readObject();

		switch(req.type) {
		case Request.GET:
		    ProcessGetRequest(sock, output, req.groupname); // closes sock after processing
		    break;
		case Request.REGISTER:
		    AddEntry(req.groupname, new AddressEntry(req.addr, sock, output));
		    new SocketThread(sock, input).start();
		    break;
		default:
		    System.err.println("Router: request of type " + req.type + " not recognized");
		    continue;
		}
	    }
	    catch(Exception e) {
		System.err.println("Router.Start(): " + e);
		continue;
	    }
	}
    }


    public void Stop() {
	
    }


    void ProcessGetRequest(Socket sock, ObjectOutputStream output, String groupname) {
	List                grpmbrs=(List)groups.get(groupname), ret=null;
	AddressEntry        entry;
	

	if(grpmbrs != null && grpmbrs.Size() > 0) {
	    ret=new List();
	    for(Enumeration e=grpmbrs.Elements(); e.hasMoreElements();) {
		entry=(AddressEntry)e.nextElement();
		ret.Add(entry.addr);
	    }
	}
	try {
	    output.writeObject(ret);
	}
	catch(Exception e) {
	    System.err.println(e);
	}
	finally {
	    try {
		if(output != null) 
		    output.close(); 
		sock.close();
	    } 
	    catch(Exception e) {}
	}	
    }


	
    synchronized void Route(Message msg) {
	Address         dest=(Address)msg.GetDest(), src=(Address)msg.GetSrc();
	TunnelHeader    hdr=(TunnelHeader)msg.PeekHeader();
	String          dest_group=hdr != null ? hdr.channel_name : null;

	if(dest == null) { // send to all members in group 'dest.GetChannelName()'	
	    if(dest_group == null) {
		System.err.println("Router.Route(): both dest address and group are null");
		return;
	    }
	    else
		SendToAllMembersInGroup(dest_group, msg);
	}                      
	else {                  // send to destination address
	    ObjectOutputStream out=FindSocket(dest);
	    if(out != null)
		SendToMember(out, msg);
	    else
		System.out.println("Router.Route(" + msg + "): failed; outstream is null !");
	}
    }
	


    
    void AddEntry(String groupname, AddressEntry e) {
	List         val;
	AddressEntry old_entry;
	
	// Util.Print("AddEntry(" + groupname + ", " + e + ")");

	if(groupname == null) {
	    System.err.println("AddEntry(): groupname was null, not added !");
	    return;
	}

	synchronized(groups) {
	    val=(List)groups.get(groupname);
	    
	    if(val == null) {
		val=new List();
		groups.put(groupname, val);
	    }
	    if(val.Contains(e)) {
		old_entry=(AddressEntry)val.RemoveElement(e);
		if(old_entry != null)
		    old_entry.Destroy();
	    }
	    val.Add(e);
	}
    }
    

    
    void RemoveEntry(Socket sock) {
	List          val;
	AddressEntry  entry;
	
	synchronized(groups) {
	    for(Enumeration e=groups.keys(); e.hasMoreElements();) {
		val=(List)groups.get((String)e.nextElement());

		for(Enumeration e2=val.Elements(); e2.hasMoreElements();) {
		    entry=(AddressEntry)e2.nextElement();
		    if(entry.sock == sock) {
			try {
			    entry.sock.close();
			}
			catch(Exception ex) {
			    System.err.println(ex);
			}
			//Util.Print("Removing entry " + entry);
			val.RemoveElement(entry);
			return;
		    }
		}
	    }
	}
    }


    void RemoveEntry(OutputStream out) {
	List          val;
	AddressEntry  entry;
	
	synchronized(groups) {
	    for(Enumeration e=groups.keys(); e.hasMoreElements();) {
		val=(List)groups.get((String)e.nextElement());

		for(Enumeration e2=val.Elements(); e2.hasMoreElements();) {
		    entry=(AddressEntry)e2.nextElement();
		    if(entry.output == out) {
			try {
			    if(entry.sock != null)
				entry.sock.close();
			}
			catch(Exception ex) {
			    System.err.println(ex);
			}
			//Util.Print("Removing entry " + entry);
			val.RemoveElement(entry);
			return;
		    }
		}
	    }
	}
    }



    void RemoveEntry(String groupname, Address addr) {
	List          val;
	AddressEntry  entry;
	

	synchronized(groups) {
	    val=(List)groups.get(groupname);
	    if(val == null || val.Size() == 0)
		return;
	    for(Enumeration e2=val.Elements(); e2.hasMoreElements();) {
		entry=(AddressEntry)e2.nextElement();
		if(entry.addr.equals(addr)) {
		    try {
			if(entry.sock != null)
			    entry.sock.close();
		    }
		    catch(Exception ex) {
			System.err.println(ex);
		    }
		    //Util.Print("Removing entry " + entry);
		    val.RemoveElement(entry);
		    return;
		}
	    }
	}
    }


    ObjectOutputStream FindSocket(Address addr) {
	List          val;
	AddressEntry  entry;
	
	synchronized(groups) {
	    for(Enumeration e=groups.keys(); e.hasMoreElements();) {
		val=(List)groups.get((String)e.nextElement());
		for(Enumeration e2=val.Elements(); e2.hasMoreElements();) {
		    entry=(AddressEntry)e2.nextElement();
		    if(addr.equals(entry.addr))
			return entry.output;
		}
	    }
	    return null;
	}
    }

    
    
    
    
    void SendToAllMembersInGroup(String groupname, Message msg) {
	List val;
	
	synchronized(groups) {
	    val=(List)groups.get(groupname);
	    if(val == null || val.Size() == 0)
		return;
	    for(Enumeration e=val.Elements(); e.hasMoreElements();) {
		SendToMember(((AddressEntry)e.nextElement()).output, msg);
	    }
	}
    }
	
	
    void SendToMember(ObjectOutputStream out, Message msg) {
	try {
	    if(out != null)
		out.writeObject(msg);
	}
	catch(Exception e) {
	    System.err.println("Router.SendToMember(): " + e);
	    RemoveEntry(out); // closes socket
	}
    }
	









    class AddressEntry {
	Address             addr=null;
	Socket              sock=null;
	ObjectOutputStream  output=null;


	
	public AddressEntry(Address addr, Socket sock, ObjectOutputStream output) {
	    this.addr=addr; this.sock=sock; this.output=output;
	}


	void Destroy() {
	    if(output != null) {
		try {output.close();}
		catch(Exception e) {}
		output=null;
	    }
	    if(sock != null) {
		try {
		    sock.close();
		}
		catch(Exception e) {}
		sock=null;
	    }
	}
	
	public boolean equals(Object other) {
	    return addr.equals(((AddressEntry)other).addr);
	}

	public String toString() {
	    return "addr=" + addr + ", sock=" + sock;
	}
    }


    
    /** A SocketThread manages one connection to a client. Its main task is message routing. */
    class SocketThread extends Thread {
	Socket             sock=null;
	ObjectInputStream  input=null;


	public SocketThread(Socket sock, ObjectInputStream ois) {
	    this.sock=sock;
	    input=ois;
	}

	void CloseSocket() {
	    try {
		if(input != null)
		    input.close();
		if(sock != null)
		    sock.close();
	    }
	    catch(Exception e) {}
	}


	public void run() {
	    Object obj;

	    while(true) {
		try {
		    obj=input.readObject();
		    if(!(obj instanceof Message)) {
			System.err.println("Router.SocketThread.run(): received request of " +
					   "type " + obj.getClass().getName() + obj + "\nCannot " + 
					   "process; must be of type Message !");
			continue;
		    }
		    Route((Message)obj);
		}
		catch(IOException io_ex) {
		    System.err.println("Router: client " + sock.getInetAddress().getHostName() + ":" +
				       sock.getPort() + " closed " +
				       "connection; removing it from routing table");
		    RemoveEntry(sock); // will close socket
		    return;
		}
		catch(Exception e) {
		    System.err.println("Router.SocketThread.run(): " + e);
		    break;
		}
	    }
	    CloseSocket();
	}       
	
    }



    





    public static void main(String[] args) {
	String           arg, next_arg;
	int              port=8080;
	Router          router=null; 

	for(int i=0; i < args.length; i++) {
	    arg=args[i];
	    if(arg.equals("-help")) {
		System.out.println("Router [-port <port> ]");
		return;
	    }
	    else if(arg.equals("-port")) {
		port=new Integer(args[++i]).intValue();
		continue;
	    }
	}
	try {
	    router=new Router(port);
	    router.Start();
	    System.out.println("Router was created at " + new Date());
	    System.out.println("Listening on port " + port);
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }

    
}
