

package JavaGroups.JavaStack.Protocols;

import java.io.*;
import java.net.*;
import java.util.*;
import JavaGroups.*;
import JavaGroups.JavaStack.*;




/**
   Replacement for UDP. Instead of sending packets via UDP, a TCP connection is opened to a Router
   (using the RouterStub client-side stub),
   the IP address/port of which was given using channel properties <code>router_host</code> and 
   <code>router_port</code>. All outgoing traffic is sent via this TCP socket to the Router which
   distributes it to all connected TUNNELs in this group. Incoming traffic received from Router will
   simply be passed up the stack.
   <p>A TUNNEL layer can be used to penetrate a firewall, most firewalls allow creating TCP connections
   to the outside world, however, they do not permit outside hosts to initiate a TCP connection to a host
   inside the firewall. Therefore, the connection created by the inside host is reused by Router to
   send traffic from an outside host to a host inside the firewall.
 */
public class TUNNEL extends Protocol implements Runnable {
    Properties          properties=null;
    String              channel_name=null;
    boolean             trace=false;
    Vector              members=new Vector();    
    String              router_host=null;
    int                 router_port=0;
    Address             local_addr=null;  // sock's local addr and local port
    Thread              receiver=null;
    RouterStub          stub=null;
    


    public TUNNEL() {}



    public String toString() {
	return "Protocol TUNNEL(local_addr=" + local_addr + ")";
    }





    /* ------------------------------ Private methods -------------------------------- */




    public synchronized void HandleIncomingMessage(Message msg) {
	TunnelHeader hdr=(TunnelHeader)msg.RemoveHeader();

	if(trace) Util.Print("TUNNEL.HandleIncomingMessage(): received msg " + msg);
		

	/* Discard all messages destined for a channel with a different name */

	String ch_name=hdr != null ? hdr.channel_name : null;
	if(ch_name != null && !channel_name.equals(ch_name))
	    return;
	
	PassUp(new Event(Event.MSG, msg));	
    }
    
    



    void HandleDownEvent(Event evt) {
	boolean rc;

	switch(evt.GetType()) {

	case Event.TMP_VIEW:
	case Event.VIEW_CHANGE:
	    synchronized(members) {
		members.removeAllElements();
		Vector tmpvec=(Vector)((View)evt.GetArg()).GetMembers();
		for(int i=0; i < tmpvec.size(); i++)
		    members.addElement(tmpvec.elementAt(i));		
	    }
	    break;

	case Event.GET_LOCAL_ADDRESS:   // return local address -> Event(SET_LOCAL_ADDRESS, local)
	    PassUp(new Event(Event.SET_LOCAL_ADDRESS, local_addr));
	    break;

	case Event.CLEANUP:
	    PassUp(new Event(Event.CLEANUP_OK));
	    break;

	case Event.SET_LOCAL_ADDRESS:
	    local_addr=(Address)evt.GetArg();
	    break;
	    
	case Event.CONNECT:
	    channel_name=(String)evt.GetArg();
	    if(stub == null)
		System.err.println("TUNNEL.HandleDownEvent(CONNECT): router stub is null !");
	    else {
		rc=stub.Register(channel_name);
	    }

	    receiver=new Thread(this);
	    receiver.start();

	    PassUp(new Event(Event.CONNECT_OK));
	    break;

	case Event.DISCONNECT:
	    if(receiver != null) {
		receiver.stop();
		receiver=null;
	    }
	    TeardownTunnel();
	    PassUp(new Event(Event.DISCONNECT_OK));
	    break;


	case Event.STOP:
	    if(receiver != null) {
		receiver.stop();
		receiver=null;
	    }
	    PassUp(new Event(Event.STOP_OK));
	    TeardownTunnel();
	    break;
	}
    }



    /* ------------------------------------------------------------------------------- */







    public void run() {
	Message msg;

	if(stub == null) {
	    System.err.println("TUNNEL.run(): router stub is null; cannot receive messages from router !");
	    return;
	}

	while(true) {
	    msg=stub.Receive();
	    if(msg == null) {
		System.err.println("TUNNEL.run(): received a null message. Trying to reconnect to router");
		if(stub.Reconnect())
		    stub.Register(channel_name);
		System.out.println("TUNNEL.run(): reconnected !");
		continue;
	    }
	    HandleIncomingMessage(msg);
	}
    }






    /*------------------------------ Protocol interface ------------------------------ */

    public String GetName() {return "TUNNEL";}


    /** Setup the Protocol instance acording to the configuration string */
    public boolean SetProperties(Properties props) {
	String               str;

	this.props=props;
	str=props.getProperty("router_host");
	if(str != null) {
	    router_host=new String(str);
	    props.remove("router_host");
	}

	str=props.getProperty("router_port");
	if(str != null) {
	    router_port=new Integer(str).intValue();
	    props.remove("router_port");
	}

	str=props.getProperty("trace");
	if(str != null) {
	    trace=new Boolean(str).booleanValue();
	    props.remove("trace");
	}

	if(trace) 
	    Util.Print("TUNNEL(router_host=" + router_host + ";router_port=" + router_port +")");

	if(router_host == null || router_port == 0) {
	    System.err.println("TUNNEL.SetProperties(): both router_host and router_port have " +
			       "to be set !");
	    System.exit(-1);
	}

	if(props.size() > 0) {
	    System.err.println("TUNNEL.SetProperties(): the following properties are not recognized:");
	    props.list(System.out);
	    return false;
	}
	return true;
    }



    public void Up(Event evt) {
	if(evt.GetType() == Event.START) {
	    PassUp(evt);
	    CreateTunnel();  // will generate and pass up a SET_LOCAL_ADDRESS event
	    PassUp(new Event(Event.START_OK));
	    return;
	}	
	PassUp(evt);
    }






    /** Caller by the layer above this layer. We just pass it on to the router. */
    public synchronized void Down(Event evt) {
	Message       msg;
	Object        dest_addr, tmp_addr;
	TunnelHeader  hdr;
	boolean       rc;

	if(trace) Util.Print("TUNNEL.Down(): event is " + evt);
	
	if(evt.GetType() != Event.MSG) {
	    HandleDownEvent(evt);
	    return;
	}

	hdr=new TunnelHeader(channel_name);
	msg=(Message)evt.GetArg();
	msg.AddHeader(hdr);

	if(msg.GetSrc() == null)
	    msg.SetSrc(local_addr);

	if(!(rc=stub.Send(msg))) { // if msg is not sent okay, try reconnecting until router is ok again
	    if(stub.Reconnect())
		stub.Register(channel_name);
	}
    }




    /** Creates a TCP connection to the router */
    void CreateTunnel() {
	if(router_host == null || router_port == 0) {
	    System.err.println("TUNNEL.CreateTunnel(): router_host and/or router_port not set " +
			       "correctly; tunnel cannot be created !");
	    return;
	}

	stub=new RouterStub(router_host, router_port);
	local_addr=stub.Connect();
	if(local_addr == null) {
	    System.err.println("TUNNEL.CreateTunnel(): could not obtain local address !");
	    return;
	}
	PassUp(new Event(Event.SET_LOCAL_ADDRESS, local_addr));
    }




    /** Tears the TCP connection to the router down */
    synchronized void TeardownTunnel() {
	if(stub != null) {
	    stub.Disconnect();
	    stub=null;
	}
    }

    /*--------------------------- End of Protocol interface -------------------------- */





    
}
