package JavaGroups.JavaStack.Protocols;


import java.util.*;
import JavaGroups.*;
import JavaGroups.JavaStack.*;





/**
   Computes the broadcast messages that are stable, i.e. have been received by all members. Sends
   STABLE events up te stack when this is the case. Uses a probabilistic scheme to do so, as described
   in "GSGC: An Efficient Gossip-Style Garbage Collection Scheme for Scalable Reliable Multicast".<p>
   The only difference is that instead of using counters for an estimation of messages received from each
   member, we retrieve this actual information from the NAKACK layer (which must be present for the STABLE
   protocol to work).
 */

public class STABLE extends RpcProtocol implements Runnable {
    Object   local_addr=null;
    ViewId   vid=null;
    Vector   mbrs=new Vector();
    long     round=1;      // gossip round
    long     seqnos[];     // highest seqno received for each member (corresponds to membership)
    boolean  heard_from[]; // array of members from which we have received a gossip in the current round
    boolean  trace=false;
    double   subset=0.1;   // percentage of members to which gossip is sent (parameterizable by user)
    Thread   gossip_thread=null;
    Object   gossip_thread_mutex=new Object(); // notify gossip thread when it should gossip
    int      max_msgs=100;         // wait for n messages until sending gossip ...
    long     max_wait_time=10000;  // ... or until max_wait_time has elapsed, whichever comes first
    long     num_msgs=max_msgs;
    Object   highest_seqnos_mutex=new Object();  // mutex for interacting with NAKACK layer (GET_MSGS_RECVD)
    long     highest_seqnos_timeout=4000;




    public String  GetName() {return "STABLE";}



    public Vector RequiredUpServices() {
	Vector retval=new Vector();
	retval.addElement(new Integer(Event.GET_MSGS_RECEIVED));  // NAKACK layer
	return retval;
    }






    public boolean SetProperties(Properties props) {
	String     str;

	this.props=props;
	str=props.getProperty("trace");
	if(str != null) {
	    trace=new Boolean(str).booleanValue();
	    props.remove("trace");
	}

	str=props.getProperty("subset");
	if(str != null) {
	    subset=new Float(str).floatValue();
	    props.remove("subset");
	}

	str=props.getProperty("max_msgs");
	if(str != null) {
	    num_msgs=max_msgs=new Integer(str).intValue();
	    if(max_msgs <= 1) {
		System.err.println("STABLE.SetProperties(): value for 'max_msgs' must be greater than 1 !");
		return false;
	    }
	    props.remove("max_msgs");
	}

	str=props.getProperty("max_wait_time");
	if(str != null) {
	    max_wait_time=new Long(str).longValue();
	    props.remove("max_wait_time");
	}


	str=props.getProperty("highest_seqnos_timeout");
	if(str != null) {
	    highest_seqnos_timeout=new Long(str).longValue();
	    props.remove("highest_seqnos_timeout");
	}

	if(props.size() > 0) {
	    System.err.println("STABLE.SetProperties(): these properties are not recognized:");
	    props.list(System.out);
	    return false;
	}
	return true;
    }




    /* ------------------------- Request handler methods ----------------------------- */



    /** Contains the highest sequence numbers as seen by <code>sender</code> 
	@param view_id The view ID in which the gossip was sent. Must be the same as ours, otherwise
	               it is discarded
	@param round The round in which the gossip was sent
	@param highest_seqnos A vector with the highest sequence numbers as seen bu <code>sender</code>
	@param heard The sender's <code>heard_from</code> array. This allows us to minimize the gossip msgs
	             for a given round as a member does not have to receive gossip msgs from each member,
		     but members pass gossips they've received from others on in their own gossips. E.g.
		     when a member P (of group {P,Q,R}) receives a gossip from R, its own gossip to Q might
		     be {R,P}. Q, who hasn't received a gossip from R, will not need to receive it anymore
		     as it is already sent by P. This simple scheme reduces the number of gossip messages
		     needed.
	@param sender The sender of the gossip message (obviously :-))
     */
    public void Gossip(ViewId view_id, long round, long[] highest_seqnos, boolean[] heard, Object sender) {
	if(trace) 
	    System.out.println("<-- Gossip(sender=" + sender + ", round=" + round + ", seqnos=" + 
			       Util.Array2String(highest_seqnos) + ", heard=" + Util.Array2String(heard));

	if(vid == null || view_id == null || !vid.equals(view_id)) {
	    if(trace)
		System.err.println("STABLE.Gossip(): view ID s are different (" + vid + " != " + view_id +
				   "). Discarding gossip received");
	    return;
	}

	if(round > this.round) {
	    if(trace)
		System.out.println("STABLE.Gossip(): received a gossip from a higher round (" + round +
				   "); adopting my round (" + this.round + ") to " + round + 
				   " and retrieving highest sequence numbers");
	    Initialize(mbrs.size());
	    GetHighestSeqnos();
	    this.round=round;
	    return;
	}
	if(round < this.round) {
	    if(trace) 
		System.err.println("STABLE.Gossip(): received a gossip from a previous round (" + round +
				   "); my round is " + this.round +". Discarding gossip");
	    return;
	}

	if(highest_seqnos == null || seqnos == null || seqnos.length != highest_seqnos.length) {
	    if(trace)
		System.err.println("STABLE.Gossip(): size of seqnos and highest_seqnos are not equal ! " +
				   "Discarding gossip");
	    return;
	}

	UpdateSeqnos(highest_seqnos);
	UpdateHeardFrom(sender, heard);

	if(trace) System.out.println("heard_from=" + Util.Array2String(heard_from));

	if(HeardFromAll()) {
	    CallRemoteMethods(null, "Stability", vid, new Long(round), (long[])seqnos.clone(), local_addr,
			      GroupRequest.GET_NONE, 0);
	}
    }


    /** Contains the highest message sequence numbers (for each member) that can safely be
	deleted (because they have been seen by all members). */
    public void Stability(ViewId view_id, long round, long[] stability_vector, Object sender) {
	if(trace)
	    System.out.println("<=== Stability(sender=" + sender + ", round=" + round + ", vector=" +
			       Util.Array2String(stability_vector) + ")");
	
	PassUp(new Event(Event.STABLE, stability_vector));
	
	SuspendGossip();
	this.round++;
	GetHighestSeqnos();
	for(int i=0; i < heard_from.length; i++)
	    heard_from[i]=false;
	ResumeGossip();
    }

    


    /* --------------------- End of Request handler methods -------------------------- */




    
    /**
       <b>Callback</b>. Called by superclass when event may be handled.<p>
       <b>Do not use <code>PassUp</code> in this method as the event is passed up
       by default by the superclass after this method returns !</b>
       @return boolean Defaults to true. If false, event will not be passed up the stack.
     */
    public boolean HandleUpEvent(Event evt) {
	Message msg;

	switch(evt.GetType()) {

	case Event.MSG:
	    msg=(Message)evt.GetArg();
	    if(msg.GetDest() != null && (!((Address)msg.GetDest()).IsMulticastAddress()))
		break;
	    if(--num_msgs <= 0) {
		synchronized(gossip_thread_mutex) {
		    gossip_thread_mutex.notify();
		}
		num_msgs=max_msgs;
	    }
	    break;
	    
	case Event.START:
	    if(_corr != null)
		_corr.SetDeadlockDetection(false); // we use only asynchronous method invocations...

	    StartGossip();
	    break;
	    
	case Event.SET_LOCAL_ADDRESS:
	    local_addr=evt.GetArg();
	    break;
	}

	return true;
    }


    /**
       <b>Callback</b>. Called by superclass when event may be handled.<p>
       <b>Do not use <code>PassDown</code> in this method as the event is passed down
       by default by the superclass after this method returns !</b>
       @return boolean Defaults to true. If false, event will not be passed down the stack.
    */
    public boolean HandleDownEvent(Event evt) {
	switch(evt.GetType()) {

	case Event.VIEW_CHANGE:
	    View v=(View)evt.GetArg();
	    Vector tmp=v.GetMembers();
	    if(tmp != null) {
		mbrs.removeAllElements();
		for(int i=0; i < tmp.size(); i++)
		    mbrs.addElement(tmp.elementAt(i));
	    }
	    vid=v.GetVid();
	    Initialize(mbrs.size());
	    break;
	    
	case Event.STOP:
	    StopGossip();
	    PassDown(evt);
	    break;

	case Event.GET_MSGS_RECEIVED_OK:
	    long[]  tmp_seqnos=(long[])evt.GetArg();
	    boolean rcvd=true;

	    if(tmp_seqnos == null) rcvd=false;
	    if(rcvd && tmp_seqnos.length != seqnos.length) {
		if(trace)
		    System.err.println("STABLE.HandleDownEvent(GET_MSGS_RECEIVED): array of highest seqnos " +
				       "seen so far (received from NAKACK layer) has a different length (" +
				       tmp_seqnos.length + ") from 'seqnos' array (" + seqnos.length + ")");
		rcvd=false;
	    }
	    
	    synchronized(highest_seqnos_mutex) {
		if(rcvd)
		    for(int i=0; i < seqnos.length; i++)
			seqnos[i]=tmp_seqnos[i];
		highest_seqnos_mutex.notify();
	    }
	    break;
	    
	}
	return true;
    }



    /**
     */
    public void run() {
	if(mbrs != null && mbrs.size() > 0)
	    Initialize(mbrs.size());
	while(true) {
	    synchronized(gossip_thread_mutex) {
		try {gossip_thread_mutex.wait(max_wait_time);}
		catch(Exception e) {System.err.println(e);}
		num_msgs=max_msgs;
		SendGossip();
	    }
	}
    }





    void Initialize(int size) {
	seqnos=new long[size];
	for(int i=0; i < seqnos.length; i++) seqnos[i]=-1;
	heard_from=new boolean[size];
	for(int i=0; i < heard_from.length; i++) heard_from[i]=false;
    }


    void UpdateSeqnos(long[] gossip_seqnos) {
	synchronized(seqnos) {
	    for(int i=0; i < gossip_seqnos.length; i++) {
		seqnos[i]=Math.min(seqnos[i], gossip_seqnos[i]);
	    }
	}
    }

    void UpdateHeardFrom(Object sender, boolean[] heard) {
	int index;

	if(mbrs == null) {
	    if(trace) System.err.println("STABLE.UpdateHeardFrom(): membership is null !");
	    return;
	}

	index=mbrs.indexOf(sender);
	if(index < 0) {
	    if(trace) 
		System.err.println("STABLE.UpdateHeardFrom(): sender " + sender + " not found in mbrs !");
	    return;
	}
	
	heard_from[index]=true;
	for(int i=0; i < heard_from.length; i++)
	    if(heard[i])
		heard_from[i]=true;
    }


    boolean HeardFromAll() {
	if(heard_from == null) return false;
	for(int i=0; i < heard_from.length; i++)
	    if(!heard_from[i])
		return false;
	return true;
    }


    /**
       Send our <code>seqnos</code> array to a subset of the membership
     */
    void SendGossip() {
	Vector gossip_subset;

	if(seqnos == null)
	    GetHighestSeqnos();
	
	if(mbrs == null || mbrs.size() < 1) {
	    if(trace) System.err.println("STABLE.SendGossip(): membership not known or null !");
	    return;
	}

	gossip_subset=Util.PickSubset(mbrs, subset);
	if(gossip_subset == null || gossip_subset.size() < 1) {
	    if(trace) System.err.println("STABLE.SendGossip(): picked empty subset !");
	    return;
	}

	if(trace)
	    System.out.println("--> Gossip(subset=" + gossip_subset + ", round=" + round + ", seqnos=" + 
			       Util.Array2String(seqnos));

	for(int i=0; i < gossip_subset.size(); i++) {
	    try {
		CallRemoteMethod(gossip_subset.elementAt(i), "Gossip", vid, new Long(round), 
				 (long[])seqnos.clone(), heard_from.clone(), 
				 local_addr, GroupRequest.GET_NONE, 0);
	    }
	    catch(Exception e) {
		System.err.println("STABLE.SendGossip(): " + e);
	    }
	}
    }




    /** Sends GET_MSGS_RECEIVED to NAKACK layer (above us !) and stores result in <code>seqnos</code>.
	In case <code>seqnos</code> does not yet exist it creates and initializes it.
     */
    void GetHighestSeqnos() {
	if(seqnos == null)
	    Initialize(mbrs.size());
	synchronized(highest_seqnos_mutex) {
	    PassUp(new Event(Event.GET_MSGS_RECEIVED));
	    try {
		highest_seqnos_mutex.wait(highest_seqnos_timeout);
	    }
	    catch(Exception e) {}
	}
    }




    void StartGossip() {
	if(gossip_thread == null) {
	    gossip_thread=new Thread(this, "STABLE GossipThread");
	    gossip_thread.start();
	}
    }


    void StopGossip() {
	if(gossip_thread != null) {
	    gossip_thread.stop();
	    gossip_thread=null;
	}
    }

    void SuspendGossip() {
	if(gossip_thread != null)
	    gossip_thread.suspend();
    }

    void ResumeGossip() {
	if(gossip_thread != null)
	    gossip_thread.resume();
	else
	    StartGossip();
    }




}
