package JavaGroups.JavaStack.Protocols;

import java.io.*;
import java.util.*;
import JavaGroups.*;
import JavaGroups.JavaStack.*;


class PiggybackHeader implements Serializable {

    public PiggybackHeader() {
    }

    public String toString() {
	return "[PIGGYBACK: <variables> ]";
    }
}


/**
   Combines multiple messages into a single large one. As many messages as possible are combined into
   one, after a max timeout or when the msg size becomes too big the message is sent. On the receiving
   side, the large message is spliced into the smaller ones and delivered.
 */

public class PIGGYBACK extends Protocol {
    boolean  trace=false;
    long     max_wait_time=20; // milliseconds: max. wait between consecutive msgs
    long     max_size=8192;    // don't piggyback if created msg would exceed this size (in bytes)
    Queue    msg_queue=new Queue();
    Packer   packer=null;
    boolean  packing=false;
    Object   local_addr=null;


    class Packer extends Thread {
	

	public void run() {
	    long     current_size=0;
	    long     start_time, time_to_wait=max_wait_time;
	    Message  m, new_msg;
	    Vector   msgs;

	    while(true) {
		try {
		    m=(Message)msg_queue.Remove();
		    m.SetSrc(local_addr);
		    start_time=System.currentTimeMillis();
		    current_size=0;
		    new_msg=new Message();
		    msgs=new Vector();
		    msgs.addElement(m);
		    current_size+=m.Size();

		    while(System.currentTimeMillis() - start_time <= max_wait_time &&
			  current_size <= max_size) {
			
			time_to_wait=max_wait_time - (System.currentTimeMillis() - start_time);
			if(time_to_wait <= 0)
			    break;

			try {
			    m=(Message)msg_queue.Peek(time_to_wait);
			    m.SetSrc(local_addr);
			}
			catch(Timeout timeout) {
			    break;
			}
			if(m == null || m.Size() + current_size > max_size)
			    break;
			m=(Message)msg_queue.Remove();
			current_size+=m.Size();
			msgs.addElement(m);			
		    }

		    try {
			new_msg.AddHeader(new PiggybackHeader());
			new_msg.SetBuffer(Util.ObjectToByteBuffer(msgs));
			PassDown(new Event(Event.MSG, new_msg));
			if(trace)
			    System.out.println("----> PIGGYBACK.run(): combined " + msgs.size() +
					       " messages of a total size of " + current_size + " bytes");
		    }
		    catch(Exception e) {
			System.err.println("PIGGYBACK.run(): " + e);
		    }
		}
		catch(QueueClosed closed) {
		    if(trace) System.out.println("PIGGYBACK: packer stopped as queue is closed");
		    break;
		}
	    }
	}


    }



    /** All protocol names have to be unique ! */
    public String  GetName() {return "PIGGYBACK";}


    public boolean SetProperties(Properties props) {
	String     str;

	this.props=props;
	str=props.getProperty("trace");
	if(str != null) {
	    trace=new Boolean(str).booleanValue();
	    props.remove("trace");
	}
	str=props.getProperty("max_wait_time");
	if(str != null) {
	    max_wait_time=new Long(str).longValue();
	    props.remove("max_wait_time");
	}
	str=props.getProperty("max_size");
	if(str != null) {
	    max_size=new Long(str).longValue();
	    props.remove("max_size");
	}

	if(props.size() > 0) {
	    System.err.println("PIGGYBACK.SetProperties(): these properties are not recognized:");
	    props.list(System.out);
	    return false;
	}
	return true;
    }






    public void Up(Event evt) {
	Message          msg;
	PiggybackHeader  hdr;
	Object           obj;
	Vector           messages;

	switch(evt.GetType()) {

	case Event.START:
	    StartPacker();
	    break;

	case Event.SET_LOCAL_ADDRESS:
	    local_addr=evt.GetArg();
	    break;

	case Event.MSG:
	    msg=(Message)evt.GetArg();
	    obj=msg.PeekHeader();
	    if(obj == null || !(obj instanceof PiggybackHeader))
		break;
	    
	    msg.RemoveHeader();
	    try {
		messages=(Vector)Util.ObjectFromByteBuffer(msg.GetBuffer());
		if(trace) System.out.println("-----> PIGGYBACK.Up(): unpacking " + messages.size() +
					     " messages");
		for(int i=0; i < messages.size(); i++)
		    PassUp(new Event(Event.MSG, (Message)messages.elementAt(i)));
	    }
	    catch(Exception e) {
		System.err.println("PIGGYBACK.Up(): piggyback message does not contain a vector of " +
				   "piggybacked messages, discarding message ! Exception is " + e);
		return;
	    }

	    return;             // don't pass up !
	}

	PassUp(evt);            // Pass up to the layer above us
    }





    public void Down(Event evt) {
	Message          msg;

	switch(evt.GetType()) {
	case Event.STOP:
	    packing=false;
	    msg_queue.Close(true);  // flush pending messages, this should also stop the packer ...
	    StopPacker();           // ... but for safety reasons, we stop it here again
	    PassDown(evt);
	    break;


	case Event.MSG:
	    msg=(Message)evt.GetArg();

	    if(msg.GetDest() != null && !((Address)msg.GetDest()).IsMulticastAddress())
		break;  // unicast message, handle as usual

	    if(!packing)
		break;  // pass down as usual; we haven't started yet

	    try {
		msg_queue.Add(msg);
	    }
	    catch(QueueClosed closed) {
		break;  // pass down regularly
	    }
	    return;
	}

	PassDown(evt);          // Pass on to the layer below us
    }




    void StartPacker() {
	if(packer == null) {
	    packing=true;
	    packer=new Packer();
	    packer.start();
	}
    }


    void StopPacker() {
	if(packer != null) {
	    packing=false;
	    packer.stop();
	    packer=null;
	}
    }

}
