package JavaGroups.JavaStack.Protocols;

import java.io.*;
import java.util.Vector;
import java.util.Hashtable;
import java.util.Properties;
import JavaGroups.*;
import JavaGroups.JavaStack.*;



class FragHeader implements Serializable {
    public long    id=0;
    public int     frag_id=0;
    public int     num_frags=0;


    public FragHeader(long id, int frag_id, int num_frags) {
	this.id=id;
	this.frag_id=frag_id;
	this.num_frags=num_frags;
    }

    public String toString() {
	return "[FRAG: id=" + id + ", frag_id=" + frag_id + ", num_frags=" + num_frags + "]";
    }
}


/**
   Keeps track of the fragments that are received. Reassembles fragements into entire
   messages when all fragments have been received.
 */
class FragmentationTable {
    private Hashtable h=new Hashtable();



    class Entry {
	int       tot_frags=0;
	byte[]    fragments[]=null;  // each fragment is a byte buffer
	int       number_of_frags_recvd=0;

	Entry(int tot_frags) {
	    this.tot_frags=tot_frags;
	    fragments=new byte[tot_frags][];
	    for(int i=0; i < tot_frags; i++) fragments[i]=null;
	}

	public void Set(int frag_id, byte[] frag) {
	    fragments[frag_id]=frag;
	    number_of_frags_recvd++;
	}

	public boolean IsComplete() {
	    if(number_of_frags_recvd < tot_frags)
		return false;
	    for(int i=0; i < fragments.length; i++) {
		if(fragments[i] == null)
		    return false;
	    }
	    return true;
	}

	public byte[] AssembleBuffer() {
	    return Util.DefragmentBuffer(fragments);
	}

	public String toString() {
	    StringBuffer ret=new StringBuffer();
	    ret.append("[tot_frags=" + tot_frags + ", number_of_frags_recvd=" + number_of_frags_recvd + "]");
	    return ret.toString();
	}
    }




    /**
       Creates a new entry if not yet present. Adds the fragment. If all fragements for a given
       message haven been received, an entire message is reassembled and returned. Otherwise
       null is returned.
     */
    public synchronized byte[] Add(Object src, long id, int frag_id, int tot_frags, 
				    byte[] fragment) {

	//System.out.println("Add(): src=" + src + ", id=" + id + ", frag_id=" + frag_id +
	//	   ", tot_frags=" + tot_frags + ", fragment=" + fragment.length + "bytes");

	byte[]     retval=null;
	Hashtable  val=(Hashtable)h.get(src);
	Entry      e;

	if(val == null) {         // Create new entry if not yet present
	    val=new Hashtable();
	    h.put(src, val);
	    e=new Entry(tot_frags);
	    val.put(new Long(id), e);
	}
	else {
	    e=(Entry)val.get(new Long(id));
	    if(e == null) {
		e=new Entry(tot_frags);
		val.put(new Long(id), e);
	    }
	}
	e.Set(frag_id, fragment);
	if(e.IsComplete()) {
	    retval=e.AssembleBuffer();
	    val.remove(new Long(id));
	}

	return retval;
    }

    public void Reset() {}
}



/**
   Fragmentation layer. Fragments messages larger than FRAG_SIZE into smaller packets.
   Reassembles fragmented packets into bigger ones. The fragmentation number is prepended
   to the messages as a header (and removed at the receiving side).<p>
   Each fragment is identified by (a) the sender (part of the message to which the header is appended),
   (b) the fragmentation ID (which is unique per FRAG layer (monotonically increasing) and (c) the 
   fragement ID which ranges from 0 to number_of_fragments-1.<p>
   Requirement: lossless delivery (e.g. NAK, ACK). No requirement on ordering. Works for both unicast and
   multicast messages.

   Typical stack:
   <pre>
   FIFO
   FRAG
   NAK
   UDP
   </pre>

 */

public class FRAG extends Protocol {
    private int                 frag_size=8192;  // conservative value
    private boolean             trace=false;
    private FragmentationTable  frag_table=new FragmentationTable();
    private int                 curr_id=1;



    public String  GetName() {return "FRAG";}


    /** Setup the Protocol instance acording to the configuration string */
    public boolean SetProperties(Properties props) {
	String     str;

	super.SetProperties(props);

	str=props.getProperty("frag_size");
	if(str != null) {
	    frag_size=new Integer(str).intValue();
	    props.remove("frag_size");
	}

	str=props.getProperty("trace");
	if(str != null) {
	    trace=new Boolean(str).booleanValue();
	    props.remove("trace");
	}
	if(props.size() > 0) {
	    System.err.println("FRAG.SetProperties(): the following properties are not recognized:");
	    props.list(System.out);
	    return false;
	}
	return true;
    }



    /** Just remove if you don't need to reset any state */
    public void Reset() {frag_table.Reset();}






    /**
       Fragment a packet if larger than frag_size (add a header). Otherwise just pass down. Only
       add a header if framentation is needed !
     */
    public void Down(Event evt) {
	if(evt.GetType() == Event.MSG) {
	    Message msg=(Message)evt.GetArg();
	    long    size=msg.Size();
	    
	    if(size > frag_size) {
		Fragment(msg);  // Fragment and pass down
		return;
	    }
	}	
	PassDown(evt);  // Pass on to the layer below us
    }






    /**
       If event is a message, if it is fragmented, re-assemble fragments into big message and pass up
       the stack.
     */
    public void Up(Event evt) {
	byte[] m;

	if(evt.GetType() == Event.MSG) {
	    Message    msg=(Message)evt.GetArg(), assembled_msg=null;
	    Object     obj=msg.PeekHeader();
	    FragHeader hdr;


	    
	    if(obj != null && obj instanceof FragHeader) { // needs to be defragmented
		hdr=(FragHeader)msg.RemoveHeader();
		// 1. Get all the fragment buffers
		// 2. When all are received -> Assemble them into one big buffer
		// 3. Read headers and byte buffer from big buffer
		// 4. Set headers and buffer in msg
		// 5. Pass msg up the stack

		//System.out.println("FRAG: received msg from " + msg.GetSrc());

		m=frag_table.Add(msg.GetSrc(), hdr.id, hdr.frag_id, hdr.num_frags, msg.GetBuffer());
		if(m != null) {
		    try {
			assembled_msg=(Message)Util.ObjectFromByteBuffer(m); // Create new message from byte buffer
			assembled_msg.SetSrc(msg.GetSrc());
			PassUp(new Event(Event.MSG, assembled_msg));
		    }
		    catch(Exception e) {
			System.err.println("FRAG.Up(): " + e);
		    }
		}
		return;
	    }
	}	
	PassUp(evt); // Pass up to the layer above us
    }








    private void Fragment(Message msg) {	
	ByteArrayOutputStream out_stream;
	ObjectOutputStream    out;
	byte[]                buffer;
	byte[]                fragments[];
	Event                 evt;
	FragHeader            hdr;
	Message               frag_msg;
	Object                dest=msg.GetDest(), src=msg.GetSrc();
	long                  id=curr_id++;


	try {

	    // Write message into a byte buffer and fragment it
	    buffer=Util.ObjectToByteBuffer(msg);
	    fragments=Util.FragmentBuffer(buffer, frag_size);
	    
	    if(trace)
		System.out.println("FRAG.Fragment(): fragmenting packet (size=" + buffer.length + 
				   ") into " + fragments.length + " fragment(s)");

	    
	    /* Send all fragments as separate messages (with same ID !).
	       Example:
	       Given the generated ID is 2344, number of fragments=3, message {dst,src,buf}
	       would be fragmented into:

	       [2344,3,0]{dst,src,buf1}, 
	       [2344,3,1]{dst,src,buf2} and 
	       [2344,3,2]{dst,src,buf3}
	    */

	    for(int i=0; i < fragments.length; i++) {
		frag_msg=new Message(dest, src, fragments[i]);
		hdr=new FragHeader(id, i, fragments.length);
		frag_msg.AddHeader(hdr);
		evt=new Event(Event.MSG, frag_msg);
		PassDown(evt);
	    }	   
	}
	catch(Exception e) {
	    System.err.println("FRAG.Fragment(): " + e);
	}

    }

}
