package JavaGroups;

import java.io.*;

/** 
    Container for header data which is added to the front of a Message by a protocol layer
    when traveling down the stack. When traveling up on the other side, each protocol layer
    will remove its Header. Headers can only contain objects, which have to be serializable
    (or support externalization). Both the object and its byte representation are maintained
    for faster access of its contents, 
    <em>but the byte buffer is only created when needed (e.g. when Size() or writeExternal() 
    is called) ! </em>.
*/

public class Header implements Externalizable, Cloneable {
    private   byte     data[]=null;
    transient Object   obj=null;
    

    public Header() {}



    public Header(Object obj) {	
	this.obj=obj;
    }


    public Header(byte[] data, Object obj) {
	this.data=data;
	this.obj=obj;
    }


    public void SetObject(Object obj) {
	this.obj=obj; data=null;
    }



    public Object GetObject() {
	return obj;
    }


    public long Size() {
	if(obj == null) return 0;
	if(data == null)
	    ReconstructData();
	return data != null ? data.length : 0;
    }


    public String toString() {	
	return obj != null? "Header[" + obj.toString() + "]" : "Header[null]";
    }


    public Object clone() throws CloneNotSupportedException {
	return new Header(data, obj);
    }



    void ReconstructObject() {

	// System.out.println("-- Header.ReconstructObject(): data = " + data.length);
	
	try {
	    obj=Util.ObjectFromByteBuffer(data);
	}
	catch(Exception e) {
	    obj=null;
	    System.err.println("Header.ReconstructObject: " + e);
	}	
    }



    void ReconstructData() {

	//System.out.println("-- Header.ReconstructData(): obj = " + obj.toString());

	try {
	    data=Util.ObjectToByteBuffer(obj);
	}
	catch(Exception e) {
	    System.err.println("Header.ReconstructData(): " + e);
	}
    }




    public void writeExternal(ObjectOutput out) throws IOException {
	if(obj == null)
	    out.writeInt(0);
	else {
	    if(data == null)
		ReconstructData();
	    out.writeInt(data.length);
 	    out.write(data);
	}
    }



    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException {
 	int tmp=in.readInt();
 	if(tmp > 0) {
 	    data=new byte[tmp];
 	    in.readFully(data);
	    ReconstructObject();
 	}
    }





    public static void main(String[] args) {
	Header             h1=new Header("Bela Ban");
	Header             h2;
	FileOutputStream   outstream;
	FileInputStream    instream;
	ObjectOutputStream os;
	ObjectInputStream  in;



	try {

	    System.out.println("h1=" + h1 + ", size=" + h1.Size());
	    



	    System.out.println(h1 + ", " + h1.GetObject() + ", " + h1.Size());
	    outstream=new FileOutputStream("hdr.out");
	    os=new ObjectOutputStream(outstream);
	    os.writeObject(h1);

	    instream=new FileInputStream("hdr.out");
	    in=new ObjectInputStream(instream);
	    h2=(Header)in.readObject();
	    System.out.println(h2 + ", " + h2.GetObject() + ", " + h2.Size());
	}
	catch(Exception e) {
	    System.err.println(e);
	}


    }


}
