package JavaGroups;


import java.util.*;

/**
   ACK-based sliding window for a sender. Messages are added to the window according to message
   IDs. When an ACK is received, the corresponding entry is marked and the window advanced as
   much as possible. A retransmission thread continously iterates over the entries,
   retransmitting messages for which no ACK has been received in <code>timeout</code> time.
 */
public class AckSenderWindow implements Runnable {
    long                 next_seqno=0;              // next sequence number to expect
    Thread               retransmit_thread=null;
    long                 timeout=2000;
    final long           min_wait_time=500;
    RetransmitCommand    retransmit_command=null;
    SortedList           msgs=new SortedList();


    public interface RetransmitCommand {
	void Retransmit(long seqno, Message msg, int num_tries);
    }


    class Entry implements Comparable {
	long      seqno=0;
	long      wakeup_at=System.currentTimeMillis() + timeout;
	Message   msg;
	int       num_tries=0;
	boolean   ack_received=false;

	Entry(long seqno, Message msg) {
	    this.seqno=seqno;
	    this.msg=msg;
	}

	public int Compare(Object other) {
	    Entry e=(Entry)other;
	    return seqno == e.seqno ? 0 : seqno < e.seqno ? -1 : 1;
	}

	public String toString() {
	    StringBuffer ret=new StringBuffer();
	    ret.append(seqno);
	    if(ack_received)
		ret.append("+");
	    else
		ret.append("-");
	    return ret.toString();
	}
    }


    /**
       Creates a new instance. Thre retransmission thread has to be started separately with
       <code>Start</code>.
       @param com If not null, its method <code>Retransmit</code> will be called when a message
                  needs to be retransmitted (called by the retransmitter thread).
       @param initial_seqno The first sequence number to expect an ACK for.
     */
    public AckSenderWindow(RetransmitCommand com, long initial_seqno) {
	retransmit_command=com;
	next_seqno=initial_seqno;
	Start();
    }
    


    public long GetNextSeqno()            {return next_seqno;}
    public long GetTimeout()              {return timeout;}
    public void SetTimeout(long timeout)  {this.timeout=timeout;}


    /**
       Starts the retransmission thread (will not be started automatically !). The thread scans over
       the messages sent and requests retransmission for those that haven't received an ACK for
       a certain period of time. 
     */
    public void Start() {
	if(retransmit_thread == null) {
	    retransmit_thread=new Thread(this, "AckSenderWindow.RetransmitThread");
	    retransmit_thread.start();
	}
    }


    public void Stop() {
	if(retransmit_thread != null) {
	    retransmit_thread.stop();
	    retransmit_thread=null;
	}
    }


    public void Reset() {
	Stop();
	msgs.RemoveAll();
    }



    /** 
	Add a new message. <code>seqno</code> should always be 1 higher than the previous
	one. Messages will be ordered according to seqno. Note that <code>msg</code> must be a copy of
	the message, otherwise the entry stored in this table might be modified by the message traveling
	down the stack (e.g. addition of headers) !
    */
    public void Add(long seqno, Message msg) {
	Entry     e;
	boolean   inserted=false;

	//System.out.println("Add(" + seqno + "), next_seqno=" + next_seqno);

	synchronized(msgs) {
	    if(seqno < next_seqno)
		return;

	    msgs.Add(new Entry(seqno, msg));

	    //System.out.println("Add(" + seqno + "): " + toString() + ", next_seqno=" + next_seqno);
	    msgs.notifyAll();  // wake up retransmission thread
	}
    }





    /**
       Marks one message as received. Removes as many messages as possible (removing them also from
       retransmission), sets <code>next_seqno</code> correspondingly.
     */
    public void Ack(long seqno) {
	Entry    entry;
	boolean  found=false;

	synchronized(msgs) {
	    if(seqno < next_seqno)
		return;

	    for(Enumeration e=msgs.Elements(); e.hasMoreElements();) {
		entry=(Entry)e.nextElement();
		if(entry.seqno == seqno) {
		    found=true;
		    entry.ack_received=true;		    
		    break;
		}
	    }

	    if(!found)
		return;

	    while(msgs.Size() > 0) {  // remove as many messages as possible
		entry=(Entry)msgs.PeekAtHead();
		if(next_seqno == entry.seqno && entry.ack_received) {
		    msgs.RemoveFromHead();
		    next_seqno++;
		}
		else
		    break;
	    }

	    //System.out.println("Ack(" + seqno + "): " + this); System.out.flush();
	    
	    msgs.notifyAll();  // wake up retransmission thread
	}
    }





    /**
       Iterate over msgs. For entries that haven't received an ACK within <code>timeout</code>
       time, issue a retransmission command.
     */
    public void run() {
	Entry   entry;
	long    current_time=0, min_time, time_to_wait, already_waited, earliest_wakeup;

	while(true) {
	    synchronized(msgs) {
		while(msgs.Size() < 1) {
		    try {
			msgs.wait();
		    }
		    catch(Exception ex) {}
		}
		while(msgs.Size() > 0) {
		    current_time=System.currentTimeMillis();
		    earliest_wakeup=current_time + timeout + 1;  // greater than any timeout value
		    
		    for(Enumeration e=msgs.Elements(); e.hasMoreElements();) {
			entry=(Entry)e.nextElement();
			if(!entry.ack_received) {
			    if(entry.wakeup_at <= current_time) {
				if(retransmit_command != null)
				    retransmit_command.Retransmit(entry.seqno, entry.msg, entry.num_tries);
				entry.num_tries++;
				entry.wakeup_at=current_time + timeout;
			    }
			    earliest_wakeup=Math.min(earliest_wakeup, entry.wakeup_at);
			}
		    }
		    
		    /* keep the time to wait in the range [min_wait_time .. timeout+min_wait_time] */
		    time_to_wait=earliest_wakeup-current_time;
		    time_to_wait=Math.max(time_to_wait, min_wait_time);
		    time_to_wait=Math.min(time_to_wait, timeout+min_wait_time);

		    try {
			//System.out.println("time_to_wait=" + time_to_wait);
			msgs.wait(time_to_wait); // gives others a chance to add / remove messages
		    }
		    catch(Exception ee) {}
		}
	    }
	}   
    }



    public String toString() {
	StringBuffer retval=new StringBuffer();
	retval.append(msgs);
	return retval.toString();
    }



    static class Dummy implements RetransmitCommand {
	public void Retransmit(long seqno, Message msg, int num_tries) {
	    System.out.println("Retransmit(" + seqno + ")");
	}
    }


    public static void main(String[] args) {
	AckSenderWindow win=new AckSenderWindow(new Dummy(), 3);


	

	win.Add(3, new Message());

	Util.Sleep(4000);

	win.Add(5, new Message());
	win.Add(4, new Message());
	win.Add(8, new Message());
	win.Add(9, new Message());
	win.Add(6, new Message());
	win.Add(7, new Message());
	win.Add(3, new Message());


	try {
	    Thread.currentThread().sleep(5000);
	    win.Ack(5);
	    win.Ack(4);
	    win.Ack(6);
	    win.Ack(7);
	    win.Ack(8);
	    win.Ack(6);
	    win.Ack(9);

	    Thread.currentThread().sleep(5000);
	    win.Ack(3);

	    Thread.currentThread().sleep(3000);
	    win.Add(10, new Message());
	    win.Add(11, new Message());
	    Thread.currentThread().sleep(3000);
	    win.Ack(10); win.Ack(11);

	    win.Add(12, new Message());
	    win.Add(13, new Message());
	    win.Add(15, new Message());
	    win.Add(16, new Message());

	    Util.Sleep(1000);
	    win.Ack(12);
	    win.Ack(13);
	    win.Ack(15);

	    
	    Util.Sleep(5000);
	    win.Ack(16);

	    Util.Sleep(1000);
	    win.Add(14, new Message());
	    Util.Sleep(1000);

	    win.Stop();
	}
	catch(Exception e) {
	    System.err.println(e);
	}
    }


//      public static void main(String[] args) {

//  	class Retr implements RetransmitCommand {

//  	    public void Retransmit(long seqno, Message msg, int num_tries) {
//  		System.out.println("--> Retransmit(" + seqno + ", " + num_tries + ")");
//  		// sender.Ack(seqno);
//  	    }	    

//  	}

//  	class Adder extends Thread {
//  	    AckSenderWindow s;
//  	    Message         mm=new Message();

//  	    public Adder(AckSenderWindow s) {this.s=s;}

//  	    public void run() {
//  		for(int i=1; i < 100; i++) {
//  		    s.Add(i, mm);
//  		    System.out.println("Add(" + i + ")");
//  		    try {
//  			sleep(100);
//  		    }
//  		    catch(Exception e) {
//  			System.err.println(e);
//  		    }
//  		}
//  	    }
//  	}



//  	class Acker extends Thread {
//  	    AckSenderWindow s;
//  	    Message         mm=new Message();
//  	    Random          rand=new Random(System.currentTimeMillis());
//  	    long            sleep_time=0;

//  	    public Acker(AckSenderWindow s) {this.s=s;}

//  	    public void run() {
//  		for(int i=1; i < 100; i++) {
//  		    System.out.println("ACK(" + i + ")");
//  		    s.Ack(i);
//  		    try {
//  			if(i > 50)
//  			    sleep_time=Math.abs(rand.nextLong() % 10) + 100;
//  			else
//  			    sleep_time=Math.abs(rand.nextLong() % 500) + 100;
//  			System.out.println("Acker: sleeping for " + sleep_time);
//  			sleep(sleep_time);
//  		    }
//  		    catch(Exception e) {
//  			System.err.println(e);
//  		    }
//  		}
//  	    }
//  	}



//  	AckSenderWindow  sender=new AckSenderWindow(new Retr(), 0);
//  	Message          m=new Message();


//  	sender.Start();
//  	Thread adder=new Adder(sender);
//  	Thread acker=new Acker(sender);
	
//  	adder.start(); acker.start();
//  	try {
//  	    adder.join();
//  	    acker.join();
//  	}
//  	catch(Exception e) {
//  	    System.err.println(e);
//  	}
//  	sender.Stop();
//      }

}
