import java.io.File;
import java.io.FileNotFoundException;
import java.util.HashMap;
import java.util.Map;
import java.util.Scanner;
import java.util.Iterator;

public class Tree {
    Variable root;
    
    /* Read the structure of the tree from a file and
     * initialize the variables in the tree.
     */
    public void loadStructure(String filename){
        try {
            Scanner sc = new Scanner(new File(filename));
            root = parseStructure(sc);
            root.setRootPrior();
        } catch (FileNotFoundException e) {
            System.err.println("File not found "+filename);
            e.printStackTrace();
            System.exit(1);            
        }
    }
    
    private static Variable parseStructure(Scanner sc){
        String name;
        int i,number;
        Variable t;
        name = sc.next();
        number = sc.nextInt();
        t = new Variable(name);
        for(i=0; i<number; i++){
            t.addChild(parseStructure(sc));
        }
        return t;
    }
    
    /* Read the values of the observed variables (the leaves)
     * from a file and clamp each leaf to its observed value. 
     */
    public void loadEvidence(String filename){
        try {
            Scanner sc = new Scanner(new File(filename));
            Map<String,Integer> m = parseEvidence(sc);
            clampEvidence(root,m);
        } catch (FileNotFoundException e) {
            System.err.println("File not found "+filename);
            e.printStackTrace();
            System.exit(1);            
        }
    }
    
    private static Map<String,Integer> parseEvidence(Scanner sc){
        Map<String,Integer> m = new HashMap<String,Integer>();
        while (sc.hasNext()){
            String name = sc.next();
            int value = sc.nextInt();
            m.put(name, value);
        }
        return m;
    }
    
    private static void clampEvidence(Variable node, Map<String, Integer> m) {
        if (m.containsKey(node.getName())){
            node.clamp(m.get(node.getName()));
        }
        Iterator<Variable> i=node.getChildren().iterator();
        while(i.hasNext()){
            clampEvidence(i.next(), m);            
        }        
    }
    
    /* Asks the children of v to recursively 
     * compute the messages they will send to v. 
     * Afterwards, it computes a message that
     * will be sent to v's parent.
     * If v is the root, it retuns the likelihood
     * of the data. Otherwise, the returned value
     * is not important.
     */
    private static float collect(Variable v){
        Iterator<Variable> i = v.getChildren().iterator();
        while(i.hasNext())
            collect(i.next());
        v.computeOutMsg();
        return v.getOutMsg()[0];
    }
    
    /* For every child of v, it computes the message from
     * v to that child and recursively asks the child to 
     * compute the message it will send to each of its 
     * own children.
     */    
    private static void distribute(Variable v) {
        Iterator<Variable> i = v.getChildren().iterator();
        while(i.hasNext()){
            Variable n =i.next();
            n.computeInMsg(v);
            distribute(n);
        }    
    }
    
    /* Compute the marginal probabilities of each
     * unobserved variable in the tree 
     * given the observed data.
     */
    private static void marginals(Variable v) {
        Iterator<Variable> i = v.getChildren().iterator();
        while(i.hasNext()){
            Variable n =i.next();
            marginals(n);
        }
        v.computeMarginal();
    }

    /* Sum-Product algorithm for infering various 
     * marginals in a graphical model.
     * Returns the likelihood of the data. 
     * Any previous messages passed are overwritten.
     */
    public float sumproduct(){
        float likelihood = collect(root);
        distribute(root);
        marginals(root);
        return likelihood;
    }
    
    /* Max sum algorithm (max product algorithm in log space)
     * for infering the most probable configuration
     * of the unobserved variables in a graphical model
     * Returns the maximized log probability. The most probable 
     * values of the unobserved variables are stored in the nodes 
     * of the tree in the field 'argmax'.
     * Any previous messages passed are overwritten.
     */
    public float maxsum(){
        float maxprob = maxcollect(root);
        root.maximizeRoot();
        maxdistribute(root,root.getArgmax());
        return maxprob;
    }
    
    /* This is similar to collect, except that
     * nodes send max-messages to their parent. 
     */
    private static float maxcollect(Variable v){
        Iterator<Variable> i = v.getChildren().iterator();
        while(i.hasNext())
            maxcollect(i.next());
        v.computeMaxMsg();
        return v.getOutMsg()[0];
    }
    
    /* Distributes the maximizing value in the parent to the children so 
     * that they can compute their maximizing value too.
     */
    private static void maxdistribute(Variable node, int parentArgmax) {
        node.setArgmax(node.backpointer[parentArgmax]);
        Iterator<Variable> it = node.getChildren().iterator();
        while(it.hasNext())
            maxdistribute(it.next(), node.getArgmax());
    }
    
    public void printmarginals(){
        printmarginal(root);
    }
    
    private static void printmarginal(Variable v){
        v.printmarg();
        Iterator<Variable> i = v.getChildren().iterator();
        while(i.hasNext())
            printmarginal(i.next());
    }
    
    public void printargmax(){
        printmax(root);
    }
    
    private static void printmax(Variable v){
        v.printmax();
        Iterator<Variable> i = v.getChildren().iterator();
        while(i.hasNext())
            printmax(i.next());
    }
        
    public static void main(String[] args){
        Tree t = new Tree();
        t.loadStructure("structure.txt");
        t.loadEvidence("observed.txt");
        //TODO: fill this in
    }
}
