/**
 * <p>Title: Shotgun Project</p>
 * <p>Description: Implementation of a fixed size cache for storing performances.</p>
 * @author M. Arthur Munson
 * @version 1.0
 * History:
 *  2005/03/03  Art Munson created.
 */

package shotgun.metrics;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import shotgun.Predictions;

/**
 * Class implementing a finite and fixed size cache.  The cache is indexed
 * by Predictions objects and stores performances for those predictions.
 * The cache uses a LRU replacement policy.
 *
 * Given that Preditions can occupy a lot of memory, it is recommended to make
 * the cache relatively small.  Otherwise 
 *   a) the cache will keep references to potentially many large objects that 
 *      could otherwise be garbage collected (risking exhausting RAM), and
 *   b) cache performance may suffer while maintaining LRU information.
 */
public class PerfCache
{
    /**
     * Recommended default capacity for the performance cache.
     */
    public static final int DEFAULT_CAPACITY = 10;

    private int capacity;  // Max # of entries in cache.
    private List queue;    // Storage for insertion order.
    private Map cache;     // Storage for (prediction,score) entries.

    /**
     * Constructs a performance cache with default capacity.
     */
    public PerfCache()
    {
	this(DEFAULT_CAPACITY);
    }

    /**
     * Constructs a performance cache that can remember up to
     * capacity entries.
     * @param capacity The length of the history stored in the cache.
     */
    public PerfCache(int capacity)
    {
	this.capacity = capacity;
	queue = new LinkedList();
	cache = new HashMap();
    }

    /**
     * Gets the cached performance for a set of predictions.
     * Running time is O(n).
     * @param pred The predictions to lookup.
     * @return The performance scores, or null if no cached values were found.
     */
    public double[] get(Predictions pred)
    {
	double[] scores = (double[])cache.get(pred);
	
	// If we found a cached value, we need to make sure that the
	// predictions are stored as the most recently used key. In other 
	// words, update the position if the predictions are not at the end
	// of the queue.
	boolean updatePosition = scores != null
	    && !pred.equals(queue.get(queue.size() - 1));

	if (updatePosition) {
	    // Update position in list to be last.
	    queue.remove(pred);
	    queue.add(pred);
	}

	return scores;
    }

    /**
     * Associate the given scores with the predictions.
     * Insertion takes O(1) if the predictions are not already in the cache;
     * otherwise the operation takes O(n).
     * @param pred The predictions to use as a key.
     * @param scores The performance scores to store with the key.
     */
    public void put(Predictions pred, double[] scores)
    {
	double[] old = (double[])cache.put(pred, scores);
	if (old != null) {
	    // First we need to remove the old cache entry.
	    queue.remove(pred);
	}
	queue.add(pred);

	// Guarantee that we have not exceeded our capacity.
	while (queue.size() > capacity) {
	    // Remove the entry that has been in the cache longest.
	    this.remove((Predictions)queue.get(0));
	}
    }

    /**
     * Removes the cached value for the predictions.  Running time is O(n).
     * @param pred The predictions to un-cache.
     */
    public void remove(Predictions pred)
    {
	double[] old = (double[])cache.remove(pred);
	if (old != null) {
	    queue.remove(pred);
	}
    }

    public String toString()
    {
	String desc = "";
	Iterator it = queue.iterator();
	while (it.hasNext()) {
	    Predictions p = (Predictions)it.next();
	    desc += p + "->" + cache.get(p) + "\n";
	}
	return desc;
    }
}
