/**
 * <p>Title: Shotgun Project</p>
 * <p>Description: Flexible F-measure implementation.</p>
 * @author M. Arthur Munson
 * @version 1.0
 * History:
 *  2005/02/23  Art Munson created.
 */

package shotgun.metrics;
import shotgun.Predictions;

/**
 * Implementation of F-measure that can compute more than just F1.
 */
public class FMeasure implements MetricBundle
{
  private static final double EPS = 1.0e-99;
  private double beta = 1.0;      // Default value.
  private String label = null;    // Cached name, for efficiency.
  private int recallIndex = -1;
  private int precIndex = -1;

  public FMeasure()
  { }

  ///////////////////////////////////////////////////////////
  // MetricBundle implementation.
  ///////////////////////////////////////////////////////////

  public int count()
  {
    return 1;
  }

  public boolean init(String arg)
  {
    recallIndex = Predictions.lookupMode("REC");
    precIndex = Predictions.lookupMode("PRE");
    if (recallIndex == -1 || precIndex == -1) {
      return false;
    }

    String [] params = arg.split(" ");
    int curr = 0;
    while (curr < params.length) {
      if (params[curr].equals("-beta")) {
        ++curr;
        beta = Double.parseDouble(params[curr]);
      }
      ++curr;
    }

    return true;
  }

  public void invalidateCache(Predictions pred)
  {
  }

  public String name(int i)
  {
    if (label == null) {
      label = "F" + Double.toString(beta);
    }
    return label;
  }

  public double performance(int i, Predictions pred)
  {
    double recall = pred.compute(recallIndex);
    double prec = pred.compute(precIndex);

    double denom = beta*beta*prec + recall;
    if (denom >= EPS) {
      double numer = (beta*beta + 1) * prec * recall;
      return numer / denom;
    }
    else {
      return 0.0;
    }
  }

  public double loss(int i, Predictions pred, double perf)
  {
    return 1.0 - perf;
  }

  public boolean smallerIsBetter(int i)
  {
    return false;
  }

  public boolean requiresStrictOrder(int i)
  {
    return false;
  }

  public boolean thresholdSensitive(int i)
  {
    return true;
  }
}

