/**
 * <p>Title: Shotgun Project</p>
 * <p>Description: Balanced loss implementation.</p>
 * @author M. Arthur Munson
 * @version 1.0
 * History:
 *  2005/02/23  Art Munson created.
 */

package shotgun.metrics;
import shotgun.Predictions;
import shotgun.CMatrix;

/**
 * Balanced loss performance metric.  The performance is equal to
 *
 *      BL = 1/2 [b/(a+b) + c/(c+d)]
 *
 * where a, b, c, and d correspond to the following confusion matrix:
 *
 *
 *                        MODEL PREDICTION
 *
 *                   |       1       0       |
 *             - - - + - - - - - - - - - - - + - - - - -
 *  TRUE         1   |       A       B       |    A+B
 * OUTCOME           |                       |
 *               0   |       C       D       |    C+D
 *             - - - + - - - - - - - - - - - + - - - - -
 *                   |      A+C     B+D      |  A+B+C+D
 *
 *
 *              1 = POSITIVE
 *              0 = NEGATIVE
 *
 * Basically balanced loss gives equal weight to the positive cases
 * and negative cases, regardless of the number of positives and
 * negatives.
 *
 * To use this metric in shotgun, add this option
 *   -addmetrics shotgun.metrics.BalancedLoss ""
 * To optimize to it, use -custom BL  (or bl).
 */
public class BalancedLoss implements MetricBundle
{
  private static final double EPS = 1.0e-99;
  private static String label = "BL";
  private static CMatrix cmatrix = new CMatrix();

  public BalancedLoss()
  { }

  ///////////////////////////////////////////////////////////
  // MetricBundle implementation.
  ///////////////////////////////////////////////////////////

  public int count()
  {
    return 1;
  }

  public boolean init(String arg)
  {
    return true;
  }

  public void invalidateCache(Predictions pred)
  {
  }

  public String name(int i)
  {
    return label;
  }

  public double performance(int i, Predictions pred)
  {
    pred.fillConfusionMatrix(cmatrix);
    double balancedLoss =
      ratio(cmatrix.falseNeg, cmatrix.truePos+cmatrix.falseNeg) +
      ratio(cmatrix.falsePos, cmatrix.trueNeg+cmatrix.falsePos);
    return balancedLoss * 0.5;
  }

  public double loss(int i, Predictions pred, double perf)
  {
    return perf;
  }

  public boolean smallerIsBetter(int i)
  {
    return true;
  }

  public boolean requiresStrictOrder(int i)
  {
    return false;
  }

  public boolean thresholdSensitive(int i)
  {
    return true;
  }

  ///////////////////////////////////////////
  // Helper methods
  ///////////////////////////////////////////

  /**
   * Helper function to compute the ratio of the numerator to the denominator.
   * @param num The numerator.  Should be >= 0.
   * @param den The denominator.  Should be >= 0.
   * @return num / den
   */
  private static double ratio(int num, int den)
  {
    double denom = den;
    if (denom >= EPS) {
      return (double)num / denom;
    }
    else {
      return 0.0;
    }
  }
}

