/**
 * <p>Title: Shotgun Project</p>
 * <p>Description: </p>
 * <p>Copyright: </p>
 * <p>Company: </p>
 * @author Alex Ksikes
 * @version 2.1
 **/

package shotgun;

import java.util.Random;
import java.io.FileWriter;
import java.io.IOException;
import shotgun.metrics.MetricBundle;

/**
 * A class encapsulating a single set of predictions.
 **/
public class Predictions
{
  // Symbolic constants used for looking up different kinds of
  // metrics.  MUST be kept in same order as the String array measure
  // defined below.
  private static final int
    ACC=0,
    RMS=1,
    ROC=2,
    ALL=3,
    BEP=4,
    PRE=5,
    REC=6,
    FSC=7,
    APR=8,
    LFT=9,
    CST=10,
    NRM=11,
    MXE=12,
    SLQ=13,
    BSP=14,
    CA1=15,
    CA2=16,
    CA3=17,
    CA4=18,
    CA5=19,
    CA6=20,
    CUSTOM=21;	// Must be last in list. Indicates start of user-defined metrics.

  private static final String[] measure = {
    "ACC",
    "RMS",
    "ROC",
    "ALL",
    "BEP",
    "PRE",
    "REC",
    "FSC",
    "APR",
    "LFT",
    "CST",
    "NRM",
    "MXE",
    "SLQ",
    "BSP",
    "CA1",
    "CA2",
    "CA3",
    "CA4",
    "CA5",
    "CA6",
    "CUSTOM"
  };

  private static final double eps=1.0e-99;

  private static int mode;                  // the performance measure we are hillclimbing on
  private static int bspMode;               // performance measure to do bootstrapping on
  private static Perf [] perfs = null;      // List of all perf metrics.

  private static double[] w;                // weights weighted combination of performances
  private static double weight=1;           // weight for weight decay
  private static double decayCst;           // decay cst for weight decay

  private static double globalThreshold=0.5;// threshold for perf measures
  private static double prcdata=-1;         // threshold as chosen percent of 1 of the data
  private static double costa=0, costb=0.5,	// cost for the confusion matrix
                        costc=0.5, costd=0;
  private static double norm=1;             // exponent of the norm perf measure

  private static int numBsp;                // num of bootstrap samples
  private static int numPts;                // num of randomly chosen instances of each bsp
  private static long seed;					// seed used to generate the bootstrap samples

  private static boolean verbose = false;	// print to screen?
  private static boolean lossMode = false;	// print in terms of loss?

  private static final int MAX_BINS=1000;
  private static Random slacqRandom;		// randome generator for slacq
  private static int slacqTrials;			// number of trials of slacq to run
  private static double slacqBinWidth;		// bin width for slacq
  private static int SlacqBins;				// depends on slacq_bin_width

  private static int calBinSize = 100;			// Calibration bin size

  private static final int MAX_TIES = 50000;
  private static double[] factCache = new double[MAX_TIES];


  // Instance variables
  private double a,b,c,d;          			// the confusion matrix
  private int size;                       	// number of examples
  private double[] probaClass;      		// probabilities of each example
  private Targets targets;					// the labels of each example
  private String setLabel;          // The name for the data set corresponding to predictions (e.g. test1).
  private boolean changed;  // Have the probabilities changed since last performance computation?
  // Have the probabilities changed since performance cached?
  // REVIEW: (mmunson) This is a stop gap until I move the builtin metrics to be custom metrics.
  private boolean perfCacheValid;
  private double perfCache;       // last computed value; only meant for builtin metrics
  private boolean dynamicThreshold; // set threshold dynamically?
  private Threshold modelThreshold; // threshold that is optimized per _model_


  /**
   * Builds a new set of predictions given the probabilities and the labels of each example.
   *
   * @param predictions An array containing the predictions of each example.
   * @param targets An array of the labels of each example.
   * @param setLabel Name for the data set that the predictions are over (e.g. test1).
   *    This is sometimes needed for complex custom metrics (e.g. coreference scoring).
  **/
  protected Predictions(double[] predictions, Targets targets, String setLabel)
  {
    this(targets, setLabel);
    for (int i=0; i<size; i++)
      probaClass[i]=predictions[i];
    changed = true;
    perfCacheValid = false;
    dynamicThreshold = false;
    modelThreshold = null;
  }

  /**
   * Accesses the probability values for the predictions.
   * Important: currently implemented to return reference to internal
   * data structure. DO NOT MODIFY this array.
   * @return The prediction values.
   */
  public double[] getPredictions()
  {
  	return probaClass;
  }

  /**
   * Copies the prediction's confusion matrix into the given
   * structure.
   * @param matrix The confusion matrix to fill.
   */
  public void fillConfusionMatrix(CMatrix matrix)
  {
    // TODO: replace internal variables with a CMatrix
    matrix.truePos = (int)a;
    matrix.falsePos = (int)c;
    matrix.falseNeg = (int)b;
    matrix.trueNeg = (int)d;
  }

  /**
   * Accesses the target values for the data.
   * Important: currently implemented to return reference to internal
   * data structure. DO NOT MODIFY this object.
   * @return The target values.
   */
  public Targets getTargets()
  {
    return targets;
  }

  public static void setVerbose()
  {
  	verbose=true;
  }

  public static void setLossMode()
  {
  	lossMode=true;
  }

  /**
   * Create an empty set of predictions.
   *
   * @param targets The targets of the predictions.
   * @param setLabel Name for the data set that the predictions are over (e.g. test1).
   *    This is sometimes needed for complex custom metrics (e.g. coreference scoring).
  **/
  protected Predictions(Targets targets, String setLabel)
  {
    this.targets=targets;
    this.setLabel=setLabel;
    this.size=targets.getSize();
    this.probaClass=new double[size];
    changed = true;
    perfCacheValid = false;
    dynamicThreshold = false;
    modelThreshold = null;
  }

  /**
   * Add the predictions of a model to these predictions.
   *
   * @param model The predictions to be added.
  **/
  protected void addWeighted(Predictions model, double oldWeight, double newWeight)
  {
    for (int i=0; i<size; i++)
      probaClass[i]=(probaClass[i]*oldWeight + model.probaClass[i]*newWeight*weight)/(oldWeight+newWeight);
    changed = true;
  }

  protected void addWeighted(Predictions model, int oldWeight, int newWeight)
  {
    for (int i=0; i<size; i++)
      probaClass[i]=(probaClass[i]*oldWeight + model.probaClass[i]*newWeight*weight)/(oldWeight+newWeight);
    changed = true;
  }

  /**
   * Add the predictions of a model to these predictions.
   *
   * @param model The predictions to be added.
  **/
  public void add(Predictions model, int oldWeight)
  {
    for (int i=0; i<size; i++)
      probaClass[i]=(probaClass[i]*oldWeight + model.probaClass[i]*weight)/(oldWeight+1);
    changed = true;
  }

  /**
   * Substract the predictions of a model from these predictions.
   *
   * @param model The predicitions to be substracted.
  **/
  protected void subWeighted(Predictions model, int oldWeight, int newWeight)
  {
    for (int i=0; i<size; i++)
      probaClass[i]=(probaClass[i]*oldWeight - model.probaClass[i]*newWeight*weight)/(oldWeight-newWeight);
    changed = true;
  }

  /**
   * Substract the predictions of a model from these predictions.
   *
   * @param model The predicitions to be substracted.
  **/
  protected void sub(Predictions model, int oldWeight)
  {
    for (int i=0; i<size; i++)
      probaClass[i]=(probaClass[i]*oldWeight - model.probaClass[i]*weight)/(oldWeight-1);
    changed = true;
  }

  /**
   * Computes a chosen performance of this set of predictions.
   * @param perf The performance measure.  Use {@link #lookupMode lookupMode}
   *   to find the correct ID for a metric.
   * @return The value of the performance.
   */
  public double compute(int perf)
  {
    if (changed)
    {
      Predictions.invalidateCaches(this);
      // Set the threshold for best performance on the hillclimb metric,
      // if we're doing dynamic thresholding and the hillclimb metric
      // depends on a threshold.
      if (dynamicThreshold && Predictions.thresholdSensitive(mode)) {
        setThresholdAutomatically(perfs[mode]);
      }
      computeCMatrix();
      changed = false;
    }

    Perf p = perfs[perf];
    return p.performance(this);
  }

  /**
   * Compute a chosen builtin performance of this set of predictions.
   *
   * IMPORTANT: only call from inside nested Perf class.  Otherwise you
   *  risk not caching a value for a target metric.
   *
   * @param perf The performance measure.
   * @return The value of the performance.
  **/
  private double computeBuiltin(int perf)
  {
    switch (perf)
    {
      case ACC:
        return computeACC();
      case RMS:
        return computeRMS();
      case ROC:
        return computeROC();
      case ALL:
        return computeALL();
      case BEP:
        return computeBEP();
      case PRE:
        return computePRE();
      case REC:
        return computeREC();
      case FSC:
        return computeFSC();
      case APR:
        return computeAPR();
      case LFT:
        return computeLFT();
      case CST:
        return computeCST();
      case NRM:
        return computeNRM();
      case MXE:
        return computeMXE();
      case BSP:
        return computeBSP();
      case SLQ:
        return computeSLQ();
      case CA1:
        return computeCA1();
      case CA2:
        return computeCA2();
      case CA3:
        return computeCA3();
      case CA4:
        return computeCA4();
      case CA5:
        return computeCA5();
      case CA6:
        return computeCA6();
      default:
        System.err.println("Warning: Predictions.computeBuiltin() passed unexpected perf ID");
        return 0.0;
    }
  }

  /**
   * Compute the loglikelihood to use with bayesian averaging. It
   * assumes that the targets are 0,1 and the predicions represent the
   * probability the model assignes to a case to be 1.
  **/

  public double computeLogLikelihood()
  {
    double loge = Math.log(Math.exp(1));
    double loglike = 0;
    for ( int i = 0; i < size; i++ )
    {
      int label = getLabel(i);
      double prob = probaClass[i];
      // Take the natural log of the current point's likelihood.
      // REVIEW: do we still need 1.0e-8 here to guard against log(0)?
      loglike += Math.log(label*prob + (1-label)*(1-prob) + 1.0e-8);
    }
    return loglike;
  }

  /**
   * Compute the confusion matrix.
  **/
  private void computeCMatrix()
  {
    a = b = c = d =0;
    double threshold = threshold();
    if(prcdata<0)
    {
      for (int item=0; item<size; item++)
      {
        if ( getLabel(item) == 1 )
        { // true outcome = 1
          if ( probaClass[item] >= threshold )
            a++;
          else
            b++;
        }
        else
        { // true outcome = 0
          if ( probaClass[item] >= threshold )
            c++;
          else
            d++;
        }
      }
    }
    /* set the threshold to predict x% of the cases to be 1's */
    else
    {
      // Make a copy of the true values and of the predictions to be sorted
      double[] pred=new double[size];
      int[] trueVal=new int[size];
      for (int i=0; i<size; i++)
      {
        pred[i]=probaClass[i];
        trueVal[i]=getLabel(i);
      }

      /* sort data by predicted value */
      Predictions.quicksort (0,(size-1),pred,trueVal);
      double pred_thresh=0;
      int data_split=size - (int)(size*prcdata/100);
      if(data_split <= 0)
        pred_thresh = pred[0] - 1.0;
      else if (data_split >= size)
        pred_thresh = pred[size-1]+1;
      else
        pred_thresh = (pred[data_split]+pred[data_split-1])/2.0;

      a=b=c=d=0;
      int item = size-1;
      while (item >= 0 && pred[item] > pred_thresh)
      {
        if (trueVal[item] == 1)
          a++;
        else
          c++;
        item--;
      }
      int no_remaining = item - data_split + 1;
      int cnt = 0; int onecnt = 0;
      while ( item >= 0 && Math.abs(pred[item] - pred_thresh)<=eps )  // I think this is safe
      {
        cnt++;
        if (trueVal[item] == 1)
          onecnt++;
        item--;
      }
      if (cnt>0)
      {
        a+= 1.0*no_remaining*onecnt/cnt;
        c+= 1.0*no_remaining*(cnt-onecnt)/cnt;
      }
      b=targets.getTotal_true_1()-a;
      d=targets.getTotal_true_0()-c;
    }
  }

  private void setThresholdAutomatically(Perf metric)
  {
    // Goal: find the threshold that yields the best performance w.r.t
    // the given metric.

    // Make a copy of the true values and of the predictions to be sorted
    double[] pred=new double[size];
    int[] trueVal=new int[size];
    for (int i = 0; i < size; ++i) {
      pred[i] = probaClass[i];
      trueVal[i] = getLabel(i);
    }

    // sort data by predicted value
    Predictions.quicksort(0, size-1, pred, trueVal);

    // Sweep the threshold across the range of predictions.  At each
    // point, compute the confusion matrix and get the metric's
    // performance.  Keep track of the best threshold / performance
    // seen.

    modelThreshold.value = pred[0] - 1.0;
    double bestThreshold = modelThreshold.value;
    computeCMatrix();
    metric.invalidateCache(this);
    double bestLoss = metric.loss(this, metric.performance(this));

    // index of first positive point
    int split = 0;
    double prevSplitVal = modelThreshold.value;

    do {
      // Find the next point with a predicted value > prev. split.
      prevSplitVal = pred[split];
      while (split < pred.length && pred[split] <= prevSplitVal) {
        // update confusion matrix while looking, since we know the
        // point at this split index will be classified as negative
        // under the new threshold
        if (trueVal[split] == 1) {
          // We lose one true positive and gain one false negative.
          --a; ++b;
        }
        else {
          // We lose one false positive and gain one true negative.
          --c; ++d;
        }
        ++split;
      }

      // Use the prediction for that point as the new threshold.  If
      // no such point, set threshold to make all points negative.
      modelThreshold.value =
        (split < pred.length) ? pred[split] : prevSplitVal + 1.0;

      // Compute performance with new threshold.
      metric.invalidateCache(this);
      double currLoss = metric.loss(this,metric.performance(this));

      // Update best threshold, if appropriate
      if (currLoss < bestLoss) {
        bestThreshold = modelThreshold.value;
        bestLoss = currLoss;
      }

    } while (modelThreshold.value <= pred[pred.length-1]);

    // Set the threshold to the best we found.
    modelThreshold.value = bestThreshold;
    metric.invalidateCache(this);
  }

  /**
   * Compute the accuracy performance.
  **/
  private double computeACC()
  {
    return (a+d)/(a+b+c+d);
  }

  /**
   * Compute the cost performance.
  **/
  private double computeCST()
  {
    return (costa*a+costb*b+costc*c+costd*d)/(a+b+c+d);
  }

  /**
   * Compute the precision performance.
  **/
  private double computePRE()
  {
    if (a+c<eps)
      return 0;
    return a/(a + c);
  }

  /**
   * Compute the recall performance.
  **/
  private double computeREC()
  {
    if (a+b<eps)
      return 0;
    return a/(a + b);
  }

  /**
   * Compute the lift performance.
  **/
  private double computeLFT()
  {
    if (a+c<eps)
      return 0;
    return (a/targets.getTotal_true_1()) * (size/(a+c));
  }

  /**
   * Compute the RMSE performance.
  **/
  private double computeRMS()
  {
    double sum=0;
    for (int i=0;i<size;i++)
    {
      sum+=(probaClass[i]-getLabel(i))*(probaClass[i]-getLabel(i));
    }
    return Math.sqrt(sum/size);
  }

  /**
   * Compute the norm performance.
   **/
  private double computeNRM()
  {
    double sum=0;
    for (int i=0;i<size;i++)
    {
      sum+=Math.pow(Math.abs(probaClass[i]-getLabel(i)),norm);
    }
    return Math.pow(sum/size,1.0/norm);
  }

  /**
   * Compute the mean cross entropy performance.
  **/
  private double computeMXE()
  {
    double loge=Math.log(Math.E);
    double sum=0;
    for (int i=0;i<size;i++)
    {
      sum+= getLabel(i)*Math.log(probaClass[i]+1.0e-8)/loge +
               (1.0-getLabel(i))*Math.log(1.0-probaClass[i]+1.0e-8)/loge; //cross-entropy
    }
    return -1*sum/size;
  }

  /**
   * Compute the ROC performance.
   **/
  private double computeROC()
  {
    // Make a copy of the true values and of the predictions to be sorted
    double[] pred=new double[size];
    int[] trueVal=new int[size];
    for (int i=0; i<size; i++)
    {
      pred[i]=probaClass[i];
      trueVal[i]=getLabel(i);
    }

    // sort data by predicted value
    Predictions.quicksort (0,(size-1),pred,trueVal);

    // now let's do the ROC curve and area
    int tt = 0;
    int tf = targets.getTotal_true_1();
    int ft = 0;
    int ff = targets.getTotal_true_0();

    double sens = ((double) tt) / ((double) (tt+tf));
    double spec = ((double) ff) / ((double) (ft+ff));
    double tpf = sens;
    double fpf = 1.0 - spec;
    double roc_area = 0.0;
    double tpf_prev = tpf;
    double fpf_prev = fpf;

    for (int item=size-1; item>=0; item--)
    {
      tt+= trueVal[item];
      tf-= trueVal[item];
      ft+= 1 - trueVal[item];
      ff-= 1 - trueVal[item];
      sens = ((double) tt) / ((double) (tt+tf));
      spec = ((double) ff) / ((double) (ft+ff));
      tpf  = sens;
      fpf  = 1.0 - spec;
      if ( item > 0 )
      {
        if ( pred[item] != pred[item-1] )
        {
          roc_area+= 0.5*(tpf+tpf_prev)*(fpf-fpf_prev);
          tpf_prev = tpf;
          fpf_prev = fpf;
        }
      }
      if ( item == 0 )
      {
        roc_area+= 0.5*(tpf+tpf_prev)*(fpf-fpf_prev);
      }
    }
    return roc_area;
  }

  /**
   * Compute break even point performance.
  **/
  private double computeBEP()
  {
      /* compute the Break-Even Point. We compute it as the Recall
	 when you predict total_true_1 cases as being positive cases
	 ties are split by fractional cases */

    // Make a copy of the true values and of the predictions to be sorted
    double[] pred=new double[size];
    double[] fraction = new double[size];
    int[] trueVal=new int[size];
    for (int i=0; i<size; i++)
    {
      pred[i]=probaClass[i];
      trueVal[i]=getLabel(i);
    }

      /* sort data by predicted value */
    Predictions.quicksort (0,(size-1),pred,trueVal);

    /* get the fractional weights. If there are ties we count the number
       of cases tied and how many positives there are, and we assign
       each case to be #poz/#cases positive and the rest negative */

    int item = 0;
    int begin = 0;
    int no_poz = 0;
    int count = 0;
    while ( item < size )
	{
	    begin = item;
	    count = 1;
	    if ( trueVal[item] == 1 ) no_poz = 1;
	    else no_poz = 0 ;
	    while ( ( item < size - 1 ) && ( Math.abs( pred[item] - pred[item+1]) < 1e-15 ) )
		{
		    item++;
		    count++;
		    if ( trueVal[item] == 1 ) no_poz ++;
		}
	    for ( int i = begin; i <= item; ++i )
		fraction[i] = no_poz*1.0/count;
	    item++;
	}

    /* calculate the recall performance at the break even point */
    double tt = 0;
    for ( item = size - 1; item >= targets.getTotal_true_0(); --item )
	tt+=fraction[item];
    return tt*1.0/targets.getTotal_true_1();

//     int tt = 0;
//     int tf = targets.getTotal_true_1();
//     int ft = 0;
//     int ff = targets.getTotal_true_0();

//     double precision = -1;
//     double recall = 0.0;
//     double precision_prev = precision;
//     double recall_prev = recall;

//     int cnt = 0;
//     int onecnt = 0;
//     int lasttt=tt;
//     int lasttf=tf;
//     int lastft=ft;
//     int lastff=ff;
//     double pr_break_even=0;

//     for (int item=size-1; item>=targets.getTotal_true_0(); item--)
//     {
//       cnt++;
//       if ( trueVal[item] == 1)
//         onecnt++;

//       tt+= trueVal[item];
//       tf-= trueVal[item];
//       ft+= 1 - trueVal[item];
//       ff-= 1 - trueVal[item];

//       if ( ( item > 0 && pred[item] != pred[item-1] ) || item == 0 )
//       {
//         double prcones = ((double)onecnt/cnt);
//         for (int i=1;i<=cnt;i++)
//         {
//           precision = (lasttt+i*prcones)/(lasttt+lastft+i);
//           recall    = (lasttt+i*prcones)/targets.getTotal_true_1();
//           if ( precision == recall && pr_break_even == 0 )
//           {
//             pr_break_even = precision;
//             return pr_break_even;
//           }
//           if ( precision < recall && precision_prev > recall_prev && pr_break_even == 0 )
//             if (recall != recall_prev)
//             {
//               pr_break_even = (precision - (precision_prev - precision)/(recall_prev - recall)*recall)/(1.0 - (precision_prev - precision)/(recall_prev - recall));
//               return pr_break_even;
//             }
//             else
//             {
//               pr_break_even = recall;
//               return pr_break_even;
//             }
//             precision_prev = precision;
//             recall_prev = recall;
//         }
//         cnt = 0;
//         onecnt = 0;
//         lasttt=tt;
//         lasttf=tf;
//         lastft=ft;
//         lastff=ff;
//       }
//     }
//     return pr_break_even;
  }

  /**
   * Compute the fscore performance.
  **/
  private double computeFSC()
  {
    double precision=computePRE();
    double recall=computeREC();
    if (precision + recall <= eps)
      return 0;
    else
      return 2*precision*recall/(precision+recall);
  }

    /* helper functions for computing the average precision */

    private double chooseLog(int n, int m){
	if(m < 0 || m > n)return -1.0E300;
	double ret = factCache[n] - factCache[m] - factCache[n-m];
	return ret;
    }
    private double apr_prob(int j, int k, int n, int m){
	if(j-1 < j-k-1)return 0;
	if(n-j < m-k-1)return 0;
	double ret = 0;
	ret += chooseLog(j-1,k);
	ret += chooseLog(n-j,m-k-1);
	ret -= chooseLog(n,m);
	if(ret > -50)
	    return Math.exp(ret);
	else
	    return 0;
    }
    /*
     * a is the number of cases before the tie
     * b is the number of positives before the tie
     * n is the number of cases in the tie
     * m is the number of positives in the tie
     */
    private double apr_ties(int a, int b, int n, int m){
	int j,k;
	double ret = 0;
	if(m==0)return 0;
	//code works for this case, but this way is faster
	if(n==1)return (b + 1.0) / (a + 1.0);
	if(n >= MAX_TIES){
	    System.err.println("Warning: There are more than " + MAX_TIES + " ties in your prediction, using pessimistic ordering in Average Precision.");
	    for(j = 1; j<=m; j++){
		ret += (b + j)*1.0 / (a + n - m + j);
	    }
	    return ret;
	}
	for(j = 1; j<=n; j++){
	    for(k = 0; k<j && k<=m; k++){
		double p = apr_prob(j,k,n,m);
		double contrib = (b + k + 1)*1.0 / (a + j);
		ret += p * contrib;
	    }
	}
	return ret;
    }


    /**
     * Compute the average precision performance.
     **/
    private double computeAPR()
    {
	/* now let's do Mean Average Precision*/
	/* note: the approach followed here may not be standard when it comes to ties */
	/* note: the approach followed here computes the area under the precision/recall
	   curve, not just the average precision at recall = 0,.1,.2,..,1.0 */


	// Make a copy of the true values and of the predictions to be sorted
	double[] pred=new double[size];
	int[] trueVal=new int[size];
	double[] fraction = new double[size];
	for (int i=0; i<size; i++)
	    {
		pred[i]=probaClass[i];
		trueVal[i]=getLabel(i);
	    }

	/* sort data by predicted value */
	Predictions.quicksort (0,(size-1),pred,trueVal);

    /* get the fractional weights. If there are ties we count the number
       of cases tied and how many positives there are, and we assign
       each case to be #poz/#cases positive and the rest negative */

    int item = 0;
    int begin = 0;
    int no_poz = 0;
    int count = 0;

    while ( item < size )
	{
	    begin = item;
	    count = 1;
	    if ( trueVal[item] == 1 ) no_poz = 1;
	    else no_poz = 0;
	    while ( ( item < size - 1 ) && ( Math.abs( pred[item] - pred[item+1]) < 1e-15 ) )
		{
		    item++;
		    count++;
		    if ( trueVal[item] == 1 ) no_poz ++;
		}
	    for ( int i = begin; i <= item; ++i )
		fraction[i] = no_poz*1.0/count;
	    item++;
	}

	double tt, tf, ft, ff, apr, precision, recall;
	double t_in_tie, f_in_tie, m, n;

	/* if all items negative, precision and recall are not well defined
	   so just return 0.0 and don't do PR plot */
	if ( targets.getTotal_true_1() == 0 || targets.getTotal_true_0() == 0 )
	    return 0.0;

	tt = 0;
	tf = targets.getTotal_true_1();
	ft = 0;
	ff = targets.getTotal_true_0();
	apr = 0;
	t_in_tie = 0;
	f_in_tie = 0;

	for ( item = size - 1; item >= 0; --item )
	    {
		tt += fraction[item];
		tf-= fraction[item];
		ft+= 1 - fraction[item];
		ff-= 1 - fraction[item];
		t_in_tie += fraction[item];
		f_in_tie += 1-fraction[item];
		if(item==0 || pred[item] != pred[item-1]){
		    n = tt + ft - f_in_tie - t_in_tie;
		    m = tt - t_in_tie;
		    apr += apr_ties((int)Math.round(n),(int)Math.round(m),(int)Math.round(t_in_tie+f_in_tie),(int)Math.round(t_in_tie));
		    t_in_tie = 0;
		    f_in_tie = 0;
		}
	    }
	return apr/targets.getTotal_true_1();
    }


//   private double computeAPR()
//   {
//       /* now do mean average PRECISION and find the BREAK_EVEN point */
//       /* note: the approach followed here may not be standard when it comes to ties */
//       /* note: the approach followed here computes the area under the precision/recall
//        curve, not just the average precision at recall = 0,.1,.2,..,1.0 */

//     // Make a copy of the true values and of the predictions to be sorted
//     double[] pred=new double[size];
//     int[] trueVal=new int[size];
//     for (int i=0; i<size; i++)
//     {
//       pred[i]=probaClass[i];
//       trueVal[i]=getLabel(i);
//     }

//       /* sort data by predicted value */
//     Predictions.quicksort (0,(size-1),pred,trueVal);

//     int tt = 0;
//     int tf = targets.getTotal_true_1();
//     int ft = 0;
//     int ff = targets.getTotal_true_0();

//     double apr = 0.0;

//     double precision = -1;
//     double recall = 0.0;
//     double precision_prev = precision;
//     double recall_prev = recall;

//     int cnt = 0;
//     int onecnt = 0;
//     int lasttt=tt;
//     int lasttf=tf;
//     int lastft=ft;
//     int lastff=ff;

//     for (int item=size-1; item>=0; item--)
//     {
//       cnt++;
//       if ( trueVal[item] == 1)
//         onecnt++;

//       tt+= trueVal[item];
//       tf-= trueVal[item];
//       ft+= 1 - trueVal[item];
//       ff-= 1 - trueVal[item];

//       if ( ( item > 0 && pred[item] != pred[item-1] ) || item == 0 )
//       {
//         double prcones = ((double)onecnt/cnt);
//         for (int i=1;i<=cnt;i++)
//         {
//           precision = (lasttt+i*prcones)/(lasttt+lastft+i);
//           recall    = (lasttt+i*prcones)/targets.getTotal_true_1();
//           if ( precision_prev > 0.0)
//             apr += 0.5*(precision+precision_prev)*(recall-recall_prev);
//           precision_prev = precision;
//           recall_prev = recall;
//         }
//         cnt = 0;
//         onecnt = 0;
//         lasttt=tt;
//         lasttf=tf;
//         lastft=ft;
//         lastff=ff;
//       }
//     }
//     return apr;
//   }

  /**
   * Computes a weighted combination of performances.
  **/
  private double computeALL()
  {
    return w[0]*computeACC()+w[1]*(1.0-computeRMS())+w[2]*computeROC();
  }

  /**
   * Computes bootstrapped performance.
  **/
  private double computeBSP()
  {
    double avePerf=0;
    Predictions bootstrap;
    for (int i=0; i<numBsp; i++)
    {
      bootstrap=getBootstrap(seed+i);
      avePerf+=bootstrap.compute(bspMode);
    }
    return avePerf/numBsp;
  }

  /**
   * Computes SLAC Q-Score
   */
  private double computeSLQ()
  {
	int[][] slacq_tb = new int[2][MAX_BINS];
	int[][] slacq_tbb= new int[2][MAX_BINS];
	double slac_w, slac_q;
	double slac_q_sum, slac_q_sumsquares;
	double slac_q_mean; // slac_q_sdev;

	/* calculate improved SLAC Q-score using two test set samples and evenly spaced bins */
	/* pb is predicted b, pbb is predicted bbar, tb is true b, tbb is true bbar, ...     */

	int bin,set, slac_qn;
	slac_q_sum = 0.0;
	slac_q_sumsquares = 0.0;

        /* for shotgun, replaced slq with non-randomized version to reduce variance */
        /* probably should do this everywhere! */

	for (bin=0; bin<SlacqBins; bin++)
	    {
		slacq_tb[0][bin]  = 0;
		slacq_tbb[0][bin] = 0;
	    }

	for (int i=0; i<size; i++)
	    {
		// we need to count cases with probaClass[i]==1 in the last bin
		bin = (int)Math.floor(probaClass[i] / (slacqBinWidth + eps));
		if ( getLabel(i) > 0.5 )
		    slacq_tb[0][bin]++;
		else
		    slacq_tbb[0][bin]++;
	    }

	slac_q = 0.0;
	slac_qn = 0;
	for (bin=0; bin<SlacqBins; bin++)
	    {
		if ( slacq_tb[0][bin] > slacq_tbb[0][bin] )
		    slac_w = 1.0 - ( (double)slacq_tb[0][bin]  / ( (double)(slacq_tb[0][bin]+slacq_tbb[0][bin]) + eps) );
		else
		    slac_w = 1.0 - ( (double)slacq_tbb[0][bin] / ( (double)(slacq_tb[0][bin]+slacq_tbb[0][bin]) + eps) );

		slac_q  += ((double)(slacq_tb[0][bin]+slacq_tbb[0][bin])) * (1.0 - 2.0*slac_w) * (1.0 - 2.0*slac_w);
		slac_qn += slacq_tb[0][bin]+slacq_tbb[0][bin];
	    }

	slac_q = slac_q / ((double)slac_qn);
	slac_q_mean = slac_q;

// 	for (int trial=0; trial<slacqTrials; trial++)
// 	{
// 		for (bin=0; bin<SlacqBins; bin++)
// 		{
// 			slacq_tb[0][bin]  = 0;
// 			slacq_tb[1][bin]  = 0;
// 			slacq_tbb[0][bin] = 0;
// 			slacq_tbb[1][bin] = 0;
// 		}

// 		for (int i=0; i<size; i++)
// 		{
// 			bin = (int)Math.floor(probaClass[i] / slacqBinWidth);
// 			set = slacqRandom.nextInt(2);
// 			if ( getLabel(i) > 0.5 )
// 				slacq_tb[set][bin]++;
// 			else
// 				slacq_tbb[set][bin]++;
// 		}

// 		slac_q = 0.0;
// 		slac_qn = 0;
// 		for (bin=0; bin<SlacqBins; bin++)
// 		{
// 			if ( slacq_tb[0][bin] > slacq_tbb[0][bin] )
// 				slac_w = 1.0 - ( (double)slacq_tb[1][bin]  / ( (double)(slacq_tb[1][bin]+slacq_tbb[1][bin]) + eps) );
// 			else
// 				slac_w = 1.0 - ( (double)slacq_tbb[1][bin] / ( (double)(slacq_tb[1][bin]+slacq_tbb[1][bin]) + eps) );

// 			slac_q  += ((double)(slacq_tb[1][bin]+slacq_tbb[1][bin])) * (1.0 - 2.0*slac_w) * (1.0 - 2.0*slac_w);
// 			slac_qn += slacq_tb[1][bin]+slacq_tbb[1][bin];
// 		}

// 		slac_q = slac_q / ((double)slac_qn);
// 		slac_q_sum += slac_q;
// 		slac_q_sumsquares += slac_q*slac_q;
// 	}

// 	slac_q_mean = slac_q_sum / ((double)slacqTrials);
//	if ( slacqTrials > 1 )
//		slac_q_sdev = Math.sqrt(slac_q_sumsquares/((double)slacqTrials) - slac_q_mean*slac_q_mean);
//	else
//		slac_q_sdev = 0.0;

	return slac_q_mean;
  }

  /**
   * @return
   */
  private double computeCA1()
  {
	int cal_f, cal_t, cal_sum_n;
	double sum_p, cal_sum, cal_sumsquares, obs_p, cal_err, plow, phigh;

	// Make a copy of the true values and of the predictions to be sorted
	double[] pred=new double[size];
	int[] trueVal=new int[size];
	for (int i=0; i<size; i++)
	    {
		pred[i]=probaClass[i];
		trueVal[i]=getLabel(i);
	    }

	/* sort data by predicted value */
	Predictions.quicksort (0,(size-1),pred,trueVal);

	cal_sum = 0.0;
	cal_sum_n = 0;
	for (int i=0; i<19; i++)
	{
		plow = i * 0.05;
		phigh = plow + 0.1;
		cal_f = 0;
		cal_t = 0;
		sum_p = 0.0;
		for (int item=0; item<size; item++)
		{
			if ( pred[item] >= plow && pred[item] <= phigh )
			{
				if ( trueVal[item] == 1 )
					cal_t++;
				else
					cal_f++;
				sum_p += pred[item];
			}
		}
		obs_p = ((double) cal_t) / (((double) (cal_t + cal_f)) + eps);
		cal_err = Math.abs(obs_p - (sum_p / (((double) (cal_t + cal_f)) + eps)));
		cal_sum += cal_err;
		cal_sum_n++;
	}

	return cal_sum / ((double) cal_sum_n);
  }

  /**
   * @return
   */
  private double computeCA2()
  {
	int cal_f, cal_t, cal_sum_n;
	double sum_p, cal_sum, cal_sumsquares, obs_p, cal_err, plow, phigh;

	// Make a copy of the true values and of the predictions to be sorted
	double[] pred=new double[size];
	int[] trueVal=new int[size];
	for (int i=0; i<size; i++)
	    {
		pred[i]=probaClass[i];
		trueVal[i]=getLabel(i);
	    }

	/* sort data by predicted value */
	Predictions.quicksort (0,(size-1),pred,trueVal);

	cal_sum = 0.0;
	cal_sum_n = 0;
	cal_f = 0;
	cal_t = 0;
	sum_p = 0.0;
	for (int item=0; item<size; item++)
	{
		if ( trueVal[item] == 1 )
			cal_t++;
		else
			cal_f++;
		sum_p += pred[item];
		if ( item >= (calBinSize - 1) )
		{
			obs_p = ((double) cal_t) / (((double) (cal_t + cal_f)) + eps);
			cal_err = Math.abs(obs_p - (sum_p / (((double) (cal_t + cal_f)) + eps)));
			cal_sum += cal_err;
			cal_sum_n++;
			if ( trueVal[item - calBinSize + 1] == 1 )
				cal_t--;
			else
				cal_f--;
			sum_p -= pred[item - calBinSize + 1];
		}
	}

	return cal_sum / ((double) cal_sum_n);
  }

  /**
   * @return
   */
  private double computeCA3()
  {
	return (computeCA1()+computeCA2())/2;
  }

  private double computeCA4()
  {
	return (9*computeCA2()+computeRMS())/10;
  }


  private double computeCA5()
  {
	int cal_f, cal_t, cal_sum_n;
	double sum_p, cal_sum, cal_sumsquares, obs_p, cal_err, plow, phigh;

	// Make a copy of the true values and of the predictions to be sorted
	double[] pred=new double[size];
	int[] trueVal=new int[size];
	for (int i=0; i<size; i++)
	    {
		pred[i]=probaClass[i];
		trueVal[i]=getLabel(i);
	    }

	/* sort data by predicted value */
	Predictions.quicksort (0,(size-1),pred,trueVal);

	cal_sum = 0.0;
	cal_sum_n = 0;
	cal_f = 0;
	cal_t = 0;
	sum_p = 0.0;
	for (int item=0; item<size; item++)
	{
		if ( trueVal[item] == 1 )
			cal_t++;
		else
			cal_f++;
		sum_p += pred[item];
		if ( item >= (calBinSize - 1) )
		{
			obs_p = ((double) cal_t) / (((double) (cal_t + cal_f)) + eps);
			cal_err = (obs_p - (sum_p / (((double) (cal_t + cal_f)) + eps)))*(obs_p - (sum_p / (((double) (cal_t + cal_f)) + eps)));
			cal_sum += cal_err;
			cal_sum_n++;
			if ( trueVal[item - calBinSize + 1] == 1 )
				cal_t--;
			else
				cal_f--;
			sum_p -= pred[item - calBinSize + 1];
		}
	}

	return cal_sum / ((double) cal_sum_n);
  }

  private double computeCA6()
  {
	return (computeCA2()+computeRMS())/2;
  }


  private Predictions getBootstrap(long seed)
  {
    double[] preds=new double[numPts];
    int[] targets=new int[numPts];
    Random index=new Random(seed);
    int pt;
    for (int i=0; i<numPts; i++)
    {
      pt=index.nextInt(size);
      preds[i]=probaClass[pt];
      targets[i]=getLabel(pt);
    }
    // There is no point passing a useful setLabel for the
    // bootstrap set, since it does not correspond to any pre-existing
    // set.  As a result, any custom metric that needs a set label to
    // score predictions cannot use the bootstrap option. Art Munson, 2004.
    Predictions bootstrap = new Predictions(preds, new Targets(targets), "");
    // Bootstrap should never dynamically search for best threshold.
    // If that is desired, the search is done at a higher level.  For
    // that to work, the bootstrap must share the same model
    // threshold.
    bootstrap.dynamicThreshold = false;
    bootstrap.modelThreshold = this.modelThreshold;
    return bootstrap;
  }

  /**
   * Updates the seed used for bootstrapping.
  **/
  public static void updateSeed()
  {
    Predictions.seed=seed+numBsp;
  }

  /**
   * Gets the truth label for instance i.
   * @param i The instance for which a label is desired.
   * @return 1 or 0, depending on i's target value.
   */
  public int getLabel(int i)
  {
    return targets.getTrueValue(i);
  }

  /**
   * Write this set of predictions to a file.
   *
   * @param out The file writer of the file.
  **/
  protected void write(FileWriter out)
  {
    try
    {
      for (int i=0; i<size; i++)
        out.write(probaClass[i]+"\n");
      out.flush();
      out.close();
    }
    catch (IOException e)
    {
      System.out.println("Error with write predictions!");
      System.exit(-1);
    }
  }

  /**
   * Create a new Predictions that is exactly the same as the current one
   */
  public Predictions copy()
  {
    Predictions pred=new Predictions(targets, setLabel);
    pred.perfCache=this.perfCache;
    pred.perfCacheValid = this.perfCacheValid;
    pred.dynamicThreshold = this.dynamicThreshold;
    pred.modelThreshold = this.modelThreshold;
    for (int i=0; i<size; i++)
      pred.probaClass[i]=this.probaClass[i];
    return pred;
  }

  public static void setThreshold(double threshold)
  {
    Predictions.globalThreshold=threshold;
  }

  public static void setPrcdata(double prcdata)
  {
    Predictions.prcdata=prcdata;
  }

  public static void setCost(double[] cost)
  {
    Predictions.costa=cost[0];
    Predictions.costb=cost[1];
    Predictions.costc=cost[2];
    Predictions.costd=cost[3];
  }

  public static void setNorm(double norm)
  {
    Predictions.norm=norm;
  }

  public static void setBsp(int numBsp, int numPts, long seed)
  {
    Predictions.numBsp=numBsp;
    Predictions.numPts=numPts;
    Predictions.seed=seed;
  }

  /**
   * Sets the predictions to optimize over bootstrapped data samples.
   * Call this instead of setMode() when bootstrapping should be used.
   * @param perfMode Non-null performance metric to optimize.  Must not
   *   be BSP.
   * @return True iff the perf metric was found.
   */
  public static boolean setBspMode(String perfMode)
  {
    // Sanity check that we were given a meaningful perf mode.
    if (measure[BSP].equalsIgnoreCase(perfMode)) {
      System.err.println("Error: trying to use nested bootstrapping.");
      return false;
    }

    // Use setMode() to locate the perf mode, then copy over to bspMode.
    boolean success = setMode(perfMode);
    Predictions.bspMode = Predictions.mode;
    Predictions.mode = BSP;

    return success;
  }

  public static void setSlacqBinWidth(double width)
  {
	slacqBinWidth=width;

	SlacqBins = (int)Math.floor(eps + 1.0/slacqBinWidth);
	if ( SlacqBins > MAX_BINS )
	{
		System.out.println("\nSLAC_Q bin width too small: " + slacqBinWidth);
		System.exit(-1);
	}
  }

  public static void setSlacqSeed(int seed)
  {
	Predictions.slacqRandom = new Random(seed);
  }

  public static void setSlacqTrials(int trials)
  {
	Predictions.slacqTrials=trials;
  }

    public static void computeFactCache()
    {
	factCache[0] = 0;
	for ( int i = 1; i < MAX_TIES; ++i )
	    factCache[i] = factCache[i-1] + Math.log(i);
    }

  /**
   * Sorts two aligned arrays in place.  Given two arrays whose
   * elements are implicitly aligned by index, sorts the values in
   * increasing order (according to the first array's values) while
   * maintaining the alignment.
   *
   * @param p Index of leftmost element to sort.
   * @param r Index of rightmost element to sort.
   * @param pred The array of predicted values.  This array determines the
   *   sort order.
   * @param trueVal The array of true values aligned with the predicted values.
   */
  public static void quicksort(int p, int  r, double[] pred, int[] trueVal)
  {
    int q;
    if (p < r)
    {
      q = partition (p,r,pred,trueVal);
      quicksort (p,q,pred,trueVal);
      quicksort (q+1,r,pred,trueVal);
    }
  }

  private static int partition(int p, int r, double[] pred, int[] trueVal)
  {
    int i, j;
    double tempf;
    int tempf2;
    double x = pred[p];
    i = p - 1;
    j = r + 1;

    while (true)
    {
      do j--; while (!(pred[j] <= x));
      do i++; while (!(pred[i] >= x));
      if (i < j)
      {
        tempf = pred[i];
        pred[i] = pred[j];
        pred[j] = tempf;
        tempf2 = trueVal[i];
        trueVal[i] = trueVal[j];
        trueVal[j] = tempf2;
      }
      else
        return j;
    }
  }

  /**
   * Checks if a metric considers low numbers better than high ones.
   * @param metricID The metric to check.
   * @return True iff the metric likes small numbers.
   */
  private static boolean smallMetric(int metricID)
  {
    return perfs[metricID].smallerIsBetter();
  }

  /**
   * Checks if a builtin metric considers low numbers to be better than high ones.
   * @param metricID The metric to check.
   * @return True iff the metric likes small numbers.
   */
  private static boolean builtinSmallerIsBetter(int metricID)
  {
    return
      metricID == RMS ||
      metricID == CST ||
      metricID == NRM ||
      metricID == MXE ||
      metricID == CA1 ||
      metricID == CA2 ||
      metricID == CA3 ||
      metricID == CA4 ||
      metricID == CA5 ||
      metricID == CA6;
  }

  /**
   * Checks if the metric's performance depends on a threshold.
   * @param metricID The metric to check.
   * @return True iff the metric is threshold sensitive.
   */
  public static boolean thresholdSensitive(int metricID)
  {
    if (metricID == BSP) {
      metricID = bspMode;
    }
    return perfs[metricID].thresholdSensitive();
  }

  /**
   * Checks if the builtin metric's performance depends on a threshold.
   * @param metricID The metric to check.
   * @return True iff the metric is threshold sensitive.
   */
  private static boolean builtinIsThresholdSensitive(int metricID)
  {
    return
      metricID == ACC ||
      metricID == ALL ||
      metricID == PRE ||
      metricID == REC ||
      metricID == FSC ||
      metricID == LFT ||
      metricID == CST;
  }

  /**
   * Return the performance of this model.
  **/
  protected double getPerformance()
  {
    return compute(mode);
  }

  private double builtinLoss(int mode, double performance)
  {
    // For some performance metrics smaller is better.
    // ie, for RMSE smaller is better
    // The loss for these is simply the performance value.

    boolean smallerBetter =
      smallMetric(mode) ||
      (mode == BSP && smallMetric(bspMode)); // Bootstrapping, check bootstrap mode.

    if (smallerBetter) {
  		return performance;
    }
    else if (mode==LFT) {
      double maxLift = ((double) targets.getTotal_true_1() + (double) targets.getTotal_true_0())
        / ((double) targets.getTotal_true_1() + eps);
      return (maxLift - performance) / (maxLift+eps);
    }
    else {
      // For the rest of the metrics the loss is how far the perf is
      // from the ideal (1).
      return 1-performance;
    }
  }

  public double getLoss(int mode, double performance)
  {
    return perfs[mode].loss(this, performance);
  }

  /**
   * Get the performance in terms of a loss
   */
  protected double getLoss()
  {
    return getLoss(mode,getPerformance());
  }

  /**
   * Print in a file and in stdout the performances of this model
   *
   * @param out The file writer of the file.
   * @param set The name of the test set the reported file is in.
   * @param msg name of the model to be outputted.
   * @param count The number to be outputted first.
  **/
  protected void report(FileWriter out, String set, String msg, int count)
  {
    try {
      double perf;
      Perf p = null;
      String label = null;

      for (int i = 0; i < perfs.length; ++i) {
        p = perfs[i];
        label = p.name();

        // bsp performance is expensive so only report in BSP mode
        boolean show = mode == BSP
          || i != BSP;
        if (show) {
          perf=compute(i);

          // If reporting baseline performance of F-score, we can
          // often get a better baseline by always predicting 1 for
          // each point.  This is a pretty gross hack, but I couldn't
          // find an elegant way to do it without major code
          // restructuring.
          if (i == FSC && msg.equals("baseline") && perf == 0.0) {
            double [] dummyVals = new double[probaClass.length];
            java.util.Arrays.fill(dummyVals, 1.0);
            Predictions dummyPred = new Predictions(dummyVals, targets, "dummy");
            dummyPred.dynamicThreshold = false;
            dummyPred.modelThreshold = this.modelThreshold;
            perf = dummyPred.compute(i);
          }

          if (lossMode)
            perf=getLoss(i,perf);

          String output = ""+count+" "+label+" "+perf+" "+msg;
          if (modelThreshold != null) {
            output += " "+threshold();
          }

          if (verbose)
            System.out.println(set+" "+output);
          out.write(output+"\n");
      	}
      }
      out.flush();
    }
    catch(IOException e){System.out.println(e);}
  }

  /**
   * Compare the performance of this model with another model.
   *
   * @param o The model to be compared.
   * @return 1 if this model has greater performance, 0 if equal and -1 otherwise.
  **/
  public int compareTo(Object o)
  {
    Predictions model=(Predictions) o;

    double delta = this.getPerformance() - model.getPerformance();

    int value;
    if (delta>0)
      value=1;
    else if (delta<=eps && delta>=-eps)
      value=0;
    else
      value=-1;

    // For some performance metrics smaller is better.
    // ie, for RMSE smaller is better

    boolean smallerBetter =
      smallMetric(mode) ||
      (mode == BSP && smallMetric(bspMode)); // Bootstrapping, check bootstrap mode.

    if (smallerBetter)
      return -1*value;
    else
      return value;
  }

  /**
   * Clear any caches maintained by performance metrics.
   * @param pred The predictions that have changed.
   */
  private static void invalidateCaches(Predictions pred)
  {
    for (int i = 0; i < perfs.length; ++i) {
      perfs[i].invalidateCache(pred);
    }
  }

  /**
   * Verify that instance reordering is allowed.  Some custom metrics will
   * not work if instances are shuffled, duplicated, or omitted.  This
   * should be checked before using bootstrap modes or cross-validation mode,
   * but after all the metrics have been loaded.
   * @return True if none of the metrics disallow reordering.
   */
  public static boolean reorderingAllowed()
  {
    boolean okay = true;
    for (int i = 0; okay && i < perfs.length; ++i) {
      okay = !perfs[i].requiresStrictOrder();
    }
    return okay;
  }

  /**
   * Check if reordering is allowed for the given metric.
   * @param metric - The metric index to check.
   * @return True iff reordering is allowed.
   */
  public static boolean reorderingAllowed(int metric)
  {
    if (metric == BSP) {
      metric = bspMode;
    }
    return !perfs[metric].requiresStrictOrder();
  }

  /**
   * Decay the weight that scales each added model.
  **/
  public static void decay()
  {
    weight=weight*decayCst;
  }

  /**
   * Set the decay constant.
  **/
  public static void setDecay(double decayCst)
  {
    Predictions.decayCst=decayCst;
  }

  /**
   * Set the default performance measure.
   *
   * @param mode The performance measure.
   * @return True iff the requested mode was found and set.
  **/
  public static boolean setMode(String mode)
  {
    // Make sure builtin perf metrics have been initialized.
    addBuiltinMetrics();

    // Locate the requested mode in the list of performance metrics.
    for (int i = 0; i < perfs.length; ++i) {
      Perf p = perfs[i];
      if (p.name().equalsIgnoreCase(mode)) {
  			Predictions.mode=i;
        return true;
      }
    }
    // Did not find requested mode.
    return false;
  }

  /**
   * Add custom metrics to the list of perfomance measures reported.
   * @param metricClassName The name of the java class defining the
   * measures.  The class must implement the {@link MetricBundle
   * MetricBundle} interface.
   * @param arg Argument string to be interpreted by the custom metric.
   * @exception ClassNotFoundException if the metric class was not found.
   * @exception InstantiationException if the metric class could not be instantiated
   *   / initialized.
   * @exception IllegalAccessException if ???.  Your guess is as good as mine. This
   *   comes from using the ClassLoader.
   */
  public static void addCustomMetrics(String metricClassName, String arg)
    throws ClassNotFoundException, InstantiationException, IllegalAccessException
  {
    // Make sure the builtin perf metrics have been initialized.
    addBuiltinMetrics();

    // Use reflection to load the specified class and create an instance.
    ClassLoader loader = ClassLoader.getSystemClassLoader();
    Class metricClass = loader.loadClass(metricClassName);
    MetricBundle metric = (MetricBundle)metricClass.newInstance();
    if (!metric.init(arg))
    {
      throw new InstantiationException("Failed to intialize " + metricClass + " with arg string " + arg);
    }

    // Need to grow list of performance metrics, copy the existing ones over, and
    // create the new user-defined ones.
    Perf [] old = perfs;
    perfs = new Perf[old.length + metric.count()];
    for (int i = 0; i < old.length; ++i) {
      perfs[i] = old[i];
    }
    for (int i = 0; i < metric.count(); ++i) {
      perfs[i + old.length] = new Perf(i, metric);
    }
  }

  /**
   * Guarantees that the builtin performance metrics have been initialized.
   */
  private static void addBuiltinMetrics()
  {
    if (perfs == null) {
      perfs = new Perf[CUSTOM];
      for (int i = 0; i < CUSTOM; ++i) {
        perfs[i] = new Perf(i, null);
      }
    }
  }


  /**
   * Set the weights for weighted combination of performances.
   *
   * @param weight The weights in the following order: ACC, RMS, and ROC.
  **/
  public static void setWeight(double[] weight)
  {
    Predictions.w=weight;
  }

  /**
   * Set the size of the bins used to measure probability calibration.
   * Or, it might be the number of bins to use in measuring probability calibration.
   * @param size The size of a bin.
   */
  public static void setCalibrationSize(int size)
  {
    calBinSize=size;
  }

  /**
   * Checks if mode specifies a custom metric.
   * @param mode The performance measure.
   * @return True iff mode is custom.
   */
  public static boolean isCustomMode(String mode)
  {
    for (int i = 0; i < CUSTOM; ++i) {
      if (measure[i].equalsIgnoreCase(mode)) {
        return false;
      }
    }
    return true;
  }

  /**
   * Finds the numeric code for the named mode / metric.
   * @param mode The name of the performance mode / metric.
   * @return The mode's numeric code, or -1 if the mode is not found.
   */
  public static int lookupMode(String mode)
  {
    addBuiltinMetrics();
    for (int i = 0; i < perfs.length; ++i) {
      if (perfs[i].name().equalsIgnoreCase(mode)) {
        return i;
      }
    }

    // Never found mode.
    return -1;
  }

  /**
   * Sets whether performance calculations should determine the
   * pos-neg threshold dynamically.
   * @param dynamic Determine threshold dynamically? Should only be
   * true for Predictions on the hillclimbing set.
   * @param threshold Storage for the dynamic threshold that will be
   * shared by the Predictions objects corresponding to the same
   * model.
   */
  public void setDynamicThreshold(boolean dynamic, Threshold threshold)
  {
    dynamicThreshold = dynamic;
    modelThreshold = threshold;
  }

  public double threshold()
  {
    if (modelThreshold == null) {
      return Predictions.globalThreshold;
    }
    else {
      return modelThreshold.value;
    }
  }

  /**
   * Gets the name of the test set corresponding to these predictions.
   * @return The name of the test set.
   */
  public String testName()
  {
    return setLabel;
  }


  /****************************************************************
   * Private class that serves as facade to make built-in and custom
   * metrics look the same.  This is essentially a proxy class.
   ****************************************************************/

  private static class Perf
  {
    private MetricBundle custom = null;
    private int offset = -1;

    public Perf(int offset, MetricBundle custom)
    {
	    this.offset = offset;
	    this.custom = custom;
    }

    public double performance(Predictions pred)
    {
      if (custom == null) {
        // Built-in metric.

        // Find out if this performance metric is the target
        // (hill-climbing) metric.
        boolean wantTargetMetric = (offset == pred.mode)
          || (pred.mode == pred.BSP && offset == pred.bspMode);

        // If it is, then the value might be cached for us.
        if (wantTargetMetric && pred.perfCacheValid) {
          return pred.perfCache;
        }

        // Otherwise, we need to do the work and compute it.
        double value = pred.computeBuiltin(offset);

        // Cache the value if need be.
        if (wantTargetMetric) {
          pred.perfCache = value;
          pred.perfCacheValid = true;
        }

        // Note: I considered putting the cache flag and storage for
        // the builtin metrics in this class.  That would require
        // adding storage for each metric though, most of which would
        // be wasted (only one metric is hill climbed per run).  At
        // first it seems that the variables could be made static.
        // This doesn't work in practice because there is one copy of
        // Perf per metric, shared across all the test sets.  For
        // example, one Perf object is assigned to handle ACC requests
        // for test1, test2, etc.  Static variables would result in
        // accidentally overwriting cached values from different data
        // sets, and not yield very good caching at all!
        //
        // The elegant fix is to convert the builtin metrics to be
        // custom metrics.  Not sure when that will happen though.

        return value;
      }
      else {
        return custom.performance(offset, pred);
      }
    }

    public double loss(Predictions pred, double performance)
    {
      if (custom == null) {
        // Built-in metric.
        return pred.builtinLoss(offset, performance);
      }
      else {
        return custom.loss(offset, pred, performance);
      }
    }

    public String name()
    {
      if (custom == null) {
        return Predictions.measure[offset];
      }
      else {
        return custom.name(offset);
      }
    }

    public boolean smallerIsBetter()
    {
      if (custom == null) {
        return Predictions.builtinSmallerIsBetter(offset);
      }
      else {
        return custom.smallerIsBetter(offset);
      }
    }

    public void invalidateCache(Predictions pred)
    {
      if (custom != null) {
        custom.invalidateCache(pred);
      }
      else {
        pred.perfCacheValid = false;
      }
    }

    public boolean requiresStrictOrder()
    {
      if (custom == null) {
        return false;
      }
      else {
        return custom.requiresStrictOrder(offset);
      }
    }

    public boolean thresholdSensitive()
    {
      if (custom == null) {
        return Predictions.builtinIsThresholdSensitive(offset);
      }
      else {
        return custom.thresholdSensitive(offset);
      }
    }
  } // end Perf class

}
