/**
 * @author Art Munson
 * @version 1.0
 */

package shotgun;

import java.io.File;

/**
 * Class to handle loading a model library.
 *
 * Provides hooks to customize loading via subclassing.  For example,
 * perhaps only a subset of the models should be loaded.
 *
 * Note: class is not public so it can only be accessed from the
 * shotgun package.
 */
class LibraryLoader implements ModelSampler
{
  private ModelSampler sampler;

  /**
   * Constructs a library loader that uses an optional model sampler
   * to customize which models get loaded.
   * @param _sampler A sampler which can be used to select a subset of
   * models to load.  If null is passed then load() will use all model
   * files passed to it.
   */
  public LibraryLoader(ModelSampler _sampler)
  {
    sampler = this;
    if (_sampler != null) {
      sampler = _sampler;
    }
  }

  /**
   * Loads the matrix of model files.
   * @param modelFiles First dimension indexes the test set; second
   * dimension indexes the actual file.  The file names should match
   * across test sets, modulo the file extension.  All of the files
   * listed should correspond to model predictions, and not the
   * targets file.
   * @return List of loaded models.
   */
  public final Model[] load(File[][] modelFiles)
  {
    String error = verifyFileNames(modelFiles);
    if (error != null) {
      System.out.println(error);
      System.exit(1);
    }

    modelFiles = sampler.prescreen(modelFiles);

    int numModels = modelFiles[0].length;
    int numSets = modelFiles.length;
    Model[] library = new Model[numModels];

    for (int j = 0; j < numModels; ++j) {
      String fileName = LibraryLoader.getModelName(modelFiles[0][j]);

      File[] testSets = new File[numSets];
      for (int i = 0; i < numSets; ++i) {
        testSets[i] = modelFiles[i][j];
      }

      library[j] = new Model(testSets);
    }

    return sampler.postscreen(library);
  }

  /**
   * Extracts the model name contained in a given file.
   * @param f The given file.
   * @return The model's name.  This is essentially f's name, minus
   * the file extension.
   */
  public static String getModelName(File f)
  {
    String name = f.getName();
    return name.substring(0, name.lastIndexOf('.'));
  }

  /**
   * Checks that the model names match across the different test sets.
   * This match ignores the file extensions.
   * @param modelFiles Matrix of file names.
   * @return null if all models match across sets; an error message
   * otherwise
   */
  public static String verifyFileNames(File[][] modelFiles)
  {
    int numSets = modelFiles.length;
    int numFiles = modelFiles[0].length;

    // Check that all sets have the same model name at the same
    // offset.  If this fails, calling code probably has a bug (maybe
    // forgetting to sort the files).
    for (int j = 0; j < numFiles; ++j) {
      String fileName = LibraryLoader.getModelName(modelFiles[0][j]);
      int length = fileName.length();

      for (int i = 1; i < numSets; ++i) {
        String f2name = LibraryLoader.getModelName(modelFiles[i][j]);
        if (!fileName.equals(f2name)) {
          String error = new String("Error: model names are not the same between all folders: ");
          error += modelFiles[0][j].getName() + " vs " + modelFiles[i][j].getName();
          return error;
        }
      }
    }

    // If we make it here everything is fine.
    return null;
  }

  /********************************************************
   * ModelSampler implementation
   ********************************************************/

  public File[][] prescreen(File[][] modelFiles)
  {
    return modelFiles;
  }

  public Model[] postscreen(Model[] models)
  {
    return models;
  }
}
