#!/usr/bin/perl -w

=head1 NAME

svm.lib.rangescale.pl - Linearly scales SVM model library to predict in [0,1].

=head1 SYNOPSIS

svm.lib.rangescale.pl [options] unscaledLib scaledLib

  Options:
    -help|?   brief help message
    -man      full documentation

=head1 DESCRIPTION

Copies the unscaled SVM model library and scales the model predictions to fall in the range [0,1].  The scaling is linear but always forces 0 to be scaled to 0.5.  In other words, if [min,max] are the range of the predictions for a given model (across the train, test1, etc subsets of the data), and maxAbs = max(abs(min),abs(max)), then the values are mapped from [-maxAbs,+maxAbs] to [0,1].

The argument unscaledLib should be a folder containing train/, test1/, etc. subfolders.  Each subfolder should only contain SVM models and possibly a targets file (the targets file, if present, will be copied but not scaled).  The scaled models are output to scaledLib.

=cut

#####################
# Implementation
#####################

use strict;
use Getopt::Long;
use Pod::Usage;
use FindBin;

my $scripts = $FindBin::Bin;
my $scale = "rangescale.pl";

my $help = 0;
my $man = 0;

# Parse options and print usage if there is a syntax error.
GetOptions("help|?" => \$help,
	   "man" => \$man)
    or pod2usage(-verbose => 0);

pod2usage(-verbose => 1) if $help;
pod2usage(-verbose => 2) if $man;

my $oldlib = shift @ARGV
    or pod2usage("$0: missing old SVM model library argument");
my $newlib = shift @ARGV
    or pod2usage("$0: missing new SVM model library argument");

# Copy the old library to the new one.
`cp -pr $oldlib $newlib`;

# Scale all the models except targets

my @series = enumerateSeries($newlib);
my @models = enumModels($newlib, \@series);
my $suffix = "temp";

for my $m (@models) {
    # Skip targets
    next if ($m eq "targets");

    # Create full-path list of matching models (across series).
    my @files;
    for my $s (@series) {
	push(@files, "$newlib/$s/$m.$s");
    }

    # Scale 'em
    `$scale -zfix -suffix $suffix @files`;

    # Rename the scaled models.
    for my $s (@series) {
	`mv $newlib/$s/$m.$s.$suffix $newlib/$s/$m.zscale.$s`;
    }

    # Remove the unscaled models.
    `rm @files`;
}


#############################
# Helper functions
#############################

sub enumerateSeries {
    my ($libFolder) = @_;
    my @series = split(' ', `ls $libFolder`);

    # Everything in the libfolder should be a subfolder that is
    # readable . . . let's check and make sure.
    for my $s (@series) {
	if (! (-d "$libFolder/$s" && -r "$libFolder/$s")) {
	    pod2usage("$0: $s is not a folder or is not readable");
	}
    }

    return @series;
}

sub enumModels {
    my ($libFolder, $series) = @_;

    my $s = $series->[0];
    my @models = split(' ', `ls $libFolder/$s`);

    # Strip off the trailing series extension from the models.
    for my $m (@models) {
	if ($m =~ /(.+)\.([^\.]+)$/ && $s eq $2) {
	    $m = $1;
	}
	else {
	    pod2usage("error: $m does not end in $s");
	}
    }

    return @models;
}
