#!/usr/bin/perl -w

#####################
# Documentation
#####################

=head1 NAME

rangescale.pl - Scales input numbers to given range.

=head1 SYNOPSIS

rangescale.pl [options] file1 [file2 file3 ...]

  Options:
    -bottom n        use n as bottom of range instead of 0
    -fix "old=new"   scale so that the old value maps to the new value
    -help|?          brief help message
    -man             full documentation
    -suffix s        append suffix .s to output files instead of .rscaled
    -top n           use n as top of range instead of 1
    -zfix            scale so that 0 on old scale maps to middle of new scale

=head1 DESCRIPTION

Reads numbers from given file(s) and scales them to [0,1] or a given range.  The file format should be one number per line.  When multiple files are specified the values are read from all files to find the starting range of values.  This puts all the files on the same [0,1] scale.

=head1 OPTIONS

=over 3

=item B<-bottom> n

Use n as the bottom of the new range instead of 0.

=item B<-fix> "old=new"

Force the scaling to map the given old value to the new value.  For example, if [min,max] is the range of values before scaling, and -fix "0=0.5" is specified, then [min,0] and [0,max] are linearly scaled to [0,0.5] and [0.5,1], respectively.  (Assuming -top and -bottom are not specified.)

=item B<-help|?>

Prints a brief help message and exits.

=item B<-man>

Prints the manual page and exits.

=item B<-suffix> s

Write the rescaled files with suffix s instead of rescaled.

=item B<-top> n

Use n as the top of the new range instead of 1.

=item B<-zfix>

Force the scaling to map zero to the middle of the new range while also maintaining a linear transformation across the whole range.  This is done by finding the [min,max] range of the input data, and determining the larger of abs(min) and abs(max).  Call this value x.  The values are then scaled from [-x,x] to [0,1].

=back

=cut

#####################
# Implementation
#####################

use strict;
use POSIX;
use Getopt::Long;
use Pod::Usage;

my $bottom = 0;
my $help = 0;
my $man = 0;
my $fix = "";
my $zfix = 0;
my $suffix = "rscaled";
my $top = 1;

# Parse options and print usage if there is a syntax error.
GetOptions("bottom=n" => \$bottom,
	   "fix=s" => \$fix,
	   "help|?" => \$help,
	   "man" => \$man,
	   "suffix=s" => \$suffix,
	   "top=n" => \$top,
	   "zfix" => \$zfix)
    or pod2usage(-verbose => 0);

pod2usage(-verbose => 0) if $help;
pod2usage(-verbose => 2) if $man;
pod2usage("$0: no input file(s) given") if (scalar(@ARGV) < 1);

# Only allow one kind of point fixing.
pod2usage("$0: only specify one of -fix or -zfix")
    if ($fix ne "" && $zfix);

# Initialize function pointer to do proper form of scaling.
my $scale = \&rangeScale;  # The default.
my $oldFix;   # used for -fix only
my $newFix;   # used for -fix only
if ($zfix) {
    $scale = \&zeroFixScale;
}
elsif ($fix ne "") {
    # Parse the value that should be fixed during scaling.
    my @pair = split('=', $fix);
    $oldFix = $pair[0];
    $newFix = $pair[1];
    pod2usage("$0: error: bad format for -fix option")
	if (!defined($oldFix) || !defined($newFix));
    $scale = \&fixRangeScale;
}

# Check that all the input files exist.
my @inputFiles = @ARGV;
for my $f (@inputFiles) {
    pod2usage("$0: $f does not exist") if (! -e $f);
}

# Make sure that top > bottom
pod2usage("$0: top must be > bottom") if ($top <= $bottom);

my $min = DBL_MAX;
my $max = DBL_MIN;


# Find the min/max across all the input files.
for my $input (@inputFiles) {
    open(INPUT, "<$input") or pod2usage("$0: could not open $input");
    while (my $val = <INPUT>) {
	chop $val;
	$min = $val if ($val < $min);
	$max = $val if ($val > $max);
    }
    close(INPUT);
}

die("$0: error: $max is not >= $min") if ($max < $min);

# Scale each of the files to fall in [bottom,top]

for my $input (@inputFiles) {
    open(INPUT, "<$input") or die("error: couldn't reopen $input");
    open(OUTPUT, ">$input.$suffix")
	or die("error: can't write to $input.$suffix");
    while (my $val = <INPUT>) {
	chop $val;
	$val = &$scale($val, $min, $max, $bottom, $top);
	print OUTPUT "$val\n";
    }
    close(OUTPUT);
    close(INPUT);
}

############################
# Helper functions
############################

sub rangeScale {
    my ($val, $min, $max, $bottom, $top) = @_;
    my $factor = scaleFactor($min, $max, $bottom, $top);
    return ($val - $min) * $factor + $bottom;
}

sub fixRangeScale {
    my ($val, $min, $max, $bottom, $top) = @_;
    if ($val < $oldFix) {
	return rangeScale($val, $min, $oldFix, $bottom, $newFix);
    }
    else {
	return rangeScale($val, $oldFix, $max, $newFix, $top);
    }
}

sub zeroFixScale {
    my ($val, $min, $max, $bottom, $top) = @_;
    my $minAbs = abs($min);
    my $maxAbs = abs($max);
    my $bigAbs = ($maxAbs > $minAbs) ? $maxAbs : $minAbs;

    return rangeScale($val, 0-$bigAbs, $bigAbs, $bottom, $top);
}

# pre: max > min and top > bottom
sub scaleFactor {
    my ($min, $max, $bottom, $top) = @_;
    my $EPS = 1.0e-7;

    # Compute width of gap between min and max
    my $oldWidth = $max - $min;
    # Note: this assumes that max >= min
    $oldWidth = $EPS if ($oldWidth < $EPS);
    # Compute width of gap between top and bottom
    my $newWidth = $top - $bottom;

    my $factor = $newWidth / $oldWidth;
    return $factor;
}

