###################################################
# ShotgunScaler:
#
# Object that can parse a scales file and scale shotgun performance
# scores according to that scale file.  Public interface:
#
#  initialized()
#  readScaleFile()
#  readScaleLines()
#  normalize()
#
# See comments by each method for how to use them.  The scales file
# should have the following format (lines starting with # are treated
# as comments):
#
#    # metric bottom top
#    acc 0.24817 0.14536
#    fsc 1.00000 0.30938
#    ...
#
# Note that inverted scales where low numbers are good and high
# numbers are bad (such as error or loss measurements) are acceptable.
# See the example above if this isn't clear.
#
# History:
#  2005/12/31   created by Art Munson
###################################################

package ShotgunScaler;
use strict;

# Private module variables
my $EPS = 1.0e-8;

# ShotgunScaler->new()
# Constructs and returns a scaler.
sub new {
    my $type = shift @_;  # Store the package name.
    my ShotgunScaler $self = {
	file => "",
	scales => {}
	};
    bless($self, $type);
    return $self;
}

# initialized()
# pre: true
# post: returns true if scaler is ready to normalize scores
#  (true after readScaleFile() called)
sub initialized {
    my ShotgunScaler $self = shift @_;
    my $file = $self->{file};
    return defined($file) && ($file ne "");
}

# readScaleFile(file)
# Reads in the designated scale file.
# pre: file has correct scale format
sub readScaleFile {
    my ShotgunScaler $self = shift @_;
    my ($file) = @_;

    if (!($file eq "") && -e $file) {
        # Read in the scaling data.
        my @lines = `cat $file`;
	$self->readScaleLines(@lines);
    }
    else {
	$self->{scales} = {};
    }

    $self->{file} = $file;
}

# readScaleLines(lines)
# Reads in scale information from array (ref) of lines, with one metric per
# line.  This is functionally identical to readScaleFile() but with a
# different input source.
sub readScaleLines {
    my ShotgunScaler $self = shift @_;
    my @lines = @_;

    my %scales;
    my $lossCount = 0;
    my $nonLossCount = 0;

    # For each non-comment line, map the metric name to a (bottom,
    # top) pair.
    for my $line (@lines) {
	my @fields = split(' ', $line);
	next if ($fields[0] eq "#");
	my @range = ($fields[1], $fields[2]);
	$scales{$fields[0]} = \@range;
	if ($range[0] >= $range[1]) {
	    ++$lossCount;
	}
	else {
	    ++$nonLossCount;
	}
    }

    $self->{scales} = \%scales;
    $self->{file} = "FCTN_CALL";
    $self->{lossScale} = $lossCount >= $nonLossCount;
}

# normalize(score, metric)
# Normalizes the given metric's score.
# pre: initialized();
#      metric was present in scale file read
# post: returns a normalized score where bigger is always better
sub normalize {
    my ShotgunScaler $self = shift @_;
    my ($score, $metric) = @_;

    # Look up scale range.
    my $range = $self->{scales}->{$metric};
    if (!defined($range)) {
	my $f = $self->{file};
	die("$0: cannot scale $metric; was not present in scale file $f");
    }

    my $min = $range->[0];
    my $max = $range->[1];
    my $denom = $max - $min;
    if (abs($denom) <= $EPS) {
	$denom = $EPS;
	$denom *= -1 if ($self->{lossScale});
    }

    $score = ($score - $min) / $denom;

    return $score;
}

1;
