

/* Copyright (c) 1991, 1992 Cornell University.  All Rights Reserved.
 *
 * Use, reproduction and preparation of derivative works of this software is
 * permitted.  Any copy of this software or of any derivative work must
 * include both the above copyright notices of Cornell University and this
 * paragraph.  This software is made available AS IS, and Cornell University
 * disclaims all warranties, express or implied.
 */



 
/* clarg.h */
 
/* Powerful command line argument interface for C programs */
 
#ifndef _INC_CLARG_
#define _INC_CLARG_
 
 
/* Written by Greg Klanderman    --    Summer 1991 */
 
 
/*
 
 USAGE:		compile with gcc -DDEBUG for debugging

 This is mostly wrong due to later changes.
 
  CL_INIT() must be called to initialize command line arg processing.
  It returns 0 upon successful completion, 1 if an error occurs.
  It must be passed ARGC and ARGV as indicated.  USAGE is a (possibly
  long) string containing one and only one "%s".  USAGE is printed
  if an error is encountered in command line argument parsing.  The "%s"
  is replaced by the program name.  INIT is an array of structures
  to specify the format of command line argument processing.
  Its fields hold: NAME - the name of the command line option.
                   MIN - the min # args the option can take if specified.
                   MAX - the max # args the option can take if specified.
                   REQUIRE - boolean to tell if the option is required.
		   TYPE - an int to tell what type the arguments are.
  The array must be terminated with an entry with NAME = NULL.  This
  entry marks the end of the list and also tells how processing should
  proceed for "unmatched"/"unnamed" arguments. For example, in 
  "cc -o prog prog.c a.c b.c", "prog.c", "a.c", and "b.c" are unnamed.
  REQUIRE now has slightly different meaning: if not(REQUIRE), zero
  unnamed arguments are allowed along with the usual MIN..MAX.
  For the option names, a period in the string marks the part before
  the period as being the shortest prefix required to generate
  that option.  For example, "z.oom" says that all of "z", "zo",
  "zoo", and "zoom" will generate the "zoom" option.  CL_INIT() does not 
  check to make sure no two option names have the same prefix. There 
  is no conflict having "z.oom" and "za.p", but adding "za.t" would
  be bad.
 
  eg:  CLINIT init[] = {{"a",  1, 1, 0, CL_NOTYPE},
			{"b",  0, 0, 0, CL_NOTYPE},
		        {"c",  1, 2, 1, CL_REAL},
		        {NULL, 2, 4, 0, CL_STRING}};
  allows option "-a" to take one and only one argument, "-b" is a switch
  "-c" can take one or two arguments and must be specified. There can be
  0, 2, 3, or 4 "unnamed" arguments. Types are as specified.
 
  CL_OPT() is a boolean function that returns 1 iff the option STR was specified.
  if STR == NULL, cl_opt() tells if there were any unnamed arguments.
 
  CL_ARG() returns the N th argument for option STR. If STR == NULL,
  the N th unnamed argument is returned.  The return is CL_ARGDNE
  if the argument specified does not exist.
 
  CL_NUMARG() tells how many arguments were specified for option STR,
  or how many unnamed arguments there were if STR == NULL.  The return
  is CL_NUMERR if the option was not specified.
 
  CL_ARG1(), CL_ARG2(), CL_ARG3(), CL_UNNAMED(), and CL_NUMUNNAMED()
  should be self explanatory.
 
  CL_TYPCHKFN() allows a new function FN to be used to do type checking
  for the type TYPE.  The return is the old type checking function.
  This function should be called before CL_INIT().
  There are default functions for all the types (0..19).
 
  Typechecking functions return 0 if ok, 1 if not. They take a string
  argument.
 
 
 PARSING:
  Look at ARGV[1..ARGC-1]. Any string starting with '-' specifies 
  one or more options.  The option name list given in INIT is searched for 
  the longest matching option name prefix.  When the proper option is matched
  its arguments are extracted from the ARGV[]'s that follow.  Finding
  an ARGV[] starting with '-', exceeding MAX, or finding an 
  argument of the wrong type terminates extraction
  of arguments for the current option.  Any remaining characters in 
  the original string that the option name was found in are now
  used as the next option name (more than one option name allowed per
  "-string"), and its args are gotten.  Any ARGV[n] that doesn't
  start with '-' and is not used as an argument is considered "unnamed".
 
  The option "-!" is designated the help option - USAGE is printed with no
  error message.
 
  "-" alone allows a stop to a variable # of arguments for an option.
  eg:  "xyz -a 2.0 - 3.0" forces only one arg for "-a" and makes 3.0 
  unnamed.
 
*/
 
 
/* definitions for command line arg processing */
 
/*
#define TRUE  1
#define FALSE 0
*/
 
typedef int bool;
typedef char *(*CL_TYPCHKFN)(char *s, long n);		/* type for a typecheck fn (ptr) */
 
 
typedef struct		/* structure used to initialize clarg */
{
  char *name;		/* name of option:  "-abcd" => name = "abcd" */
  long  min;		/* min number of arguments to this option */
  long  max;		/* max number of arguments to this option */
  CL_TYPCHKFN  typchk;	/* type check fun or NULL for no check */
  long  type; 		/* standard(0) or -Pcalumny(1) */
} CLINIT;
 
 
extern bool  cl_init(long argc, char **argv, char **envp, CLINIT *init,
		     char **depend, char *usage, long restrict);
extern bool  cl_opt(char *str);
extern char *cl_arg(char *str, long n);
extern long  cl_numarg(char *str);
extern char *cl_notype(char *s, long n);
extern char *cl_real(char *s, long n);
extern char *cl_int(char *s, long n);
extern char *argstring(int argc, char **argv);

 
/* get args 1, 2, or 3 for option STR */
 
#define cl_arg1(str)	cl_arg(str, 1)
#define cl_arg2(str)	cl_arg(str, 2)
#define cl_arg3(str)	cl_arg(str, 3)
 
 
/* option name for unnamed arguments */
 
#define CL_UNNAMED NULL
 
 
/* get unnamed argument N */
 
#define cl_unnamed(n)	cl_arg(CL_UNNAMED, n)
#define cl_numunnamed() cl_numarg(CL_UNNAMED)
 
 
/* error returns */
 
#define CL_ARGDNE NULL	/* returned by cl_arg() if arg does not exist */
#define CL_NUMERR (-1)  /* returned by cl_numarg() if option not specified */
 
#define CL_NORESTRICT	0
#define CL_OPTPACK	1
#define CL_NOPACK	2
 
#define CL_STDARG	0
#define CL_XXXARG	1
 
 
#endif
