/*
 * Copyright (c) 1990, 1991, 1992 Cornell University.  All Rights Reserved.  
 *  
 * Copyright (c) 1991, 1992 Xerox Corporation.  All Rights Reserved.  
 *  
 * Use, reproduction, preparation of derivative works, and distribution
 * of this software is permitted.  Any copy of this software or of any
 * derivative work must include both the above copyright notices of
 * Cornell University and Xerox Corporation and this paragraph.  Any
 * distribution of this software or derivative works must comply with all
 * applicable United States export control laws.  This software is made
 * available AS IS, and XEROX CORPORATION DISCLAIMS ALL WARRANTIES,
 * EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE,
 * AND NOTWITHSTANDING ANY OTHER PROVISION CONTAINED HEREIN, ANY
 * LIABILITY FOR DAMAGES RESULTING FROM THE SOFTWARE OR ITS USE IS
 * EXPRESSLY DISCLAIMED, WHETHER ARISING IN CONTRACT, TORT (INCLUDING
 * NEGLIGENCE) OR STRICT LIABILITY, EVEN IF XEROX CORPORATION IS ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGES.
 */
/*
 * Header file for image maintenance routines
 */

#ifndef IMAGE_H
#define IMAGE_H

/* For FILE */
#include <stdio.h>
#include "adt_strings.h"
#include "misc.h"

#ifdef	IMAGE_CHECKALL
/* Need eassert() and panic() */
#include "panic.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

/* The number of colours per component (or total for gray images) */
#define	COLDEPTH	8
#define COLRNG		(1 << COLDEPTH)

typedef unsigned short ImageType;

/* The image types */
#define	IMAGE_GRAY	((ImageType) (1 <<  0))
#define	IMAGE_FLOAT	((ImageType) (1 <<  1))
#define	IMAGE_RGB	((ImageType) (1 <<  2))
#define	IMAGE_DOUBLE	((ImageType) (1 <<  3))
#define	IMAGE_BINARY	((ImageType) (1 <<  4))
#define	IMAGE_LONG	((ImageType) (1 <<  5))
#define	IMAGE_PTR	((ImageType) (1 <<  6))
#define	IMAGE_SHORT	((ImageType) (1 <<  7))
#define IMAGE_HSV       ((ImageType) (1 <<  8))
#define IMAGE_RGBF      ((ImageType) (1 <<  9))
#define IMAGE_STRUCT    ((ImageType) (1 << 10))

#define IMAGE_ANY       (IMAGE_GRAY   | IMAGE_FLOAT | IMAGE_RGB  | IMAGE_DOUBLE |      \
			 IMAGE_BINARY | IMAGE_LONG  | IMAGE_PTR  | IMAGE_SHORT  |      \
			 IMAGE_HSV    | IMAGE_RGBF  | IMAGE_STRUCT)


struct ImageHeader;

#define DEF_IMG(imgname, elemtype) \
  typedef elemtype * imgname ## PixPtr; \
  typedef struct { \
    elemtype **data; \
    struct ImageHeader *header; \
    struct {void *ptr;} userdata; \
  } * imgname ## Image;

  DEF_IMG(Gray,     uchar);
  DEF_IMG(Float,    float);
  DEF_IMG(RGB,      RGB);
  DEF_IMG(Double,   double);
  DEF_IMG(Binary,   char);
  DEF_IMG(Long,     long);
  DEF_IMG(Ptr,      void *);
  DEF_IMG(Short,    short);
  DEF_IMG(HSV,      HSV);
  DEF_IMG(RGBFloat, RGBFloat);

  DEF_IMG(Any,    void);		     /* generic image */

#undef DEF_IMG

typedef struct ImageHeader {		     /* An invisible image header type */
  int width, height;
  int xbase, ybase;
  ImageType tag;
  short elemsize, headsize;
  AnyImage anyim;
  adtString docstring;
  boolean wrapped;
  void *Ximage;				     /* only x-stuff knows about this */
} ImageHeader;



/* and some manipulation routines */
extern void *imLoad(ImageType types, char *filename);
extern void *imLoadF(ImageType types, FILE *inFile);

extern void im_Free_(void *im, ImageType type);
extern void *im_LoadInto_(void *im, ImageHeader *header, char *filename);
extern void *im_LoadFInto_(void *im, ImageHeader *header, FILE *inFile);
extern int im_Save_(void *im, ImageHeader *header, char *filename);
extern int im_SaveF_(void *im, ImageHeader *header, FILE *outFile);
extern void *im_Dup_(void *im, ImageHeader *header);
extern void im_Init_(void *im, ImageHeader *header, void *val);
extern void im_SetOffset_(void *im, ImageHeader *header, int xbase, int ybase);
extern void im_Apply_(void *f, void *im, ImageHeader *header);
extern void *im_NewOrWrap_(ImageType type, int width, int height, int xbase, int ybase,
			   void *raw, int elemsize, int headsize, void *dupimg);
extern void im_MaxAndMin_(void *im, ImageHeader *header, void *max, void *min);
extern void im_MaxAndMinNoInf_(void *im, ImageHeader *header, void *maxp, void *minp);
extern void *im_Crop_(void *im, ImageHeader *header, int xbase, int ybase, int w, int h);
extern void *im_Crop_Into_(void *im, ImageHeader *header, void *outimage, 
				int in_x, int in_y, int out_x, int out_y,
				int w, int h);

/* and some useful routines -- some are purposely *not* lvalues */
#ifndef	IMAGE_CHECKALL
#define	imGetWidth(i)	({ (i)->header->width; })
#define	imGetHeight(i)	({ (i)->header->height; })
#define	imGetXBase(i)	({ (i)->header->xbase; })
#define	imGetYBase(i)	({ (i)->header->ybase; })
#define imGetType(i)	({ (i)->header->tag; })
#define imDocString(i)  ((i)->header->docstring)
#define imUserData(i)   ((i)->userdata)
#else
#define	imGetWidth(i)	({ assert((i) != NULL); (i)->header->width; })
#define	imGetHeight(i)	({ assert((i) != NULL); (i)->header->height; })
#define	imGetXBase(i)	({ assert((i) != NULL); (i)->header->xbase; })
#define	imGetYBase(i)	({ assert((i) != NULL); (i)->header->ybase; })
#define imGetType(i)	({ assert((i) != NULL); (i)->header->tag; })
#define imDocString(i)  (eassert((i) != NULL), ((i)->header->docstring))
#define imUserData(i)   (eassert((i) != NULL), ((i)->userdata))
#endif
			 
/* Some additions for Greg ... William, I know you'll loathe the first two */
#define imGetStore(i)      ({ &(imRef((i), imGetXBase(i), imGetYBase(i))); })
#define imGetStoreLen(i)   (imGetWidth(i) * imGetHeight(i))
#define imGetXMax(i)       (imGetXBase(i) + imGetWidth(i)  - 1)
#define imGetYMax(i)       (imGetYBase(i) + imGetHeight(i) - 1)
#define imAnyImage(i)      ((i) ? (i)->header->anyim : (AnyImage) NULL)

/* define a new structure image type */
#define imDefStructImage(IMGNAME, ELEMTYPE, USERDATA)                                  \
  typedef struct {                                                                     \
    ELEMTYPE **data;                                                                   \
    ImageHeader *header;                                                               \
    USERDATA userdata;                                                                 \
  } *IMGNAME

typedef struct {void *ptr;} StdImUserData;   /* standard image user data field */

#define imNewStruct(imtype, w, h)     imNewOffsetStruct(imtype, (w), (h), 0, 0)
#define imNewOffsetStruct(imtype, w, h, x, y)                                          \
     ({ imtype __i;                                                                    \
	(imtype) im_NewOrWrap_(IMAGE_STRUCT, (w), (h), (x), (y), NULL,                 \
		      sizeof(typeof(**(__i->data))), sizeof(typeof(*__i)), NULL);})


#define imNULL ((AnyImage) NULL)
		
/*
 * These macros are strange - they attempt to make imFree etc somewhat
 * polymorphic by figuring out, in a context where the type is known (at
 * compile-time), the type of the image and passing that in. If it was
 * not done this way, imFree etc would not be able to figure out what
 * sort of Image they had been passed (since the structures may not match).
 *
 * imFree has imGetType and so implicitly checks for (i) != NULL
 */

#define imFree(i)	   ({ if ((void *) (i) != (void *) NULL) {                     \
				 im_Free_((void *)(i), imGetType((i)));                \
			         (i) = (typeof (i)) NULL;}})

#define imNew(t, w, h)                 imNewOffset((t), (w), (h), 0, 0)
#define imNewOffset(t, w, h, x, y)     im_NewOrWrap_((t), (w), (h), (x), (y),          \
						     NULL, 0, 0, NULL)
#define imWrap(r, t, w, h)             imWrapOffset((r), (t), (w), (h), 0, 0)
#define imWrapOffset(r, t, w, h, x, y) im_NewOrWrap_((t), (w), (h), (x), (y),          \
						     (void *) (r), 0, 0, NULL)

#ifndef	IMAGE_CHECKALL
#define imLoadInto(i, n)   im_LoadInto_((void *)(i), ((i)->header), (n))
#define imLoadFInto(i, f)  im_LoadFInto_((void *)(i), ((i)->header), (f))
#define imSave(i, n)	   im_Save_((void *)(i), ((i)->header), (n))
#define imSaveF(i, f)	   im_SaveF_((void *)(i), ((i)->header), (f))
#define	imDup(i)	   im_Dup_((void *)(i), ((i)->header))
#define imInit(i, val)     ({ typeof (**((i)->data)) __value = (val);                    \
			      im_Init_((void *)(i), ((i)->header), (void *) &__value);})
#define imSetOffset(i, x, y) im_SetOffset_((void *)(i), ((i)->header), (x), (y))
#define imApply(func, i)   ({ void                                                     \
			      (*__func)(typeof (*((i)->data)) v, int x, int y) = func; \
			      im_Apply_((void *) __func, (void *)(i), ((i)->header));})
#define imMaxAndMin(i, max, min)   ({ typeof (**((i)->data)) __max, __min;             \
					im_MaxAndMin_((void *)(i), ((i)->header),      \
						      (void *)&__max, (void *)&__min); \
					(min) = (typeof (min)) __min;                  \
					(max) = (typeof (max)) __max;})
#define imMaxAndMinNoInf(i, max, min)   ({ typeof (**((i)->data)) __max, __min;        \
					im_MaxAndMinNoInf_((void *)(i), ((i)->header), \
						      (void *)&__max, (void *)&__min); \
					(min) = (typeof (min)) __min;                  \
					(max) = (typeof (max)) __max;})
#define imCrop(im, xb, yb, w, h) ((typeof (im)) im_Crop_((void *) (im), ((im)->header), \
							 (xb), (yb), (w), (h)))

#define imCropInto(in, out, in_x, in_y, out_x, out_y, w, h)  \
  ((typeof (in)) im_Crop_Into_((void *) (in), ((in)->header), \
                               (void *) (out), \
			       (in_x),(in_y),(out_x),(out_y),(w),(h))) 

#else
#define imLoadInto(i, n)   (eassert((i) != NULL),                                      \
			    (im_LoadInto_((void *)(i), ((i)->header), (n))))
#define imLoadFInto(i, f)  (eassert((i) != NULL),                                      \
			    (im_LoadFInto_((void *)(i), ((i)->header), (f))))
#define imSave(i, n)	   (eassert((i) != NULL),                                      \
			    (im_Save_((void *)(i), ((i)->header), (n))))
#define imSaveF(i, f)	   (eassert((i) != NULL),                                      \
			    (im_SaveF_((void *)(i), ((i)->header), (f))))
#define	imDup(i)	(eassert((i) != NULL), (im_Dup_((void *)(i), ((i)->header))))
#define imInit(i, val)  (eassert((i) != NULL),                                         \
			 ({ typeof (**((i)->data)) __value = (val);                    \
			    im_Init_((void *)(i), ((i)->header), (void *) &__value);}))
#define imSetOffset(i, x, y) (eassert((i) != NULL),                                    \
			    (im_SetOffset_((void *)(i), ((i)->header), (x), (y))))
#define imApply(func, i)   (eassert((i) != NULL), ({ void                              \
			      (*__func)(typeof (*((i)->data)) v, int x, int y) = func; \
			      im_Apply_((void *) __func, (void *)(i), ((i)->header));}))
#define imMaxAndMin(i, max, min)   (eassert((i) != NULL),                              \
				    ({ typeof (**((i)->data)) __max, __min;            \
					 im_MaxAndMin_((void *)(i), ((i)->header),     \
						       (void *)&__max, (void *)&__min); \
					 (min) = (typeof (min)) __min;                 \
					 (max) = (typeof (max)) __max;}))
#define imMaxAndMinNoInf(i, max, min)   (eassert((i) != NULL),                         \
				    ({ typeof (**((i)->data)) __max, __min;            \
					 im_MaxAndMinNoInf_((void *)(i), ((i)->header), \
						       (void *)&__max, (void *)&__min); \
					 (min) = (typeof (min)) __min;                 \
					 (max) = (typeof (max)) __max;}))
#define imCrop(im, xb, yb, w, h) ((typeof (im)) (eassert((im) != NULL),                \
						 im_Crop_((void *) (im), ((im)->header),\
							  (xb), (yb), (w), (h))))

/* XXX imCropInto doesn't check anything right now... mdw */
#define imCropInto(in, out, in_x, in_y, out_x, out_y, w, h)  \
  ((typeof (in)) im_Crop_Into_((void *) (in), ((in)->header), \
                               (AnyImage *)(out), \
			       (in_x),(in_y),(out_x),(out_y),(w),(h)))

#endif

#define  imBoundsCheck(i, x, y)    (imBoundsCheckX((i), (x))  &&                   \
                                    imBoundsCheckY((i), (y)))
#define  imBoundsCheckX(i, x)      (imBoundsCheckXMax((i), (x)) &&                 \
                                    imBoundsCheckXMin((i), (x)))
#define  imBoundsCheckY(i, y)      (imBoundsCheckYMax((i), (y)) &&                 \
                                    imBoundsCheckYMin((i), (y)))
#define  imBoundsCheckXMax(i, x)   ((x) <= imGetXMax((i)))
#define  imBoundsCheckXMin(i, x)   ((x) >= imGetXBase((i)))
#define  imBoundsCheckYMax(i, y)   ((y) <= imGetYMax((i)))
#define  imBoundsCheckYMin(i, y)   ((y) >= imGetYBase((i)))

/* and an access routine - gives an *lvalue* */
#ifndef	IMAGE_CHECKALL
#define	imRef(i, x, y)	(((i)->data)[(y)][(x)])
#else
static char __imPanicStr[100];
#define	imRef(i, x, y)	(eassert(i != NULL),                                       \
			 ((((x) < (i)->header->xbase) ||                           \
			   ((y) < (i)->header->ybase) ||                           \
			   ((x) >= (i)->header->xbase + (i)->header->width) ||     \
			   ((y) >= (i)->header->ybase + (i)->header->height))      \
			  ? ({sprintf(__imPanicStr,                                \
				      "image bounds check fail: "                  \
				      "%d <= %d < %d, %d <= %d < %d",              \
				      (i)->header->xbase,                          \
				      (x),                                         \
				      (i)->header->xbase + (i)->header->width,     \
				      (i)->header->ybase,                          \
				      (y),                                         \
				      (i)->header->ybase + (i)->header->height);   \
			       panic(__imPanicStr);}), ((i)->data[0][0])           \
			  : (((i)->data)[(y)][(x)])))
#endif





/* Here come macros for increased efficiency access */

#ifndef IMAGE_CHECKALL

#define imGetPixPtr(i, x, y) (&(imRef((i), (x), (y))))
#define imPtrRef(i, pp) (*(pp))
#define imPtrDup(i, pp) (pp)
#define imPtrUp(i, pp) ((pp) -= (i)->header->width)
#define imPtrDown(i, pp) ((pp) += (i)->header->width)
#define imPtrLeft(i, pp) ((pp)--)
#define imPtrRight(i, pp) ((pp)++)
#define imPtrEq(i, pp1, pp2) ((pp1) == (pp2))

#else

/* Check a pointer against bounds; eval expr if successful */
#define __IMPCHECK(i, pp, expr) (eassert((i) != NULL), ((((pp) < (&((i)->data[(i)->header->ybase][(i)->header->xbase]))) || ((pp) > &((i)->data[(i)->header->yba se+(int)(i)->header->height-1][(i)->header->xbase+(int)(i)->header->width-1]))) ? (({ panic("PixPtr bounds check fail"); }), (expr)) : (expr)))

/* imRef is error-checked already */
#define imGetPixPtr(i, x, y) (&(imRef((i), (x), (y))))
#define imPtrRef(i, pp) __IMPCHECK((i), (pp), (*(pp)))
#define imPtrDup(i, pp) __IMPCHECK((i), (pp), (pp))
#define imPtrUp(i, pp) __IMPCHECK((i), (pp), ((pp) -= (i)->header->width))
#define imPtrDown(i, pp) __IMPCHECK((i), (pp), ((pp) += (i)->header->width))
#define imPtrLeft(i, pp) __IMPCHECK((i), (pp), ((pp)--))
#define imPtrRight(i, pp) __IMPCHECK((i), (pp), ((pp)++))
#define imPtrEq(i, pp1, pp2) __IMPCHECK((i), (pp1), __IMPCHECK((i), (pp2), ((pp1) == (pp2))))

#endif

#ifdef __cplusplus
}
#endif


#endif
